#include "FdIO.h"
#include <sys/stat.h>
#include <errno.h>
#include "FdIDir.h"

/*---------------------------------------------------------------------------*/
/* The logic of this code works with integer frame duration and frame
   starts aligned on integer GPS time;
   The frames are found using the file name information  */ 
/*-----------------------------------------------declare private functions---*/
 
void FdIDirProcess(FdIDir* list, FrameH* frame);
 
/*----------------------------------------------------------------------------*/
void FdIDirFree(FdIDir* iDir)
/*----------------------------------------------------------------------------*/
{
  free(iDir->dirName);
  free(iDir->nextFile);
  if(iDir->frameInfo != NULL) free(iDir->frameInfo);
  if(iDir->prefix    != NULL) free(iDir->prefix);
  if(iDir->iFile     != NULL) FrFileIEnd(iDir->iFile);

  free(iDir);

  return;
}
/*----------------------------------------------------------------------------*/
FdIDir* FdIDirNew(char* dirName, int tStart)
/*----------------------------------------------------------------------------*/
{
  FdIDir* iDir;
  FILE *fp;
  int irc;

  /*------------------------------------------------------create the object ---*/
  iDir = (FdIDir*) calloc(1,sizeof(FdIDir));
  if(iDir == NULL) CfgMsgAddFatal("malloc FdIDir failed");

  /*------------------------------------------------------------- fill info---*/
  FrStrCpy(&(iDir->dirName), dirName);
  iDir->tStart = 1000*tStart;
  iDir->iFile  = NULL;     
  iDir->nPrint = 0;
  iDir->lastRead = FR_YES;
  iDir->openError = FR_NO;
  iDir->noFileError = FR_NO;
  iDir->prefix = NULL;
  iDir->frameInfo = NULL;
  iDir->nextFile  = calloc(1024, sizeof(char));
  if(iDir->nextFile == NULL) CfgMsgAddFatal("malloc FdIDir->nextFile failed");

  /*--------------------------------------------------------------------------*/
  CfgMsgAddInfo("We will get frame from directory:%s", iDir->dirName);

  /*--------------------------------------------------------scanning mode ---*/
  if(iDir->tStart == -2000) iDir->scanMode = -2;
  else                      iDir->scanMode = 0;

  /*-------------------------------check if we recover the last frame read---*/
  if(iDir->tStart == -1000) {
    iDir->frameInfo = malloc(1024);
    sprintf(iDir->frameInfo,"%s/LastFrameInfo-%s.info", 
            iDir->dirName, CfgGetCmName());

    CfgMsgAddInfo("Recovering the last frame read from:%s", iDir->frameInfo);
    fp = fopen(iDir->frameInfo,"r");
    if(fp == NULL) {
      CfgMsgAddWarning("Could not open last frame info");
      iDir->tStart = 0;}
    else {
      iDir->prefix = malloc(256);
      irc = fscanf(fp,"%s\n%s\n%ld\n%ld", 
                   iDir->nextFile, iDir->prefix, &(iDir->tStart),&(iDir->dtFile));
      if(irc != 4) {
        CfgMsgAddWarning("Could not be recover last frame info from %s",
                        iDir->frameInfo);
        iDir->tStart = 0;
        free(iDir->prefix);
        iDir->prefix = NULL;}
      else {
        CfgMsgAddInfo("Next file: %s, prefix:%s",iDir->nextFile,iDir->prefix);}
      fclose(fp);}}

  /*-------------------------------------------------------------------------*/
  if(iDir->tStart >  0)    CfgMsgAddInfo(" Starting at or after %ld",iDir->tStart/1000);
  if(iDir->tStart == 0)    CfgMsgAddInfo(" Starting from the latest file");
  if(iDir->tStart ==-2000) CfgMsgAddInfo(" Starting from the oldest file");

  return(iDir);
}
/*----------------------------------------------------------------------------*/
int FdIDirNewCfg(void *arg, CfgDataType_t *data)
/*----------------------------------------------------------------------------*/
{
  FdIDir* iDir;
  char* name;
  int tStart;

  /*-------------create the object and add it at the end of the linked list---*/
  name   = CfgParseGetNextString(data);
  tStart = CfgParseGetNextDec(data);
  iDir = FdIDirNew(name, tStart);

  iDir->dtMargin = CfgParseGetNextDec(data);
  if(iDir->dtMargin <= 0) iDir->dtMargin = 10; // default value

  iDir->action = FdActionNew((FdAction**) arg, (void*) iDir, 
			     FdIDirProcess, "FDIN_DIRECTORY");

  if(iDir->action != *((FdAction**) arg)) 
    CfgMsgAddFatal("FDIN_DIRECTORY  could not be use with the %s key",
		   (*((FdAction**) arg))->type);

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
void FdIDirBuildFileName(FdIDir* iDir)
/*----------------------------------------------------------------------------*/
{
  int  index;
  
  if(iDir->dtFile < 1000) {
    index = ((iDir->tStart%1000)*iDir->nFramesPerSec)/1000;
    sprintf(iDir->nextFile, "%s/%s%ld%c-%c.gwf", iDir->dirName,
            iDir->prefix, iDir->tStart/1000, index+97, iDir->nFramesPerSec+96);}
  else {
    sprintf(iDir->nextFile, "%s/%s%ld-%ld.gwf", iDir->dirName, 
	    iDir->prefix, iDir->tStart/1000, iDir->dtFile/1000);}

  return;
}
/*----------------------------------------------------------------------------*/
void FdIDirNextFile(FdIDir* iDir, int debugLvl)
/*----------------------------------------------------------------------------*/
{
  FILE *fp;
  
  iDir->tStart += iDir->dtFile;

  /*-------------------------------------------------prepare next file name---*/
  FdIDirBuildFileName(iDir);

  if(debugLvl > 0) 
    CfgMsgAddInfo("InputDir: next file: %s", iDir->nextFile);

  /*--------------------------------write frame file info only if requested---*/
  if(iDir->frameInfo == NULL) return;

  /*--------------record time of next frame to be able to restart if needed---*/
  fp = fopen(iDir->frameInfo, "w");
  if(fp == NULL) {
    CfgMsgAddError("Could not open file info:%s no more attempt",iDir->frameInfo);
    free(iDir->frameInfo);
    iDir->frameInfo = NULL;
    return;}

  fprintf(fp,"%s\n%s\n", iDir->nextFile, iDir->prefix);
  fprintf(fp,"%ld\n%ld", iDir->tStart,   iDir->dtFile);

  fclose(fp);

  return;
}
/*----------------------------------------------------------------------------*/
int FdIDirObsolete(void *arg,   CfgDataType_t *data)
/*----------------------------------------------------------------------------*/
{                   
 
  CfgMsgAddFatal("FDIN_DIRECTORY has been replaced by FDIN_DIR"
                 " with new parameters");

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
FrameH *FdIDirGetNextFrame(FdIDir* iDir, int dbgLvl)
/*----------------------------------------------------------------------------*/
/* this function returns the next frame in the open file or NULL if there are
   no more frame in this file */
{
  FrameH *frame;
  char *tag;

  if(iDir->iFile == NULL) return(NULL);

  /*------------------------------------------------------read full frames---*/
  tag = NULL;
  if(iDir->action != NULL) tag = iDir->action->tag;
  if(tag == NULL) {
    frame = FrameRead(iDir->iFile);}

  /*------------------------------------------------------read with a tag---*/
  else {
    frame = FrameReadTChnl(iDir->iFile, iDir->tNextFrame*0.001+.001, tag);
    if(frame != NULL) {
      frame->event = FrEventReadTF(iDir->iFile, tag, 
                                 frame->GTimeS + 1.e-9*frame->GTimeN, 
                                 frame->dt, 1, 0);}

    iDir->tNextFrame = 1000*FrFileITNextFrame(iDir->iFile, 
					 iDir->tNextFrame*0.001+0.001);}

  /*---------------------------------------------checks and error recovery---*/
  if(frame == NULL || iDir->tNextFrame < 0)  {
    FrFileIEnd(iDir->iFile);
    iDir->iFile = NULL;}

  if(frame != NULL) {
    iDir->lastRead = FR_YES;
    if(dbgLvl > 0) CfgMsgAddInfo(" return frame %d.%.1d from %s", 
	frame->GTimeS, frame->GTimeN/100000000,  iDir->nextFile);
    iDir->timeLastFrame = time(NULL);}

  return(frame);
}
/*----------------------------------------------------------------------------*/
FrameH *FdIDirProcessOne(FdIDir* iDir, int dbgLvl)
/*----------------------------------------------------------------------------*/
{
  FrameH *frame;
  DIR  *dirp;
  struct dirent *entry;
  struct stat sb;
  FRLONG i, gps, len, bestStart, bestDtFile, nFiles, offset, dtFile;
  int  index, timeNow, nextScan, dtNextScan;
  FILE *fp;
  char buf[256];

  /*----------------------- First we try to read from the already open file---*/
  if(iDir->iFile != NULL) {
    frame = FdIDirGetNextFrame(iDir, dbgLvl);
    if(frame != NULL) return(frame);

    if(dbgLvl > 0) CfgMsgAddInfo("InputDir: could not read next frame in file; "
         "try to open the following expected file: %s",  iDir->nextFile);
    FrFileIEnd(iDir->iFile);
    iDir->iFile = NULL;}

  /*-----------------------------check that the directory has been updated----*/
  stat(iDir->dirName, &sb);
  if(iDir->lastMtimeS == sb.st_mtime &&
     iDir->lastMtimeN == sb.st_mtimensec &&
     iDir->lastRead == FR_NO) return(NULL); /*------nothing new---*/

  iDir->lastMtimeS = sb.st_mtime;
  iDir->lastMtimeN = sb.st_mtimensec;
  iDir->lastRead = FR_NO;

  /*-------------------------------------for debuging, list available files---*/
  if(dbgLvl > 2) {
    sprintf(buf,"ls %s", iDir->dirName);
    CfgMsgAddInfo(buf);
    fp = popen(buf,"r");
    if(fp != NULL) {
      while(fgets(buf,256, fp) > 0) {
        CfgMsgAddInfo(buf);}
      pclose(fp);}}

  /*-------------------------------------try to open the expected next file---*/
  iDir->tNextFrame = iDir->tStart;
  iDir->iFile = FrFileINew(iDir->nextFile);
  if(iDir->iFile == NULL) {
    if(dbgLvl > 1) CfgMsgAddInfo("InputDir: could not open the next "
	"expected file: %s", iDir->nextFile);}
  else {
    if(dbgLvl > 1) CfgMsgAddInfo("InputDir: open %s ", iDir->nextFile);
    iDir->iFile->compress = FdIOGetCompressIn();
    iDir->iFile->chkSumFrFlag = FR_NO; // files are usually from shm
    frame = FdIDirGetNextFrame(iDir, dbgLvl);
    if(frame != NULL) {
      FdIDirNextFile(iDir, dbgLvl);
      return(frame);}
    else {
      if(dbgLvl > 0) CfgMsgAddInfo("InputDir: could not read first frame in %s",
			iDir->nextFile);}}

  /*------make sure that we wait at least 2 file periods before a full scan---*/
  /*         remember: dtFile is in ms; while dtNextScan is in second         */
  timeNow = time(NULL);
  dtNextScan = 0.002*iDir->dtFile + iDir->dtMargin; 
  nextScan = iDir->timeLastFrame + dtNextScan;
  if(timeNow <= nextScan)  return(NULL);

  if(dbgLvl > 0)  CfgMsgAddInfo(
	"Full scan for %s timeNow:%d last:%d dtFile:%ld ms nextScan:%d", 
         iDir->dirName, timeNow, iDir->timeLastFrame, iDir->dtFile, nextScan);

  int waitingTime = timeNow - iDir->timeLastFrame;
  iDir->timeLastFrame = timeNow;

  /*-----Find the next file by making a full scan of the directory;
    -----This works also for the first read ----------------------------------*/
  nFiles = 0;
  if(iDir->scanMode > -2) bestStart = iDir->tStart;
  else                    bestStart = 1.e14; 
  bestDtFile = 0;
 
  dirp  = opendir(iDir->dirName);
  if(dirp == NULL) {
    if(iDir->openError == FR_NO)
      CfgMsgAddError("Could not open dir. %s (%s); keep scanning",
                                  iDir->dirName, strerror(errno));
    iDir->openError = FR_YES;
    return(NULL);}
  else if(iDir->openError == FR_YES) {
    iDir->openError = FR_NO;
    CfgMsgAddInfo("%s has been successfully opened", iDir->dirName);}

  /*------------------------------------------- find the most recent file ----*/
  while((entry = readdir(dirp)) != NULL) {

    if(dbgLvl > 0) CfgMsgAddInfo("check file %s bestStart=%ld scanMode=%d "
      "tStart=%ld", entry->d_name, bestStart, iDir->scanMode, iDir->tStart);

    /*----------------------------------first check that it is a frame file---*/
    len = strlen(entry->d_name);
    if(strcmp(entry->d_name + len - 4, ".gwf") != 0) continue;

    /*--------------------get file duration and index for sub second frames---*/
    if(isdigit(entry->d_name[len - 5]) == 0) {
      if(entry->d_name[len - 6] != '-') { 
	if(iDir->nWarningsPrfx++ > 5) 
	  CfgMsgAddWarning("InputDir: file with unexpected format found:%s",
			   entry->d_name);
        continue;}
      iDir->nFramesPerSec = entry->d_name[len - 5] - 96;
      index = entry->d_name[len - 7] - 97;
      if     (iDir->nFramesPerSec ==  2) dtFile = 500;
      else if(iDir->nFramesPerSec ==  4) dtFile = 250;
      else if(iDir->nFramesPerSec ==  5) dtFile = 200;
      else if(iDir->nFramesPerSec == 10) dtFile = 100;
      else if(iDir->nFramesPerSec == 20) dtFile =  50;
      else {
	CfgMsgAddWarning("InputDir: file %s with unexpected index found:%d",
			   entry->d_name, index);
        continue;}
      offset = index*dtFile;
      i = len-7;}

    else { /*-------this is the usual case of frames that are least 1 sec. long--*/
      for(i=len-4; i>0; i--) {if(entry->d_name[i] == '-') break;}
      if(i == 0) continue;
      if(sscanf(entry->d_name + i + 1, "%ld", &dtFile) != 1) continue;
      dtFile *= 1000;
      offset = 0;
      iDir->nFramesPerSec = 0;}

    /*-----------------------------------------get file start from its name---*/
    for(i--; i>0; i--) {if(entry->d_name[i] == '-') break;}
    if(i == 0) continue; 
    sscanf(entry->d_name + i + 1, "%ld", &gps);
    gps = 1000*gps + offset; 
    /*--------------------------------------get prefix if not yet available---*/
    if(iDir->prefix == NULL) {
      iDir->prefix   = malloc(256);
      memcpy(iDir->prefix, entry->d_name, i+1);
      iDir->prefix[i+1] = '\0';
      CfgMsgAddInfo("InputDir: file prefix is \"%s\"",iDir->prefix);}

    /*--------------------------or check that the file has the rigth prefix---*/
    else {
      if(strncmp(iDir->prefix, entry->d_name, i+1) != 0) {
        if(iDir->nWarningsPrfx++ < 5)
          CfgMsgAddWarning("InputDir: file: %s with unexpected prefix: %s."
                " Remove the *.info file(s) and restart it",
                           entry->d_name, iDir->prefix);
        continue;}}

    nFiles++;
    if(dbgLvl > 0) CfgMsgAddInfo(" gps:%ld offset:%ld iDir->tStart:%ld",
		gps, offset, iDir->tStart);

    /*------------------------------ if it is the request file, stop scanning---*/
    if(gps == iDir->tStart) {
       bestStart  = gps;
       bestDtFile = dtFile;
       if(dbgLvl > 0) CfgMsgAddInfo("InputDir:  Requested file found");
       break;}

    /*---searching for the oldest file; only the first time if this was asked---*/
    else if(gps < bestStart && gps >= iDir->tStart && iDir->scanMode == -2) {
      bestStart  = gps;
      bestDtFile = dtFile;}

    /*--is this file the next one to read? Always search for most recent file---*/
    else if(gps > bestStart && iDir->scanMode != -2) {
      bestStart  = gps;
      bestDtFile = dtFile;}
  } /*---------------------------- end of the loop which scan the directory---*/

  closedir(dirp);

  if(dbgLvl > 0) 
    CfgMsgAddInfo("InputDir: %ld gwf files found; next: GPS=%ld ms dtFile:%ld", 
		nFiles, bestStart, bestDtFile);

  if(iDir->prefix == NULL) {
    if(iDir->noFileError == FR_NO) {
      CfgMsgAddError("No .gwf files found in %s; keep scanning", iDir->dirName);
      iDir->noFileError = FR_YES;}
    return(NULL);}
  else if(iDir->noFileError == FR_YES) {
    CfgMsgAddInfo("Files found in %s",iDir->dirName);
    iDir->noFileError = FR_NO;}

  if(bestDtFile == 0) return(NULL);

   /*---------------------a file is found, open it and read the first frame---*/
  iDir->tStart = bestStart;
  iDir->dtFile = bestDtFile;

  FdIDirBuildFileName(iDir);

  if(waitingTime > 1.7) waitingTime = 0; // this is for the first scan
  CfgMsgAddInfo("InputDir: first file to open after a scan:%s (wait: %d s)",
                 iDir->nextFile, waitingTime);

  /*----------------------------open the next file and read the first frame---*/
  iDir->iFile = FrFileINew(iDir->nextFile);
  if(iDir->iFile == NULL) {
    CfgMsgAddError("InputDir: open file error for %s",iDir->nextFile);
    return(NULL);}
  iDir->iFile->compress = FdIOGetCompressIn();
  iDir->iFile->chkSumFrFlag = FR_NO; // files are usually from shm

  iDir->tNextFrame = bestStart;

  frame = FdIDirGetNextFrame(iDir, dbgLvl);

  /*---------If the frame cannot be read; the file may already be removed @---*/
  /*                                      ->close it and move to next file---*/
  if(frame == NULL) {
    CfgMsgAddError("InputDir: could not read first frame in %s",iDir->nextFile);
    FrFileIEnd(iDir->iFile);
    iDir->iFile = NULL;}
 
  FdIDirNextFile(iDir, dbgLvl);

  return(frame);
}
/*----------------------------------------------------------------------------*/
void FdIDirProcess(FdIDir *iDir,
		   FrameH* frame)
/*----------------------------------------------------------------------------*/
{
  FdAction *next;

  frame = FdIDirProcessOne(iDir, iDir->action->debugLvl);
  if(frame == NULL) return;

  if(iDir->action->debugLvl > 0) 
    CfgMsgAddInfo("FdIDirProcess returns frame %d.%.3d", 
	frame->GTimeS, frame->GTimeN/1000000);

  next = iDir->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*----------------------------------------------------------------------------*/
void FdIDirParserAdd(FdIO* fdIO)
/*----------------------------------------------------------------------------*/
{
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_DIR",
			 FdIDirNewCfg, (void *) &(fdIO->actionsIn), 3,
			 CfgString, CfgDec, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_DIRECTORY",
			 FdIDirObsolete, NULL, 1,  CfgString);

  return;
}
