/************************************************************************/
/*  FbtServer.c                                                         */  
/*  B. Mours, D.Verkindt (LAPP Annecy)                                  */
/************************************************************************/
#include "FdIO.h"
#include "Frv.h"

/*-----------------------------------------------------private definitions---*/
typedef struct FdFbt FdFbt;

struct FdFbt               /* Trend frame builder parameters          */
{
  char   *tag;             /* tag to compute remaining trend data     */
  int     frameL;          /* trend frame length (in seconds)         */
  int     nBits;           /* number of bits used to store results    */
  FrFile *oFile;           /* output trend file pointer               */
  int     nChannels;       /* number of channel in the trend frame    */
  int     nAdc;            /* number of ADC channels                  */
  int     adcSize;         /* size of the input ADC's                 */
  double  procTime;        /* processing time                         */
  FrameH *trend;           /* current trend frame                     */
  FrCList *listTrend;      /* fast access list for the trend channels */
  FdAction*  action;      /* action object which hold this object */
};

typedef struct FdTOT FdTOT;

struct FdTOT{
  int        period;      /* new period */
  FdAction*  action;      /* action object which hold this recast object */
};  

void FdFbtProcess(FdFbt *fbt, FrameH *frame);
int FdTOTNew(void *arg, CfgDataType_t *data);

/*---------------------------------------------------------------------------*/
void  FbtFillAdc(FdFbt *fbt,
		 char  *name,
		 char  *word1,
		 double value,
		 int    index,
		 int    nData,
		 double samplingRate,
		 double slope,
		 double bias,
		 char  *units,
		 int   nBits)
/*---------------------------------------------------------------------------*/
/* This function fill (and create it if needed) an trend adc channel         */
/*---------------------------------------------------------------------------*/
{
  FrameH *trend;
  FrAdcData *adc;
  FrVect  *isThere;
  int  j, k;
  char channelName[512];
 
  trend = fbt->trend;

  /*------------------------ search the adc channel in the trend frame---*/
  if(word1 != NULL) sprintf(channelName, "%s_%s",name, word1);
  else              sprintf(channelName, "%s",   name);
  adc = (FrAdcData *) FrCListFind(fbt->listTrend,channelName);

  /*-------------- the channel is not in the trend frame. Create it ---*/
  if (adc == NULL) {

     adc = FrAdcDataNewF(trend, channelName, NULL, 0, 0, nBits, 
			bias, slope, units, samplingRate, nData);
    if (adc == NULL) {
      CfgMsgAddFatal("FbtBuildTrendFrame: malloc adc failed. "
		     "channel=%s  nData=%d  nBits=%d",
		     channelName, nData, nBits);}
    for(j=0; j<adc->data->nData; j++) {adc->data->dataI[j] = 0;}

    /*----------------------------------------add auxiliary vector ---*/
    isThere = FrVectNew1D("Available_data",  FR_VECT_C,  nData, 
			  adc->data->dx[0], adc->data->unitX[0], 
			  "1 if data is available");
    if(isThere == NULL) {
      CfgMsgAddFatal("FbtBuildTrendFrame: malloc failed"
		     " channel=%s  nData=%d", channelName, nData);}
    for(k=0; k<isThere->nData; k++) {isThere->data[k] = 0.;}
    adc->data->next = isThere;}

  /*-----------------------------adc channel is there, fill data ---*/
  if (index >= adc->data->nData) {
    CfgMsgAddError("FbtBuildTrendFrame. Vector overflow at %d index=%d "
		   "samplingRate=%g  channel=%s serData:%s",
		   trend->GTimeS, index, samplingRate, channelName,
		   name);
    CfgMsgAddError("  nData=%"FRLLD" nbits=%d",adc->data->nData, nBits);
    index =  adc->data->nData - 1;}

  if(nBits > 0)         adc->data->dataUI[index] = value;
  else if(nBits == -64) adc->data->dataD [index] = value;
  else                  adc->data->dataF [index] = value;
  adc->data->next->data[index] ++;

  return;
} 
/*------------------------------------------------------ FbtBuildTrendData --*/
void FbtBuildTrendData (FdFbt *fbt, FrameH *frame)
/*---------------------------------------------------------------------------*/
/* This function build and add trend data for each taged ADC channel.        */
/*---------------------------------------------------------------------------*/
{
  FrAdcData *adc;
  FrVect *data;
  FrProcData *proc;
  double min, max, mean, rms, samplingRate, trendRate, m, slope;
  int index, nData, i;
  unsigned int or, and;

  index = (frame->GTimeS % fbt->frameL)/frame->dt;
  nData =  fbt->frameL/frame->dt;
  trendRate = 1./frame->dt;

  fbt->nChannels = 0;

  /*---------------------------------- build the trend data for the adc's---*/
  adc = NULL;
  if (frame->rawData != NULL) adc = frame->rawData->firstAdc;

  for (; adc != NULL; adc = adc->next) {
    data = adc->data;
    if(data->nDim != 1) continue;
    if(FrVectIsValid(data) != 0) continue;
    if(data->next != NULL) continue;
    fbt->nChannels++;

    if(data->nData == 1) {
      samplingRate = 1.;
      if(data->dx[0] > 1) samplingRate = 1./data->dx[0];
      FbtFillAdc(fbt,adc->name,NULL, FrVectGetValueI(data,0),
                 index*frame->dt*samplingRate, fbt->frameL*samplingRate, 
                 samplingRate, adc->slope,
                 adc->bias, adc->units, fbt->nBits);
      continue;}

    /*-------------------------------------------------------encoded bits---*/
    if(data->type == FR_VECT_4U) {
      if(data->compress != 0) FrVectExpand(data);
      or  = data->dataUI[0];
      and = data->dataUI[0];
      for(i = 1; i<data->nData; i++) {
        or  = or  | data->dataUI[i];
        and = and & data->dataUI[i];}

      FbtFillAdc(fbt, adc->name,"or", or,   index, nData, trendRate,
 	       1., 0., adc->units, 32);
      FbtFillAdc(fbt, adc->name,"and", and, index, nData, trendRate,
	       1., 0., adc->units, 32);
      continue;}

    /*-----------------------------------faster channel with regular value---*/
    FrVectMinMax(data, &min, &max);
    FrvRms(data, &mean, &rms);
    if(adc->slope < 0) {
      slope = -adc->slope;
      m = max;
      max = -min; 
      min = -m;
      mean = -mean;}
    else {    
      slope = adc->slope;}
  
    FbtFillAdc(fbt, adc->name,"min", min,   index, nData, trendRate,
	       slope, adc->bias, adc->units, fbt->nBits);
    FbtFillAdc(fbt, adc->name,"mean", mean, index, nData, trendRate,
	       slope, adc->bias, adc->units, fbt->nBits);
    FbtFillAdc(fbt, adc->name,"max", max,   index, nData, trendRate,
	       slope, adc->bias, adc->units, fbt->nBits);
    FbtFillAdc(fbt, adc->name,"rms", rms,   index, nData, trendRate,
	       slope, 0., adc->units, fbt->nBits);}

  /*-------------------------------------- build trend data for FrProcData---*/
  for (proc = frame->procData; proc != NULL; proc = proc->next) {
    if(proc->type != 1) continue;
    data = proc->data;
    if(data == NULL) continue;
    if(data->nDim != 1) continue;
    if(FrVectIsValid(data) != 0) continue;
    if(data->next != NULL) continue;
    fbt->nChannels++;

    if(data->nData == 1) {/*--- one 1Hz signal: copy only the value--*/
      samplingRate = 1.;
      if(data->dx[0] > 1) samplingRate = 1./data->dx[0];
      FbtFillAdc(fbt, proc->name,NULL, FrVectGetValueI(data,0), 
		 index*frame->dt*samplingRate, fbt->frameL*samplingRate, 
                 samplingRate, 1., 0., data->unitY, fbt->nBits);
      continue;}

    /*-------------------------------------------------------encoded bits---*/
    if(data->type == FR_VECT_4U) {
      if(data->compress != 0) FrVectExpand(data);
      or  = data->dataUI[0];
      and = data->dataUI[0];
      for(i = 1; i<data->nData; i++) {
        or  = or  | data->dataUI[i];
        and = and & data->dataUI[i];}

      FbtFillAdc(fbt, proc->name,"or", or,   index, nData, trendRate,
 	       1., 0., data->unitY, 32);
      FbtFillAdc(fbt, proc->name,"and", and, index, nData, trendRate,
	       1., 0., data->unitY, 32);
      continue;}

    /*-----------------------------------faster channel with regular value---*/
    FrVectMinMax(data, &min, &max);
    FrvRms(data, &mean, &rms);

    FbtFillAdc(fbt, proc->name, "min",  min, index, nData, trendRate,
	       1., 0., data->unitY, fbt->nBits);
    FbtFillAdc(fbt, proc->name,"mean", mean, index, nData, trendRate,
	       1., 0., data->unitY, fbt->nBits);
    FbtFillAdc(fbt, proc->name, "max",  max, index, nData, trendRate,
	       1., 0., data->unitY, fbt->nBits);
    FbtFillAdc(fbt,  proc->name, "rms", rms, index, nData, trendRate,
	       1., 0., data->unitY, fbt->nBits);}

  return;
}
/*---------------------------------------------------------------------------*/
void  FbtBuildTrendFrame(FdFbt *fbt, FrameH *frame)
/*---------------------------------------------------------------------------*/
/* This function convert all FrSerData to a trend frame.                     */
/*---------------------------------------------------------------------------*/
{
  FrDetector *det, **lastDet;
  FrSerData *sms;
  int  nData, i, index, period, nRead;
  unsigned int valueI;
  char *data, word1[256], word2[256], word3[256];
  char channelName[512], units[256];
  float bias, slope;
  double value;
 
  if (frame == NULL) return;

  index = frame->GTimeS % fbt->frameL; /*--compute index in the trend frame--*/

  /*---------------------------------- create a new trend frame if needed ---*/
  if (fbt->trend == NULL)  {
    fbt->trend = FrameHNew("Trend_frame");
    if (fbt->trend == NULL) CfgMsgAddFatal("could not create trend frame");
    fbt->trend->run       = 0;
    fbt->trend->frame     = frame->GTimeS / fbt->frameL;
    fbt->trend->GTimeS    = frame->GTimeS - index;
    fbt->trend->GTimeN    = 0;
    fbt->trend->ULeapS    = frame->ULeapS;
    fbt->trend->dt        = fbt->frameL;

    lastDet = &(fbt->trend->detectProc);
    for(det = frame->detectProc; det != NULL; det = det->next) {
      *lastDet = FrDetectorNew(det->name);
      lastDet = &((*lastDet)->next);
      fbt->trend->detectProc->prefix[0]    = det->prefix[0];
      fbt->trend->detectProc->prefix[1]    = det->prefix[1];
      fbt->trend->detectProc->longitude    = det->longitude;
      fbt->trend->detectProc->latitude     = det->latitude;
      fbt->trend->detectProc->elevation    = det->elevation;
      fbt->trend->detectProc->armXazimuth  = det->armXazimuth;
      fbt->trend->detectProc->armYazimuth  = det->armYazimuth;
      fbt->trend->detectProc->armXaltitude = det->armXaltitude;
      fbt->trend->detectProc->armYaltitude = det->armYaltitude;
      fbt->trend->detectProc->armXmidpoint = det->armXmidpoint;
      fbt->trend->detectProc->armYmidpoint = det->armYmidpoint;
      fbt->trend->detectProc->localTime    = det->localTime;}
  }

  /*---------- build a fast access list for the channel in the trend frame---*/
  fbt->listTrend = FrCListBldAdc(fbt->trend);

  /*------------------------convert sms data to adc vectors ---*/
  if (frame->rawData == NULL) return;
  for(sms = frame->rawData->firstSer; sms != NULL; sms = sms->next) {
  
    if (sms->sampleRate <= 0) sms->sampleRate = 1.;
    period = (0.5 + 1. / sms->sampleRate);

    if(fbt->frameL % period != 0) {
      CfgMsgAddError("At %d: the trend frame length (%d) is not "
                     "a multiple of the period (%d) of %s",
                     frame->GTimeS,fbt->frameL, period, sms->name);
      continue;}

    i      = index / period;
    nData  = fbt->frameL / period;
    if(nData < 1) nData = 1;

    /*------------------------------------- decode each individual blocks ---*/
    strcpy(units,"Counts");
    bias  = 0.;
    slope = 1.;
    data = sms->data;
    while ((nRead = sscanf(data,"%255s %255s %255s",word1,word2,word3)) >= 2) {
  
      if(strlen(word1) > 200) {
	CfgMsgAddError("FbtBuildTrendFrame: %d too long word for %s: %s",
		       frame->GTimeS, sms->name, word1);
	break;}

      /*------------------------------------ save calibration info if any ---*/
      if     (strcmp(word1,"bias")  == 0)  {sscanf (word2,  "%e", &bias);} 
      else if(strcmp(word1,"slope") == 0)  {sscanf (word2,  "%e", &slope);} 
      else if(strcmp(word1,"units") == 0)  {strcpy(units,word2);}

      /*------------------------ search the adc channel in the trend frame---*/
      else {
        sprintf(channelName,"%s_%s",sms->name, word1);

        if (word2[0] == 'x') {
	  sscanf (word2+1,"%x", &valueI);
	  FbtFillAdc(fbt, sms->name, word1, valueI, i, nData, 1./period,
		     slope, bias, units, 32);}
        else if(strncmp(word2,"0x",2) == 0) {
	  sscanf (word2,  "%x", &valueI);
	  FbtFillAdc(fbt, sms->name, word1, valueI, i, nData, 1./period,
		     slope, bias, units, 32);}
        else if(isalpha((int) word2[0]) == 0) {
	  sscanf (word2,  "%le", &value);
	  FbtFillAdc(fbt, sms->name, word1, value, i, nData, 1./period,
		     slope, bias, units, fbt->nBits);}
        else  {
	  sscanf (word2+1,"%le", &value);
	  FbtFillAdc(fbt, sms->name, word1, value, i, nData, 1./period,
		     slope, bias, units, fbt->nBits);}}

      if (nRead != 3) break;
      data = strstr(data+strlen(word1)+strlen(word2), word3);}
  }

  return;
}
/*---------------------------------------------------------------------------*/
void FbtCleanAuxVector(FrameH *frame)
/*---------------------------------------------------------------------------*/
/* This function remove the auxiliary vectors if all the bins are filled     */
/*---------------------------------------------------------------------------*/
{
  FrAdcData *adc;
  FrVect  *isThere;
  int i;

  if (frame == NULL) return;
  if (frame->rawData == NULL) return;

  for(adc = frame->rawData->firstAdc; adc != NULL; adc = adc->next)  {
    isThere = adc->data->next;
    if(isThere == NULL) continue;

    for(i=0; i<isThere->nData; i++) {
      if(isThere->data[i] == 0) break;}
     
    if(i == isThere->nData) {
      FrVectFree(isThere);
      adc->data->next = NULL;}}

  return;
}
/*---------------------------------------------------------------------------*/
int FdFbtNew(void *arg, CfgDataType_t *data)
/*---------------------------------------------------------------------------*/
{
  FdFbt *fbt;

  /*------------create the object and add it at the end of the linked list---*/
  fbt = (FdFbt*) calloc(1,sizeof(FdFbt));
  if(fbt == NULL) CfgMsgAddFatal("malloc FdFbt failed");

  fbt->action = FdActionNew((FdAction**) arg, 
			    (void*) fbt, FdFbtProcess, "Fbt");

  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&(fbt->tag),       CfgParseGetNextString(data));
  fbt->frameL          =      CfgParseGetNextDec(data);
  fbt->nBits           =      CfgParseGetNextDec(data);
 
  fbt->trend  = NULL;
  fbt->oFile  = NULL;

  if(fbt->nBits != -64) fbt->nBits = -32;

  CfgMsgAddInfo("Add trend frame building; tag=%s trend frame length=%ds "
                "nBits=%d", fbt->tag, fbt->frameL, fbt->nBits);
  if(fbt->frameL < 1) CfgMsgAddFatal("Too small trend frame length");

  return(CFG_OK);
}
/*-----------------------------------------------------------FdFbtParseAdd---*/
void FdFbtParserAdd(FdIO *fdIO)
/*---------------------------------------------------------------------------*/
{
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_TREND_FRAME",
			 FdFbtNew, (void *) &(fdIO->actionsOut), 3,
			 CfgString, CfgDec, CfgDec);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_TREND_OF_TREND",
			 FdTOTNew, (void *) &(fdIO->actionsIn), 1, CfgDec); 
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_TREND_OF_TREND",
			 FdTOTNew, (void *) &(fdIO->actionsOut), 1, CfgDec); 

  return;
}
/*----------------------------------------------------FdFbtProcessOneFrame---*/
FrameH* FdFbtProcessOneFrame(FdFbt *fbt, FrameH *frame)
/*---------------------------------------------------------------------------*/
{
  struct timeval tS,tE;
  FrameH *frameReturned;

  gettimeofday(&tS, NULL);

  /*------------------------ write uncomplete trend frame in case of a gap---*/
  frameReturned = NULL;
  if(fbt->trend != NULL) {
    if(fbt->trend->GTimeS + fbt->trend->dt <= frame->GTimeS) {
      FbtCleanAuxVector(fbt->trend);
      frameReturned = fbt->trend;
      fbt->trend = NULL;}}

  /*---------------------------------------- select the channel to be used---*/
  FrameTag(frame, fbt->tag);

  /*-------------------------------------Convert ser data to a trend frame---*/
  FbtBuildTrendFrame (fbt, frame);

  /*--------------------------------- compute trend data for ADC and proc ---*/
  FbtBuildTrendData (fbt, frame);

  FrCListFree(fbt->listTrend);
  fbt->listTrend = NULL;
 
  /*---------------------------------------------- output frame if needed ---*/
  if(fbt->trend->GTimeS + fbt->trend->dt == frame->GTimeS + 1 &&
     frameReturned == NULL) {
    FbtCleanAuxVector(fbt->trend);
    frameReturned = fbt->trend;
    fbt->trend = NULL;}
 
  FrameUntag(frame);

  gettimeofday(&tE, NULL);
  fbt->procTime  = tE.tv_sec -tS.tv_sec + 1.e-6 * (tE.tv_usec - tS.tv_usec);

  /*------- free the input frame and return the trend frame if available-----*/
  FrameFree(frame);
  return(frameReturned);
}
/*---------------------------------------------------------------------------*/
void FdFbtProcess(FdFbt *fbt,
		  FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) {
    frame = FdFbtProcessOneFrame(fbt, frame);
    if(frame == NULL) return;}

  else { /*--------------------------------------call for the termination---*/
    if(fbt->oFile != NULL) {
      FrFileOEnd(fbt->oFile);
      fbt->oFile = NULL;}
    else {
      if(fbt->trend != NULL) {
	FbtCleanAuxVector(fbt->trend);
        next = fbt->action->next;  /*------- to write the last trend frame---*/
        if(next != NULL) next->action(next->data, fbt->trend);
        fbt->trend = NULL;}}}

  /*------------------------------------------call next action in the list---*/
  next = fbt->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}


/*---------------------------------------------------------------------------*/
void FrAdcDataSetSuffix(FrAdcData *adc, char *suffix)
/*---------------------------------------------------------------------------*/
{
  int lenName, lenSuffix;

  lenName   = strlen(adc->name);
  lenSuffix = strlen(suffix);

  adc->name = realloc(adc->name, lenName+lenSuffix+1);
  sprintf(adc->name+lenName,suffix);

  adc->data->name = realloc(adc->data->name, lenName+lenSuffix+1);
  sprintf(adc->data->name+lenName,suffix);

  if(adc->data->next == NULL) return;

  lenName   = strlen(adc->data->next->name);
  adc->data->next->name = realloc(adc->data->next->name, lenName+lenSuffix+1);
  sprintf(adc->data->next->name+lenName,suffix);

  return;
}
/*---------------------------------------------------------------------------*/
void FdTOTProcessOne(FdTOT *tot, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrAdcData **adc, *adcMin, *adcMax;
  FrVect *vect, *aux;
  int nOut, iOut, j, nGroup;
  char *suffix;

  if(frame->rawData == NULL) return;

  if(tot->period >= frame->dt) 
     CfgMsgAddFatal("To short trend frame (%g) to make new trend of, %d",
                  frame->dt, tot->period);

  for(adc = &(frame->rawData->firstAdc); *adc != NULL;) {

    vect = (*adc)->data;
    if(vect == NULL) {
      *adc = FrAdcDataFreeOne(*adc);
      continue;}
 
   if(vect->compress != 0) FrVectExpand(vect);
    nGroup = tot->period/vect->dx[0];

    suffix = vect->name+strlen(vect->name)-4;

    if(strcmp("_rms",suffix) == 0) {
      *adc = FrAdcDataFreeOne(*adc);
      continue;}
    else if(strcmp("_min",suffix) == 0) FrVectDecimateMin(vect, nGroup);
    else if(strcmp("_max",suffix) == 0) FrVectDecimateMax(vect, nGroup);
    else if(strcmp("_mean",suffix-1) == 0) FrVectDecimate(vect, nGroup, vect);
    else  { /*----------case of the slow channels without (yet) trend info---*/
      if(nGroup > 1) {/*------------------create min/max for slow channels---*/
        adcMin = FrAdcDataCopy(*adc, NULL);
        if(adcMin == NULL) CfgMsgAddFatal("FdTOTProcessOne: malloc min");
        FrAdcDataSetSuffix(adcMin, "_min");

        adcMax = FrAdcDataCopy(*adc, NULL);
        if(adcMax == NULL) CfgMsgAddFatal("FdTOTProcessOne: malloc max");
        FrAdcDataSetSuffix(adcMax, "_max");

        adcMin->next = adcMax;
        adcMax->next = (*adc)->next;
        (*adc)->next = adcMin;
 
        FrAdcDataSetSuffix(*adc, "_mean");}

      FrVectDecimate(vect, nGroup, vect);}

    (*adc)->sampleRate /= nGroup;

    /*--------------------- if there are missing data; set values to zero ---*/
    if(vect->next != NULL) {
      aux = vect->next;
      if((aux->type == FR_VECT_C) &&
        (strncmp(aux->name,"Available_data",14) == 0)) {

        if(vect->nData * nGroup != aux->nData) CfgMsgAddError(
		"FdTOTProcessOne: %d nData  error for %s", 
			frame->GTimeS, vect->name);
        else {
          char* data = aux->data;
          nOut  = aux->nData/nGroup;
          for(iOut=0; iOut<nOut; iOut++) {
            data[iOut] = 1;
            for(j=0; j<nGroup; j++) {
              if(data[iOut*nGroup + j] != 1) data[iOut] = 0;}}
   
          aux->nData /= nGroup;
          aux->nx[0] /= nGroup;
          aux->dx[0] *= nGroup;}}}

     adc = &((*adc)->next);}

  return;
}
/*---------------------------------------------------------------------------*/
void FdTOTProcess(FdTOT *tot, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdTOTProcessOne(tot, frame);

  next = tot->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdTOTNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdTOT *tot;

  tot = (FdTOT*) calloc(1,sizeof(FdTOT));
  if(tot == NULL) CfgMsgAddFatal("malloc FdTOT failed");

  tot->action = FdActionNew((FdAction**) arg, 
			  (void*) tot, FdTOTProcess, "ButTOT");

  tot->period = CfgParseGetNextDec(data);

  CfgMsgAddInfo("Add trends of trends with period = %d seconds", tot->period);

  return(CFG_OK);
}
