/*----------------------------------------------------------------------------*/
#include "FdIO.h"
#include "FdRangeGating.h"

typedef struct FdRangeGating FdRangeGating;

struct FdRangeGating {
  char*   hoftChannel;     /* name of the h(t) channel                        */
  double  mass;            /* mass used for the range computation             */
  double  freqMin;         /* starting frequency for the horizon computation  */
  int     rangeRate;       /* sample rate at which the range is computed      */
  double  dtFFT;           /* FFT duration                                    */
  double  dtTrans;         /* Time to go from 0 to 1 for the window           */
  double  minRange;        /* minimal range to apply gating                   */
  int     tRangeMoni;      /* time used to monitor the range                  */
  char*   extGateName;     /* name of the external gate channel               */
  int     mode;            // 0: classic; 1 =compute only gate flag; 2 = all
  double  marginMinRate;   // minimal rate of bad triggers to start long veto
  double  marginDuration;  // time to apply veto
  double  marginStart;     // time when the veto start
  /*-------------------------------------------end of configuration parameters*/
  double  rangePart;       /* fraction of the current range to get minRange   */
  double  rangeMedian;     /* range for the median PSD                        */
  double  median;          /* last median range value                         */
  long    nPsdProc;        // number of PSD processed
  int     nPsdSaved;       // number of PSD stored    
  int     downRejection;   // number of PSD to reject to keep one
  int     nRejected;       // number of rejected PSD (within downRejection)
  int     nFramesMedian;   /* number of frames used to compute the median     */
  int     overSampling;    /* number of sub-steps to compute the ranges       */
  FRBOOL   savePSD;        /* if FR_YES save the PSD in frames                */
  FRBOOL   hoftMissing;    // tell if the h(t) channel is missing
  FrVect*  gateFlagRaw;    // vector for the gate flag accross three frames 
  FrVect*  gateFlagMar;    // same as gateFLagRaw plus margin 
  FrVect*  rangeV;         /* vector holding the range over two frames        */
  FrVect*  prevExtGate;    // external flag vector from the previous frame
  FrVect*  prevHoft;       // h(t) vector from the previous frame
  FrvRFFT* fft;
  FrVect** psd;
  FrVect*  psdMedian;
  FrVect*  psdLast; 
  int      nPsdPerFrame;   /* number of PSD per frame */
  int      nSamplesPerFFT;
  FrvFilterB* shapingFilter;/* high pass shaping filter to reduce the dynamic */
  double   shapingFreq;    /* cut of frequency for the shaping filter         */
  int      shapingOrder;   /* order of the shaping filter                     */
  FrVect*  hoftHP;         /* last output of the shaping filter               */
  double*  valuesAtOneFreq;
  FrameH*  previousFrame;
  FdAction* action;       /* this is the action process                       */
};

FdRangeGating* LastRG = NULL;

void FdRangeGatingProcess(FdRangeGating *rg, FrameH* frame);

static int compare_double( const void *p1, const void *p2 )
{
  double x1 = *(const double *)p1;
  double x2 = *(const double *)p2;
  return (x1 > x2) - (x1 < x2);
}


typedef struct FdGating FdGating;

struct FdGating {
  char*   hoftIn;      // name of the input h(t) channel   
  char*   gateName;    // name of the gate channel 
  FRBOOL  hoftOK;
  FRBOOL  gateOK;
  FdAction* action;    // this is the action process           
};

void FdGatingProcess(FdGating *rg, FrameH* frame);

/*----------------------------------------------------------------------------*/
int FdGatingNew(void *arg,   CfgDataType_t *data)
/*----------------------------------------------------------------------------*/
{
  FdGating *ga = (FdGating*) calloc(1,sizeof(FdGating));
  if(ga == NULL) CfgMsgAddFatal("malloc FdGating failed");
  ga->hoftOK = FR_YES;
  ga->gateOK = FR_YES;

  ga->action = FdActionNew((FdAction**) arg, (void*) ga, 
                           FdGatingProcess, "Gating");

  FrStrCpy(&(ga->hoftIn),   CfgParseGetNextString(data));
  FrStrCpy(&(ga->gateName), CfgParseGetNextString(data));

  if(ga->gateName == NULL)  CfgMsgAddFatal("FdGatingNew: no gate channel");
      
  CfgMsgAddInfo(" %s will be gated when %s is set  different from 1", 
			ga->hoftIn, ga->gateName);

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
void FdGatingProcessOne(FdGating *ga, FrameH *frame)
/*----------------------------------------------------------------------------*/
{  
  FrVect* hoft = FrameFindVect(frame, ga->hoftIn);
  if(hoft == NULL) {
    if(ga->hoftOK == FR_YES) CfgMsgAddWarning("%d: %s start to be missing", 
			frame->GTimeS, ga->hoftIn);
    ga->action->state = CfgServerActive;
    ga->hoftOK = FR_NO;
    return;}
  else if(ga->hoftOK == FR_NO) {
     CfgMsgAddWarning("%d: %s is back", frame->GTimeS, ga->hoftIn);
     ga->action->state = CfgServerGolden;
     ga->hoftOK = FR_YES;}

  FrVect* gate = FrameFindVect(frame,ga->gateName);
  if(gate == NULL) {
    if(ga->gateOK == FR_YES) CfgMsgAddWarning("%d: %s start to be missing", 
			frame->GTimeS, ga->gateName);
    ga->action->state = CfgServerActive;
    ga->gateOK = FR_NO;
    return;}
  else if(ga->gateOK == FR_NO) {
     CfgMsgAddWarning("%d: %s is back", frame->GTimeS, ga->gateName);
     ga->action->state = CfgServerGolden;
     ga->gateOK = FR_YES;}

  if(hoft->nData != gate->nData) CfgMsgAddFatal("FdGatingProcessOne: "
     "%s and %s nData missmatch", ga->hoftIn, ga->gateName);
  int i;
  if      (hoft->type == FR_VECT_4R) {
    for(i=0; i<gate->nData; i++) {
      if(FrVectGetValueI(gate, i) != 1) hoft->dataF[i] = 0;}}
  else if (hoft->type == FR_VECT_8R) {
    for(i=0; i<gate->nData; i++) {
      if(FrVectGetValueI(gate, i) != 1) hoft->dataD[i] = 0;}}
  else CfgMsgAddFatal("FdGatingProcessOne: bad type");

  return;
}
/*----------------------------------------------------------------------------*/
void FdGatingProcess(FdGating *ga, FrameH* frame)
/*----------------------------------------------------------------------------*/
{
  if(frame != NULL) FdGatingProcessOne(ga, frame);

  FdAction *next = ga->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*----------------------------------------------------------------------------*/
void FdRGMarginSet(FdRangeGating *rg)
/*----------------------------------------------------------------------------*/
/* Veto h(t) for few seconds if the gating flag is apply for too long 
   This functions assumed the gating is performed on one second long frames.
------------------------------------------------------------------------------*/
{  
  int nBadMin = rg->marginMinRate * rg->rangeRate;

  //---check the central frame of the gateFlagRaw which cover three frames
  int i, j;
  int  nData = rg->rangeRate; //  this is just one frame
  for(i = 2*nData-1;  i>=nData; i--) {
    int nBad = 0;
    for(j = 0; j < nData; j++) {
      if(rg->gateFlagRaw->dataI[i+j] == 0) nBad++;}
    double timeNow = rg->gateFlagRaw->GTime + i * rg->gateFlagRaw->dx[0];
    if(nBad >= nBadMin) rg->marginStart = timeNow;
    if(timeNow - rg->marginStart < rg->marginDuration) 
         rg->gateFlagMar->dataI[i+nData] = 0;
    else rg->gateFlagMar->dataI[i+nData] = rg->gateFlagRaw->dataI[i+nData];}

  return;
}
/*----------------------------------------------------------------------------*/
int FdRGMarginNew(void *arg,   CfgDataType_t *data)
/*----------------------------------------------------------------------------*/
{
  if(LastRG == NULL) CfgMsgAddFatal(
	"No gating declared before FDIN/OUT_RANGE_GATING_MARGIN");

  LastRG->marginMinRate  = CfgParseGetNextReal(data);
  LastRG->marginDuration = CfgParseGetNextReal(data);

  CfgMsgAddInfo("  a time margin of %g s. will be apply when gating rate "
	"is larger than %g %%", LastRG->marginDuration, LastRG->marginMinRate);

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
int FdRGDownRejectionNew(void *arg,   CfgDataType_t *data)
/*----------------------------------------------------------------------------*/
{
  if(LastRG == NULL) CfgMsgAddFatal(
	"No gating declared before FDIN/OUT_RANGE_GATING_DOWN_REJECTION");

  LastRG->downRejection = CfgParseGetNextDec(data);

  CfgMsgAddInfo("  Down rejection rate is set to %d ", LastRG->downRejection);

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
double FdRGGetRangeOne(
   FrVect *psd,  /* pointer to PSD vector                                     */
   double mass,
   double freqMin,
   double freqShaping,
   int    orderShaping)
/*----------------------------------------------------------------------------*/
{
  double pi, c, G, Mpc, Msun, m1, m2, xLSO, nuLSO, norm, K, df;
  double h2, snr, freq, eta, range, freqRatio, shaping;
  int i;

  pi = acos(-1.);
  c = 299792458.;
  G = 6.6726e-11;
  Mpc = 3.0856775807e22;
  Msun = 1.98892e30;

  /*--------------------------------------------- stop at last stable orbit---*/
  if(mass < 0) {
    m1 = 1.4;
    m2 = 1.4;}
  else {
    m1 = mass;
    m2 = mass;}
  eta = m1*m2/(m1+m2)/(m1+m2);
  xLSO = (2.*(-9. - eta + sqrt(1539. - 1008.*eta + 19.*eta*eta)))/
         (3.*(81. - 57.*eta + eta*eta));
  nuLSO = 1./(pi*(m1+m2)*4.92554e-6)*pow(xLSO,3./2.);

  norm = 5./4.*pi/6.*pow(G,5./3.)/c/c/c*pow(Msun,5./3.)/Mpc/Mpc*pow(pi,-7./3.);
  K = m1*m2/(m1+m2)*pow(m1+m2,2./3.);
  norm = norm*K;

  if(nuLSO > 1024) nuLSO =1024;
  if(mass < 0) nuLSO = -mass;

  df = psd->dx[0];
  snr = 0;
  for(i=1; i<psd->nData; i++) {
    freq = i*df;
    if(freq < freqMin) continue;
    if(freq > nuLSO) break;
    h2 = norm*pow(freq,-7./3.);
    freqRatio = pow(freqShaping/freq,orderShaping);
    shaping = 1./(1+freqRatio*freqRatio);
    snr += h2*shaping/psd->dataD[i];}

  snr = sqrt(snr * 4. * df);
  
  range = snr/(8.*2.26);

  return(range);
}
/*----------------------------------------------------------------------------*/
void FdRGInit(
   FrVect *hoft, /* pointer to PSD vector                                     */
   FdRangeGating *rg)
/*----------------------------------------------------------------------------*/
{
  int i, dtFrame;
  char name[512];

  /*-------------create the FFT object to save the PSD---*/
  rg->nSamplesPerFFT = rg->dtFFT/hoft->dx[0];
  rg->fft = FrvRFFTNewT("HN",hoft->dx[0]*rg->nSamplesPerFFT, 1.e9);    
  if(rg->fft == NULL) CfgMsgAddFatal("FdHProcess: Could not create FFT");

  /*------------------ create storage for the PSD---*/
  dtFrame = 0.5*hoft->dx[0]*hoft->nData;
  rg->nFramesMedian = rg->tRangeMoni/dtFrame;
  if(rg->tRangeMoni == 0) rg->nFramesMedian = 1;
  if(rg->nFramesMedian <= 0) CfgMsgAddFatal("Gating: too short dtMedian (%d)"
     " for the frame duration (%d)", rg->tRangeMoni, dtFrame); 
  rg->nPsdPerFrame = rg->rangeRate*dtFrame;
  rg->nPsdSaved = rg->nFramesMedian * rg->nPsdPerFrame; 
  rg->psd = (FrVect**) calloc(rg->nPsdSaved, sizeof(FrVect*));
  rg->valuesAtOneFreq = (double*) calloc(rg->nPsdSaved, sizeof(double));

  double dnu = 1./(rg->nSamplesPerFFT*hoft->dx[0]);
  for(i = 0; i < rg->nPsdSaved; i++) {
     sprintf(name,"PSD_%d",i);
     rg->psd[i] = FrVectNew1D(name, FR_VECT_8R, rg->nSamplesPerFFT/2,
                              dnu,"frequency[Hz]","h^{2}/sqrt(Hz)");}
  snprintf(name, 512, "%s_PSD_median", hoft->name);
  rg->psdMedian = FrVectNew1D(name, FR_VECT_8R, rg->nSamplesPerFFT/2,
                               dnu,"frequency[Hz]","h^{2}/sqrt(Hz)");
  snprintf(name, 512, "%s_PSD_Last", hoft->name);
  rg->psdLast   = FrVectNew1D(name,   FR_VECT_8R, rg->nSamplesPerFFT/2,
                               dnu,"frequency[Hz]","h^{2}/sqrt(Hz)");

  /*---------------------------------------- create vector for range values---*/
  sprintf(name,"%s_GateRange",rg->hoftChannel);
  rg->rangeV = FrVectNewTS(name, rg->rangeRate, rg->nPsdPerFrame, -32);
  FrVectSetUnitY(rg->rangeV,"Mpc");

  /*------------------------------------create the vector for the gate flag---*/
  sprintf(name,"%s_GateFlagRaw",rg->hoftChannel);
  rg->gateFlagRaw = FrVectNewTS(name,rg->rangeRate, 3*rg->nPsdPerFrame, 32);
  FrVectSetUnitY(rg->gateFlagRaw,"flag 1=OK, 0=gated");

  /*------------------------------------create the vector for the gate flag---*/
  sprintf(name,"%s_GateFlagMargin",rg->hoftChannel);
  rg->gateFlagMar = FrVectNewTS(name,rg->rangeRate, 3*rg->nPsdPerFrame, 32);
  FrVectSetUnitY(rg->gateFlagMar,"flag 1=OK, 0=gated");

  return;
}
/*----------------------------------------------------------------------------*/
void FdRGGetRange(
   FdRangeGating *rg,
   FrameH *currentFrame,
   FrVect *hoft) // the hoft vector cover the previous and current frame
/*----------------------------------------------------------------------------*/
{
  int i,j, iBin, start, step;

  /*-------------if we have several (> 4) bins set to zero, h(t) is invalid---*/
  int nZeros = 0;
  if     (hoft->type == FR_VECT_8R)
    for(i=0; i<hoft->nData; i++) {if(hoft->dataD[i] == 0) nZeros++;}
  else if(hoft->type == FR_VECT_4R)
    for(i=0; i<hoft->nData; i++) {if(hoft->dataF[i] == 0) nZeros++;}
  else return;

  if(nZeros > 4) { //----------------------if h(t) = 0, set to zero range values 
    for(i = 0; i<rg->rangeV->nData; i++) {rg->rangeV->dataF[i] = 0;}
    return;}

  //---------------------compute the median PSD on the PSDs from previous frames
  if(rg->nPsdProc > 0) {
    int nPSD = rg->nPsdSaved;
    if(nPSD > rg->nPsdProc) nPSD = rg->nPsdProc;
    for(iBin = 0; iBin< rg->psdMedian->nData; iBin++) {
      for(i = 0; i< nPSD; i++) { 
        rg->valuesAtOneFreq[i] = rg->psd[i]->dataD[iBin];}
      qsort(rg->valuesAtOneFreq, nPSD, sizeof(double), compare_double);
      rg->psdMedian->dataD[iBin] = rg->valuesAtOneFreq[nPSD/2];}

    if(rg->savePSD == FR_YES) { //------------------save median PSD if requested
      FrProcDataNewVT(currentFrame, FrVectCopy(rg->psdMedian), 2);}

    //---------------------get range for the median PSD once we have enough PSDs
    rg->rangeMedian = FdRGGetRangeOne(rg->psdMedian, rg->mass, rg->freqMin,
  				      rg->shapingFreq, rg->shapingOrder);

    if(rg->tRangeMoni > 0) rg->minRange = rg->rangePart * rg->rangeMedian;}
    
  //--------loop on each h(t) subset, using data from the end of previous frame 
  //       with the last chunk ending at the end of the current frame
  step   = 1./(rg->rangeRate*hoft->dx[0]);
  start  = hoft->nData - rg->nSamplesPerFFT - (rg->nPsdPerFrame-1) * step; 
  start -= 0.5 * step; //-----align range values to make time series plots rigth

  for(i = 0; i < rg->nPsdPerFrame; i++) {
    FrVectZoomInI(hoft, start + i*step, rg->nSamplesPerFFT);
    FrvRFFTFor(rg->fft, hoft);
    FrVectZoomOut(hoft);
 
    double* fftD = rg->fft->output->dataD;
    double* psd  = rg->fft->output->dataD; //-----the PSD will overwrite the FFT
    for(j=0; j<rg->fft->output->nData; j++) {
       psd[j] = fftD[2*j  ]*fftD[2*j] + 
                fftD[2*j+1]*fftD[2*j+1];}

    //--------------build the PSD as the max of the local value and median value
    for(j=0; j<rg->psdLast->nData; j++) {
      if(psd[j] < rg->psdMedian->dataD[j])
           rg->psdLast->dataD[j] = rg->psdMedian->dataD[j];
      else rg->psdLast->dataD[j] = psd[j];}

    //-------get the range with a rough correction since a single PSD is biaised
    if(rg->nPsdProc > rg->nPsdSaved) // ask for enough PSD 
       rg->rangeV->dataF[i] =  1.2 *
                 FdRGGetRangeOne(rg->psdLast, rg->mass, rg->freqMin, 
				 rg->shapingFreq, rg->shapingOrder);

    //-----if the range is too low, do not save this PSD (but keep some of them)
    if(rg->rangeV->dataF[i] < rg->minRange &&
       rg->nPsdProc > rg->nPsdSaved) { // to keep all PSD at the beggining
      rg->nRejected ++;
      rg->nRejected = rg->nRejected % rg->downRejection;
      if(rg->nRejected != 0) continue;}

    //--------------------------save this PSD for next frame median computation
    int iPSD = rg->nPsdProc % rg->nPsdSaved;
    memcpy(rg->psd[iPSD]->dataD, psd, rg->psdLast->nBytes);
    rg->nPsdProc++;}

  //---------------------------------------------save corrected PSD if requested
  if(rg->savePSD == FR_YES) 
    FrProcDataNewVT(currentFrame, FrVectCopy(rg->psdLast), 2);

  return;
}
/*----------------------------------------------------------------------------*/
int FdRangeGatingNew(void *arg,   CfgDataType_t *data)
/*----------------------------------------------------------------------------*/
{
  FdRangeGating *rg;

  rg = (FdRangeGating*) calloc(1,sizeof(FdRangeGating));
  if(rg == NULL) CfgMsgAddFatal("malloc FdRangeGating failed");
  LastRG = rg;

  rg->action = FdActionNew((FdAction**) arg, (void*) rg, 
                           FdRangeGatingProcess, "RangeGating");
  rg->downRejection = 100;

  FrStrCpy(&(rg->hoftChannel), CfgParseGetNextString(data));
  rg->mass       = CfgParseGetNextReal(data);
  rg->freqMin    = CfgParseGetNextReal(data);
  rg->rangeRate  = CfgParseGetNextDec(data);
  rg->minRange   = CfgParseGetNextReal(data);
  rg->dtFFT      = CfgParseGetNextReal(data);
  rg->dtTrans    = CfgParseGetNextReal(data);
  rg->tRangeMoni = CfgParseGetNextDec(data);
  FrStrCpy(&(rg->extGateName), CfgParseGetNextString(data));
  rg->mode       = CfgParseGetNextDec(data);

  if(rg->dtFFT < 0) { /*-------- a dirty trick to active the saving of PSD---*/
       rg->dtFFT = -rg->dtFFT;
       rg->savePSD = FR_YES;} 
  else rg->savePSD = FR_NO;

  if(rg->mass > 0) 
    CfgMsgAddInfo("Range will be computed on %s, mass=%g from %g Hz"
		" rate=%d on FFT of %g s.",
 		rg->hoftChannel, rg->mass, rg->freqMin,
		rg->rangeRate,   rg->dtFFT);
  else CfgMsgAddInfo("Range will be computed on %s, mass=1.4 from %g to %g Hz"
		" rate=%d on FFT of %g s.",
 		rg->hoftChannel, rg->freqMin, -rg->mass,
		rg->rangeRate,   rg->dtFFT);
  if(rg->minRange > 0) {
    if(rg->tRangeMoni <= 0)
      CfgMsgAddInfo("  Range gating will be applied if below %g Mpc "
		     "transition will last %g s.",
 		   rg->minRange, rg->dtTrans);
    else {
      rg->rangePart = rg->minRange;
      CfgMsgAddInfo("  Range gating will be applied if below %g the "
		    "median range computed over %d s. "
		     "transition will last %g s.",
 		   rg->minRange, rg->tRangeMoni, rg->dtTrans);}
    if(rg->extGateName != NULL) 
      CfgMsgAddInfo("  %s will be used as additional gating trigger",
		rg->extGateName);
    CfgMsgAddInfo("  Only one bad PSD out of %d will be used for the median",
		rg->downRejection);}
 
  if(rg->mode == 1)
       CfgMsgAddInfo("  Create only the flag channel, not the gated h(t)");
  else if(rg->mode == 2)
       CfgMsgAddInfo("  Current and previous gated channels will be created");
  else CfgMsgAddInfo("  %s_Gated channel will be created",rg->hoftChannel);

  if(rg->savePSD == FR_YES) CfgMsgAddInfo("  PSD will be saved");

  rg->previousFrame = NULL;
  rg->hoftMissing = FR_NO;

  /*-------high pass the h(t) vector to reduce signal dynamic at low freq.---*/
  rg->shapingFreq  = 50;
  rg->shapingOrder = 4;
  rg->shapingFilter = FrvFilterButNew(rg->shapingOrder,rg->shapingFreq,0,NULL);

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
FrVect* FdGatingBuildGate(FrVect* hoft, FrVect* gateFlag, 
			double dtTrans, double tOffset)
/*----------------------------------------------------------------------------*/
/* Create a gate vector with the same bounadry and sampling rate of hoft 
   using the gateFlag information that should at least cover the hoft range.
------------------------------------------------------------------------------*/
{ 
  if(hoft     == NULL) return(NULL);
  if(gateFlag == NULL) return(NULL);
 
  //------------------------------------------------------create the gate vector
  char* name = malloc(strlen(hoft->name) + 8);
  sprintf(name,"%s_Gate",hoft->name);
  double rate = 1./hoft->dx[0];
  int nData   = hoft->nData;
  FrVect* gate = FrVectNewTS(name, rate, nData, -32);
  if(gate == NULL) CfgMsgAddFatal("Could not create %s",name);
  FrVectSetUnitY(gate,"weight");
  free(name);

  //------------------------------------------------------------check boundaries
  double dtStart = hoft->GTime - gateFlag->GTime;
  if(dtStart < 0) { 
      CfgMsgAddWarning("FdGatingBuildGate: start time missmatch: %.2f %.2f", 
				gateFlag->GTime, hoft->GTime);
      return(gate);}
  double gateFlagEnd = gateFlag->GTime + gateFlag->nData *  gateFlag->dx[0];
  double hoftEnd     = hoft    ->GTime + hoft    ->nData *  hoft    ->dx[0];
  if(hoftEnd >  gateFlagEnd) {
      CfgMsgAddWarning("FdGatingBuildGate: end time missmatch: %.2f %.2f",
				gateFlagEnd, hoftEnd);
      return(gate);}

  //-----------------------------------------------------Tukey window parameters
  int nTrans = dtTrans / hoft->dx[0];
  if(2 * nTrans  > hoft->nData) 
    CfgMsgAddFatal("Too long trantion (%d, %ld)", nTrans, hoft->nData);
  double b = FRTWOPI / nTrans;

  //---- compute the cumulative number of good samples from the end of the vector
  //--- we add a time offset since the gate flag are usually shifted (end of FFT)
  int* gateCum = malloc(gateFlag->nData * sizeof(int));
  int rateRatio = gateFlag->dx[0] / hoft->dx[0];
  int value = 2 * nTrans;
  int i, j;
  for(i = gateFlag->nData - 1; i >= 0; i--) {
    value += rateRatio;
    if(gateFlag->dataI[i] == 0) value = 0;
    gateCum[i] = value;}

  //---fill the gate. Reminder: gateFlag cover three frames
  //  the gate is for one frame, with a time offset due to the FFT size
  //  to go from 0 to 1, we need two times the transition time
  //  to go from 1 to 0 when the number of sample left is the transition time
  int offset = (dtStart -tOffset) / gate->dx[0];
  if(offset < 0) CfgMsgAddFatal("FdGatingBuildGate called with %g",tOffset);

  int lastGate = 1;
  if(gateCum[0] < nTrans) { // there is no time to go down, set all point to 0
    for(i=0; i<gateCum[0]/rateRatio;i++) {gateFlag->dataI[i] = 0;}
    lastGate = 0;}

  int jLast = 0;
  int step  = nTrans/rateRatio;
  if(step == 0) step = 1;

  for(i=0; i<gateFlag->nData; ) {
    int jStart = 0;  
    int jOff   = i*rateRatio - offset; 
    if (jOff < 0) jStart = -jOff; 
    if (jStart + jOff >= (int) gate->nData) break;
    int jEnd;
                     //------ is zero and enough good samples coming up: go to 1
    if(lastGate == 0. && gateCum[i] >= 2*nTrans) { 
        jEnd = nTrans; 
        if (jEnd + jOff >= (int) gate->nData) jEnd = gate->nData - jOff;
        for(j=jStart; j<jEnd; j++) {
            gate->dataF[j + jOff] = 0.5*(1-cos(b*(j*.5)));}
        i += step;
        lastGate = 1.;}
 
    else if(lastGate == 1 && gateCum[i] <= nTrans) { //--- it's time to go to 0  
        jEnd = nTrans; 
        if (jEnd + jOff >= (int) gate->nData) jEnd = gate->nData - jOff;
        for(j=jStart; j<jEnd; j++) {
            gate->dataF[j + jOff] = 0.5*(1+cos(b*(j*.5)));}
        i += step;
        lastGate = 0.;}

    else {   //------------------------------------ stay at previous value
        jEnd = rateRatio;
        if (jEnd + jOff >= (int) gate->nData) jEnd = gate->nData - jOff;
        for(j=jStart; j<jEnd; j++) {gate->dataF[j + jOff] = lastGate;}
	i++;}

    jLast = jEnd + jOff;}

  for(j=jLast; j<gate->nData; j++) {gate->dataF[j] = lastGate;}
 
  free(gateCum);
     
  return(gate);
}
/*----------------------------------------------------------------------------*/
void FdRangeGatingPrepOut(FdRangeGating *rg, FrameH *frame, 
				FrVect* hoft, char* suffix)
/*----------------------------------------------------------------------------*/
// this function gates and attach the hoft channel to a frame
{
  if(hoft == NULL) return;

  double timeOffset = 0;
  if(suffix != NULL && suffix[0] == 'P') timeOffset =  -frame->dt;

  //--- if a gate finish near the end of the frame, and two tapering windows 
  //   plus the range offset in time is longer than a frame, extend the gate
  int nLeft = (rg->dtFFT/2 + 2*rg->dtTrans - frame->dt)/rg->gateFlagMar->dx[0];
  if(nLeft > 0) {
    int nPerFrame = rg->gateFlagMar->nData/3;
    FRBOOL thereIsAGate = FR_NO;
    int i;
    for(i=2*nPerFrame; i<2*nPerFrame+nLeft-1; i++) {
      if(rg->gateFlagMar->dataI[i] == 0) thereIsAGate = FR_YES;}
    if(thereIsAGate == FR_YES) {
      for(i=2*nPerFrame+nLeft-1; i>=2*nPerFrame-nLeft; i--) {
        if(rg->gateFlagMar->dataI[i] == 0) break;
        rg->gateFlagMar->dataI[i] = 0;}}}

  //---compute a gate vector matching the hoft time using the gate flag channels
  FrVect* gateV = FdGatingBuildGate(hoft, rg->gateFlagMar, 
				rg->dtTrans, - rg->dtFFT/2);

  //----------------------------------------------record the effective gate flag
  char* name;
  if(suffix != NULL) {
       name = malloc(strlen(rg->hoftChannel) + 16 + strlen(suffix));
       sprintf(name,"%s_GateFlag%s",rg->hoftChannel, suffix);}
 else {name = malloc(strlen(rg->hoftChannel) + 16);
       sprintf(name,"%s_GateFlag",rg->hoftChannel);}
  FrVect* gateFlag = FrVectNewTS(name,rg->rangeRate, rg->nPsdPerFrame, 32);
  if(gateFlag == NULL) CfgMsgAddFatal("Could not create gate flag");
  FrVectSetUnitY(gateFlag,"flag 1=OK, 0=gated");
  FrProcData* proc = FrProcDataNewVT(frame, gateFlag, 1);
  if(proc == NULL) CfgMsgAddFatal("FdRangeGatingPrepOut 1");
  proc->timeOffset = timeOffset;

  int rateRatio = gateV->nData/gateFlag->nData;
  int offset = 1;  //add a one bin offset since the first gate bin is 1.
  int i;
  for(i=0; i< gateFlag->nData; i++) {
    if(gateV->dataF[i*rateRatio+offset] == 1) gateFlag->dataI[i] = 1;}

  //--------------------gate the previous hoft vector, name it gated and save it
  if(suffix != NULL) sprintf(name, "%s_Gated%s",  hoft->name, suffix);
  else               sprintf(name, "%s_Gated",    hoft->name);
  FrVectSetName(hoft, name);
  free(name);
  if(hoft->type == FR_VECT_8R) 
       for(i=0; i<hoft->nData; i++) {hoft->dataD[i] *= gateV->dataF[i];}
  else for(i=0; i<hoft->nData; i++) {hoft->dataF[i] *= gateV->dataF[i];}

  proc = FrProcDataNewVT(frame, hoft, 1); //-save gated h(t)
  if(proc == NULL) CfgMsgAddFatal("FdRangeGatingPrepOut 1");
  proc->timeOffset = timeOffset;
  proc = FrProcDataNewVT(frame, gateV,1); //-save gate vector
  if(proc == NULL) CfgMsgAddFatal("FdRangeGatingPrepOut 1");
  proc->timeOffset = timeOffset;
  
  return;
}
/*----------------------------------------------------------------------------*/
FrameH* FdRangeGatingProcessOne(FdRangeGating *rg, FrameH *frame)
/*----------------------------------------------------------------------------*/
/*
Code version with no delay.

This fonction computes the inspiral range on one frame long.
If the range is below a threshold, a gating is applied on a that data chunk 
plus two Tukey windows of half a frame each.
This is why the code is working with two frames, in order to take care of the
glitches at the overlap and apply gating with a window.
------------------------------------------------------------------------------*/
{  
  int i, nData;

  if(frame == NULL) return(frame);

  if(frame->dt != 1.) CfgMsgAddFatal("Gating expect one second long frames");

  /*------------------------------------------------------get h(t) channels---*/
  FrVect* hoftCurr = FrameFindVect(frame, rg->hoftChannel);
  if(hoftCurr == NULL) {
    if(rg->hoftMissing == FR_NO) CfgMsgAddWarning("%d: %s start to be missing", 
			frame->GTimeS, rg->hoftChannel);
    rg->action->state = CfgServerActive;
    rg->hoftMissing = FR_YES;}
  else if(rg->hoftMissing == FR_YES) {
     CfgMsgAddInfo("%d: %s is back", frame->GTimeS, rg->hoftChannel);
     rg->action->state = CfgServerGolden;
     rg->hoftMissing = FR_NO;}
  FrVectFixNAN(hoftCurr);

  FrVect* hoftPrev = rg->prevHoft;
  rg->prevHoft = FrVectCopy(hoftCurr);
  if(rg->prevHoft == NULL) {
    if(rg->hoftMissing == FR_NO) CfgMsgAddWarning(
	"FdRangeGatingProcessOne: no rg->prevHoft at %d", frame->GTimeS);

    if(rg->mode == 0) {
      FrameH* previousFrame = rg->previousFrame;
      rg->previousFrame = frame;
      return(previousFrame);}
    return(frame);}

  FrVect* extGate = rg->prevExtGate;
  if(rg->extGateName != NULL) {
    rg->prevExtGate = FrVectCopy(FrameFindVect(frame,rg->extGateName));
    if(rg->prevExtGate != NULL) {
      if(rg->prevExtGate->dx[0] != 1./rg->rangeRate) 
	CfgMsgAddFatal("FdRangeGating: sampling rate mismatch for %s",
			rg->extGateName);}}

  /*------------------------------------there was no h(t) channel up to now---*/
  if(hoftPrev == NULL && hoftCurr == NULL) { 
    if(rg->mode == 0) {
      FrameH* previousFrame = rg->previousFrame;
      rg->previousFrame = frame;
      return(previousFrame);}
    return(frame);}

  /*---------------------------------------------detect frame discontinuity---*/
  FRBOOL discontinuity = FR_YES;
  if(hoftPrev != NULL) {
    double expectedGTime = hoftPrev->GTime + hoftPrev->nData * hoftPrev->dx[0];
    if(expectedGTime == frame->GTimeS + 1e-9*frame->GTimeN) 
		discontinuity = FR_NO;}

  /*--------create a high pass filtered h(t) channel to avoid dynamic issue---*/
  /*   (the FFT for the gating are short and the h(t) LF component is large)  */
  FrVect *hoftHP;
  if(hoftCurr != NULL) {
    double min, max;
    FrVectMinMax(hoftCurr, &min, &max);
    if(max-min < 1.e-30) FrvFilterButReset(rg->shapingFilter);
    FrvFilterButProc(rg->shapingFilter,hoftCurr);
    hoftHP = FrVectCopy(rg->shapingFilter->output);}
  else {  /*------------------------ h(t) just disapear; add a dummy vector---*/
    hoftHP = FrVectCopy(rg->hoftHP);
    discontinuity = FR_YES;}

  FrVect* hoftHPPrev = rg->hoftHP;
  rg->hoftHP = hoftHP;
    
  /*---------------------the previous frame do not contains an h(t) channel---*/
  if(hoftHPPrev == NULL) {
    hoftHPPrev = FrVectCopy(hoftHP);
    discontinuity = FR_YES;}
 
  //----------------------------get HP filtered h(t) channel over the two frames
  hoftHP = hoftHPPrev;
  hoftHP->next = FrVectCopy(rg->hoftHP);
  nData = hoftHP->nData;
  hoftHP->GTime = hoftHP->next->GTime - frame->dt;
  hoftHP = FrVectConcat(hoftHP, hoftHP->GTime, 2 * nData * hoftHP->dx[0]);

  /*------------ in case of discontinuity, reset the vector to force gating---*/
  if(discontinuity == FR_YES) FrvScale(0, hoftHP, hoftHP, NULL);

  /*------------------------------------------------- initialize if needed---*/
  if(rg->nPsdPerFrame == 0) FdRGInit(hoftHP, rg);

  //---------------------------------------------------get and save range values
  FdRGGetRange(rg, frame, hoftHP);
  FrProcData* proc = FrProcDataNewVT(frame, FrVectCopy(rg->rangeV),1);
  if(proc == NULL) CfgMsgAddFatal("Could not create rangeV");
  proc->timeOffset = - rg->dtFFT/2;

  FrVectFree(hoftHP);

  /*--------------------------------------returns if no gating is requested---*/
  snprintf(rg->action->userInfo, rg->action->userInfoSize, 
	"range: %g Mpc",rg->rangeMedian);

  if(rg->minRange < 0) return(frame);

  //----------------------------shift previously computed gate flag by one frame
  int nBytes = rg->gateFlagRaw->nBytes/3;
  double dt = hoftCurr->GTime - rg->gateFlagRaw->GTime;
  if(dt != 3*frame->dt) { // in case of discontinuity, reset gate flag
    if(rg->gateFlagRaw->GTime > 0) CfgMsgAddWarning(
	"FdRangeGatingProcessOne: Frame discontinuty, dt = %.0f",dt-2*frame->dt);
    memset(rg->gateFlagRaw->data, 0, 2 * nBytes);
    memset(rg->gateFlagMar->data, 0, 2 * nBytes);}
  else { 
    memmove(rg->gateFlagRaw->data, rg->gateFlagRaw->data + nBytes, 2*nBytes);
    memmove(rg->gateFlagMar->data, rg->gateFlagMar->data + nBytes, 2*nBytes);}

  rg->gateFlagRaw->GTime = hoftCurr->GTime - 2 * frame->dt;
  rg->gateFlagMar->GTime = hoftCurr->GTime - 2 * frame->dt;

  /*-----------------------------------------computes the range gating flag---*/
  for(i=0; i<rg->rangeV->nData; i++) { 
    if(rg->rangeV->dataF[i] < rg->minRange) 
          rg->gateFlagRaw->dataI[i + 2*rg->nPsdPerFrame] = 0;
    else  rg->gateFlagRaw->dataI[i + 2*rg->nPsdPerFrame] = 1;}

  FdRGMarginSet(rg);

  /*------------ if an external gating flag is provided, merge both of them---*/
  char* name = malloc(strlen(rg->hoftChannel) + 16);
  if(extGate != NULL) {  // but first save the gateFlag as gateFlagRange
    sprintf(name,"%s_GateFlagRange",rg->hoftChannel);
    FrVectZoomInI(rg->gateFlagMar, 2*rg->nPsdPerFrame, rg->nPsdPerFrame);
    FrVect* gateFlagRange = FrVectCopy(rg->gateFlagMar);
    FrVectSetName(gateFlagRange, name);
    FrVectZoomOut(rg->gateFlagMar);
    FrProcData* proc = FrProcDataNewVT(frame, gateFlagRange, 1);
    if(proc == NULL) CfgMsgAddFatal("Could not create range flag vector");
    proc->timeOffset = -rg->dtFFT/2;

    int offset = (rg->dtFFT/2) * rg->rangeRate;
    for(i=0; i<extGate->nData; i++) {
      int extValue = 0;
      if(i < offset) {
        if(rg->prevExtGate != NULL)  extValue = FrVectGetValueI(rg->prevExtGate, 
						i - offset + extGate->nData);}
      else {
        extValue = FrVectGetValueI(extGate, i - offset);}
      if(extValue != 1) rg->gateFlagMar->dataI[i + 2*rg->nPsdPerFrame] = 0;}
    FrVectFree(extGate);}

  /*--------------------------------------------------record minimal range---*/
  sprintf(name,"%s_GateMinRange",rg->hoftChannel);
  FrVect* vect = FrVectNew1D(name, FR_VECT_4R, 1, frame->dt,"time","Mpc");
  proc = FrProcDataNewVT(frame, vect, 1);
  if(proc == NULL) CfgMsgAddFatal("Could not create XX_GateMinRange");
  proc->timeOffset = -rg->dtFFT/2; // the range is store at the end of the FFT
  if(rg->tRangeMoni > 0) rg->minRange = rg->rangePart * rg->rangeMedian;
  vect->dataF[0] = rg->minRange;
  free(name);

  //----------------------------------------------save the last gate flag vector
  nData = rg->gateFlagRaw->nData/3;
  FrVectZoomInI(rg->gateFlagRaw, 2*nData, nData);
  FrVectZoomInI(rg->gateFlagMar, 2*nData, nData);
  proc = FrProcDataNewVT(frame, FrVectCopy(rg->gateFlagRaw), 1);
  if(proc == NULL) CfgMsgAddFatal("Could not create gate flag vector");
  proc->timeOffset = -rg->dtFFT/2; // the range is store at the end of the FFT
  proc = FrProcDataNewVT(frame, FrVectCopy(rg->gateFlagMar), 1);
  if(proc == NULL) CfgMsgAddFatal("Could not create gate flag vector");
  proc->timeOffset = -rg->dtFFT/2; // the range is store at the end of the FFT
  FrVectZoomOut(rg->gateFlagRaw);
  FrVectZoomOut(rg->gateFlagMar);

  //------if only gating flag requested, return without introducting frame delay
  if(rg->mode == 1) {
    FrVectFree(hoftPrev);
    return(frame);}

  //------if producing latest and previous gated channels
  if(rg->mode == 2) {
    FdRangeGatingPrepOut(rg, frame, FrVectCopy(hoftCurr), "Last");
    FdRangeGatingPrepOut(rg, frame, hoftPrev, "Previous");
    return(frame);}

  //-----------returns previous frame with classical gating
  FrameH* previousFrame = rg->previousFrame;
  rg->previousFrame = frame;
  if(previousFrame == NULL) return(previousFrame);
  if(hoftPrev      == NULL) return(previousFrame);

  FdRangeGatingPrepOut(rg, previousFrame, hoftPrev, NULL);

  return(previousFrame);
}
/*----------------------------------------------------------------------------*/
void FdRangeGatingProcess(FdRangeGating *rg, FrameH* frame)
/*----------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) frame = FdRangeGatingProcessOne(rg, frame);

  next = rg->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*----------------------------------------------------------------------------*/
void FdRangeGatingAdd(FdIO* fdIO)
/*----------------------------------------------------------------------------*/
{
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_RANGE_GATING_DOWN_REJECTION",
			 FdRGDownRejectionNew, NULL, 1, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_RANGE_GATING_DOWN_REJECTION",
			 FdRGDownRejectionNew, NULL, 1, CfgDec);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_RANGE_GATING_MARGIN",
			 FdRGMarginNew, NULL, 2, CfgReal, CfgReal);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_RANGE_GATING_MARGIN",
			 FdRGMarginNew, NULL, 2, CfgReal, CfgReal);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_RANGE_GATING_NEW",
			 FdRangeGatingNew, (void *) &(fdIO->actionsIn), 10,
			 CfgString, CfgReal, CfgReal, CfgDec, CfgReal, 
			 CfgReal,   CfgReal, CfgDec, CfgString, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_RANGE_GATING_NEW",
			 FdRangeGatingNew, (void *) &(fdIO->actionsOut), 10,
			 CfgString, CfgReal, CfgReal, CfgDec, CfgReal, 
			 CfgReal,   CfgReal, CfgDec, CfgString, CfgDec);

  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_GATING_CHANNEL",
			 FdGatingNew, (void *) &(fdIO->actionsIn), 2,
			 CfgString, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_GATING_CHANNEL",
			 FdGatingNew, (void *) &(fdIO->actionsOut), 2,
			 CfgString, CfgString);
 return;
}
