/*-------------------------------------------------------------------FdCm.c---*/
#define _GNU_SOURCE
#include <FdIO.h>
#include "FdChList.h"
#include "FdFrMrgr.h"
#include "FdGetFirstFr.h"
#include "FdCm.h"
#include "FdIDir.h"
#include <pthread.h>
#include <sys/types.h>
#include <sys/syscall.h> 

typedef struct FdICm          FdICm;
typedef struct FdOCmServer    FdOCmServer;
typedef struct FdOCmC         FdOCmC;
typedef struct FdOCmCLRequest FdOCmCLRequest;
typedef struct FdSleep        FdSleep;

struct FdICm                 /* Use for input frame from Cm                  */
{
  int       frameListSize; /* size of the input list of frames              */
  int       maxSources;    /* maximum number of CM sources (sender) allowed */
  int       nSenders;      /* number of instance in the senders list        */
  char**    senders;       /* list of names for the Cm source(s)            */
  char*     sendersChksum; /* tell if the source provide a chekcsum         */
  FrameH**  frameList;     /* list of frame in the input buffer             */
  void**    frBufList;     /* list of frame as in the Cm buffer             */
  int*      sendersIdB;    /* list of sender id for each frame in frBufList */
  int*      sendersIdF;    /* list of sender id for each frame in frameList */
  int       iFrBufW;       /* index of the last frame written in frBufList  */
  int       iFrBufR;       /* index of the last frame read in frBufList     */
  int       iFrameW;       /* index of the last frame written in framesList */
  int       iFrameR;       /* index of the last frame read in framesList    */
  FdFrMrgr* frMrgr;        /* link to the frame merger is requested         */
  FdGetFirstFr* getFirstFr;/* link to the get first frame is requested      */
  char*     DsName;        /* if not NULL, data are sent by this dataSender */
  char*     DsiFileName;   /* data are remotely read from this file         */
  int       DsiFileStart;  /* starting GPS time of remote data to be read   */
  int       DsiFileLength; /* duration of remote data to be read            */ 
  FdIDir*   iDir;          /* link list of /dev/shm directories if used     */
  pthread_t    thrdId;     /* thread info                             */
  sem_t        semNewData; /* tell is new data should be processed    */
  pthread_mutex_t lock;    /* mutex to control the frame passing      */
  FdAction* action;        /* action for this structure                     */
};

struct FdOCmServer         /* respond to request for additional Cm output   */
{
  int       maxConsumer;   /* maximum number of frame consumer              */
  int       chListTimeSpam;/* duration to keep a channel in the lists       */
  int       queueSize;     /* default output queue size (number of frames)  */
  int       nRemoved;      /* last number of duplicated channels removed    */  
  /*------------------------- end of input parameters                       */
  FdCList*  listAdc;       /* list of FrAdcData channels                    */
  FdCList*  listProc;      /* list of FrProcData channels                   */
  FdCList*  listSer;       /* list of FrSerData channels                    */
  FdCList*  listSim;       /* list of FrSimData channels                    */
  FdOCmCLRequest* requests;/* list of channel list requests                 */
  FdAction* action;        /* action for this structure                     */
};

struct FdOCmC              /* structure for each frame consumer             */
{
  char*      name;         /* Fd frame receiver name                        */
  char*      tag;          /* list of channels to be selected               */
  int        maxTry;       /* maximum number of send try before remove it   */
  int        nTry;         /* current number of send try                    */
  int        queueSize;    /* maximum number of message in the sending queue*/
  int        compChksum;   /* tell to compute or not the frame checksum     */
  /*------------------------- end of input parameters                       */
  FRBOOL     flushing;     /* tell if we have to flush the queue            */
  long       lastSize;     /* size of the last frames send                  */
  char**     buf;          /* buffer to host the frames in the message      */
  CmMessage* messages;     /* list of message used to send frames           */
  CmMessageIterator *iterators; /* list of iterator messages                */
  FdAction*  action;       /* action for this structure                     */
};

struct FdOCmCLRequest        /* to store who request a list of channel      */
{
  char*           name;      /* Cm name of the requester                    */
  FdOCmCLRequest* next;      /* action for this structure                   */
};

struct  FdSleep              /* for test: add some waiting time            */
{
  int sleepTime;             /* sleep time in us                           */
  FdAction*  action;       /* action for this structure                     */
};

/*---------------------------------------------global variables--- 
  they are needed since we cannot associate a pointer to a Cm handler---------*/
			      
FdOCmServer *FdOCmSrvr = NULL;
FdICm *FdIn = NULL;
FRBOOL FdICmThread = FR_NO;

/*--------------------------------------------------------private functions---*/
void FdICmProcess(FdICm *iCm, FrameH *frame);
void FdICmThreadMain(FdICm* iCm);
int  FdICmWaitNew (void *arg, CfgDataType_t *data );
CmMessageStatus FdICmHandlerFrame (CmMessage message, char* from, char* server);
CmMessageStatus FdOCmHandlerAdd   (CmMessage message, char* from, char* server);
CmMessageStatus FdOCmHandlerCList (CmMessage message, char* from, char* server);
CmMessageStatus FdOCmHandlerRemove(CmMessage message, char* from, char* server);
FdAction* FdOCmFindAction (char *name);
int  FdOCmServerNew(void *arg, CfgDataType_t *data);
void FdOCmServerProcess(FdOCmServer *oCm, FrameH *frame);
FdAction* FdOCmCAdd (char *name, char *tag, int queueSize, int maxTry, int compChksum);
void FdOCmCProcess(FdOCmC *out, FrameH *frame);
void FdOCmCProcessOne(FdOCmC *out, FrameH *frame);
int  FdOCmCPutFrameInMessage (FdOCmC *out, int i, FrameH *frame, int compress);
void FdOCmCRemove (FdAction *action);
void FdOCmSendCList (char* from);
int  FdOCmGetNCsmr ();

/*----------------------------------------------------------------------------*/
int FdICmAdd(FdAction** root, int frameListSize, int maxSources)
/*----------------------------------------------------------------------------*/
{
  if(frameListSize <= 0)
    CfgMsgAddFatal("The Cm input frame buffer size must be larger than zero");

  FdIn = calloc(1,sizeof(FdICm));
  if(FdIn == NULL) CfgMsgAddFatal("Malloc FdICm failed");

  FdIn->action = FdActionNew(root, (void*) FdIn, FdICmProcess, "FDIN_CM");
  free(FdIn->action->userInfo);
  FdIn->action->userInfoSize = 400 + maxSources*100;
  FdIn->action->userInfo = malloc(FdIn->action->userInfoSize);
  FdIn->action->userInfo[0] = '\0';
  free(FdIn->action->serData);
  FdIn->action->serDataSize = 400 + maxSources*100;
  FdIn->action->serData = malloc(FdIn->action->serDataSize);
  FdIn->action->serData[0] = '\0';
  
  FdIn->frameListSize = frameListSize;
  FdIn->maxSources    = maxSources;

  /*------------------------- allocate list for the frames arriving from Cm---*/
  FdIn->frameList = (FrameH**) calloc(FdIn->frameListSize, sizeof(FrameH*));
  if(FdIn->frameList == NULL) CfgMsgAddFatal("Malloc failed for frameList (%d)",
					     FdIn->frameListSize);
  FdIn->iFrameR = 0;
  FdIn->iFrameW = 0;

  FdIn->frBufList = (void**) calloc(FdIn->frameListSize, sizeof(void*));
  FdIn->iFrBufR = 0;
  FdIn->iFrBufW = 0;

  /*---------------------allocate buffer for the monitoring of the senders---*/
  FdIn->nSenders = 0;
  if(FdIn->maxSources > 0) {
    FdIn->senders      = (char **) calloc(FdIn->maxSources, sizeof(char *));
    FdIn->sendersChksum = (char *) calloc(FdIn->maxSources, sizeof(char));
    FdIn->sendersIdB    = (int *) calloc(FdIn->frameListSize, sizeof(int));
    FdIn->sendersIdF    = (int *) calloc(FdIn->frameListSize, sizeof(int));
    if(FdIn->sendersChksum == NULL || FdIn->sendersIdF == NULL) 
      CfgMsgAddFatal("FdICm malloc senders failed (%d):",FdIn->maxSources);}

  /*----------------------------------------install receiving frame handler---*/
  CfgMsgAddInfo("Get frame from Cm; frameListSize=%d maxSource=%d",
		FdIn->frameListSize, FdIn->maxSources);

  CmMessageInstallHandler (FdICmHandlerFrame, "FdFrame");

  /*-----skip the remaining if no threading for the frame reading requested---*/
  if(FdICmThread == FR_NO) return(CFG_OK);

  /*---------------------------------------------------initialize semaphore---*/
  sem_init(&FdIn->semNewData,  0, 0); 

 /*-----------------------------------------------------------create thread---*/
  int irc = pthread_create(&(FdIn->thrdId), NULL, 
			(void *(*)(void *))FdICmThreadMain,  (void *)FdIn);
  if(irc == -1) CfgMsgAddFatal("FdTICmThreadMain creation failed");

  /*------------put the thread on the same CPU to avoid cache inefficiency---*/
  pthread_t myId;
  cpu_set_t cpuset;
  myId = pthread_self();   //----------get thread id for the main---
  pthread_getaffinity_np(myId,         sizeof(cpu_set_t), &cpuset);
  pthread_setaffinity_np(FdIn->thrdId, sizeof(cpu_set_t), &cpuset);

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
CmMessageStatus FdICmHandlerFrame(CmMessage message, 
				  char* sender,
				  char* serverName)
/*----------------------------------------------------------------------------*/
{
  int iSender, nBytes, iWrite, gtimeS, nFrames;
  char *buf;

  /*---------------------------------------decode message and extract frame---*/
  CmMessageGetText  (message);          /* for the frame name       */
  CmMessageGetInt   (message);          /* for the old run number   */
  CmMessageGetInt   (message);          /* for the old frame number */
  gtimeS = CmMessageGetInt   (message); /* for GTimeS               */
  CmMessageGetInt   (message);          /* for GTimeN               */
  CmMessageGetInt   (message);          /* for the trigger number   */
  nBytes = CmMessageGetInt   (message);
  buf    = CmMessageGetArray (message, CmMessageChar, &nBytes);

  nFrames = FrameBufCheck(buf, nBytes, FR_YES);
  if(nFrames != 1) {CfgMsgAddError(
      "FdICm: Corrupted Cm message from %s with frame, error=%d GPS=%d nBytes=%d",
                   sender, nFrames, gtimeS, nBytes);}

  if(FdIn->action->debugLvl > 0) 
    CfgMsgAddInfo("Received frame %d from %s", gtimeS, sender);

  /*--------------------------------------------- Check the source of frame---*/
  if(FdIn->maxSources > 0) {
    if(sender == NULL) {
      CfgMsgAddError("FdICm: Cm sender == NULL"); 
      return(CmMessageBreak);}

    for(iSender=0; iSender<FdIn->nSenders; iSender++) {
      if(strcmp(sender, FdIn->senders[iSender]) == 0) break;}

    if(iSender == FdIn->nSenders) { 
      CfgMsgAddInfo("FdICm: new Cm source of frames from %s GPS=%d",
		    sender, gtimeS);
      if(FdIn->nSenders >= FdIn->maxSources) {
	CfgMsgAddError("FdICm: reject frame from %s: too many Cm source (%d)",
		       sender, FdIn->nSenders);
	return(CmMessageBreak);}

      FrStrCpy(&(FdIn->senders[FdIn->nSenders]),sender);
      FdIn->nSenders++;}}
  else {
    iSender = 0;}

  /*---------------------------------------------report source with checksum---*/
  if(FdIn->sendersChksum[iSender] != buf[39]) {
     FdIn->sendersChksum[iSender] = buf[39];
    if(buf[39] == 1) 
      CfgMsgAddInfo("FdICm: checksum is activated for data from %s", sender);
    else
      CfgMsgAddInfo("FdICm: checksum is now OFF for data from %s", sender);}

  /*--------------------------------------------------get input buffer index---*/
  iWrite = FdIn->iFrBufW % FdIn->frameListSize;

  /*------------------------------------------feed the frame decoding thread---*/
  if(FdICmThread == FR_YES) {
    if(FdIn->frBufList[iWrite] != NULL) {
      CfgMsgAddError("FdICm: input buffer overflow; reject frame %d "
		   "from %s", gtimeS, sender);
      return(CmMessageBreak);}

    FdIn->frBufList[iWrite] = malloc(nBytes);
    if(FdIn->frBufList[iWrite] == NULL) 
      CfgMsgAddFatal("malloc failed in FdICmHandlerFrame for %d bytes",nBytes);
    memcpy(FdIn->frBufList[iWrite], buf, nBytes);

    if(FdIn->maxSources > 0) FdIn->sendersIdB[iWrite] = iSender;}

  /*--------------------------- or decode directly the frame in this handler---*/
  else {
    if(FdIn->frameList[iWrite] != NULL) {
      CfgMsgAddError("FdICm: input frame buffer overflow; reject frame %d "
                   "from %s", gtimeS, sender);
      return(CmMessageBreak);}

    FrameH* frame = FrameReadFromBuf (buf, nBytes, FdIOGetCompressIn());
    if(frame == NULL) {CfgMsgAddError("FdICm: Could decode frame %d "
		"from %s last error:%s", gtimeS, sender, FrErrorGetHistory());
      return(CmMessageBreak);}
    if(FdIn->action->tag != NULL) {
      FrameTag(frame, FdIn->action->tag);
      FrameRemoveUntagged(frame);}
    FdIn->frameList[iWrite] = frame;

    if(FdIn->maxSources > 0) FdIn->sendersIdF[iWrite] = iSender;}

  /*----------------------------------record source info and increment index---*/

  FdIn->iFrBufW ++;

  /*---------------wake up the FrameReadFromBuf thread if threading is used----*/
  if(FdICmThread == FR_YES) sem_post(&FdIn->semNewData);

  return(CmMessageBreak);
}
/*-----------------------------------------------------------------------------*/
void FdICmThreadMain(FdICm* iCm)
/*-----------------------------------------------------------------------------*/
{
  int tmo;
  int nFrames = 0;

  int compress = FdIOGetCompressIn();
  int tid = syscall(__NR_gettid);
    
  CfgMsgAddInfo("Start FdTICmThreadMain; tid=%d pid=%d", tid, getpid());

  /*---------------------------------- Set the thread as Asynchronous thread---*/
  pthread_setcancelstate( PTHREAD_CANCEL_ENABLE, &tmo );
  pthread_setcanceltype ( PTHREAD_CANCEL_ASYNCHRONOUS, &tmo );

  /*----------------------- process all frame that are passed to this thread---*/
  while (1) {

    /*----------------------------------------------wait for the next frame---*/
    sem_wait(&iCm->semNewData);

    /*-----------------------get the last buffer and read the frame from it---*/
/*    pthread_mutex_lock (&iCm->lock);  --- this line may not be needed */
    int iFrBufR  = iCm->iFrBufR % iCm->frameListSize;
    int senderId = iCm->sendersIdB[iFrBufR];
    char* buf = iCm->frBufList[iFrBufR];
    iCm->frBufList[iFrBufR] = NULL;
    iCm->iFrBufR++;
/*    pthread_mutex_unlock (&iCm->lock); --- this line may not be needed */

    //-----------------------read the frame from the last buffer
    nFrames++;
    FrameH* frame = FrameReadFromBuf (buf, 1.e12, compress);
    free(buf);
    if(frame == NULL) {CfgMsgAddError(
      "FdICm: Could read frame from buffer, from %s; last error:%s",
                    iCm->senders[senderId], FrErrorGetHistory());
      continue;}
 
    /*------------------------------------------select channels if requested---*/
    if(iCm->action->tag != NULL) {
      FrameTag(frame, iCm->action->tag);
      FrameRemoveUntagged(frame);}

    /*-----------------------if space is available put the frame in the list---*/
    int iWrite = iCm->iFrameW % iCm->frameListSize;
    if(iCm->frameList[iWrite] != NULL) {
      CfgMsgAddError("FdICm: input frame buffer overflow (%d); reject frame %d "
		   "from %s", iWrite, frame->GTimeS, iCm->senders[senderId]);
      continue;}

    iCm->frameList [iWrite] = frame;
    iCm->sendersIdF[iWrite] = senderId;
    iCm->iFrameW ++;
  }

  /*-------------------------------- Get thread clock Id and then clock time---*/
  struct timespec currTime;
  clockid_t threadClockId;
  pthread_getcpuclockid(iCm->thrdId, &threadClockId);
  clock_gettime(threadClockId, &currTime);
  double totTime = currTime.tv_sec + 1.e-9*currTime.tv_nsec;

  CfgMsgAddInfo("FdICm thread ended; CPU: %g s for %d frames", totTime, nFrames);

  return;
}
/*----------------------------------------------------------------------------*/
int FdICmThreadNew(void *arg, CfgDataType_t *data )
/*----------------------------------------------------------------------------*/
{
  if(*((FdAction**) arg) != NULL) 
    CfgMsgAddFatal("FDIN_CM_THREAD must be called before defining the source of frames");

  FdICmThread = FR_YES;
  CfgMsgAddInfo("Threading will be use to decode frames from Cm messages");

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
int FdICmWaitNew(void *arg, CfgDataType_t *data )
/*----------------------------------------------------------------------------*/
{
  int frameListSize, maxSources;

  if(*((FdAction**) arg) != NULL) 
    CfgMsgAddFatal("FDIN_CM_WAIT must be the first input key");

  frameListSize = CfgParseGetNextDec(data);
  maxSources    = CfgParseGetNextDec(data);

  FdICmAdd((FdAction**) arg, frameListSize, maxSources);

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
void FdICmProcess(FdICm *iCm, FrameH *frame)
/*----------------------------------------------------------------------------*/
{
  FdAction *next;
  int  iFrameR;
  char *sender;
  FdIDir *iDir;
  FrameH *framePart;

  iDir = FdIn->iDir;

  /*------------- loop as long as we get frame parts and not a full frame---*/
  while(1) {

    /*--------- get a frame from the input buffer filled by the Cm handler---*/
    CfgMsgSendWithTimeout(0.);
    iFrameR = FdIn->iFrameR % FdIn->frameListSize;
    frame = FdIn->frameList[iFrameR];

    /*------------------------------remove this frame from the input buffer---*/
    if(frame != NULL) {
      FdIn->frameList[iFrameR] = NULL;
      FdIn->iFrameR++;}

    /*---------------------------------------call frame merger if requested---*/
    if(FdIn->frMrgr     == NULL &&
       FdIn->getFirstFr == NULL) break;

    sender = FdIn->senders[FdIn->sendersIdF[iFrameR]];

    /*----------- if we are taking input from /dev/shm, look for new frame---*/
    if(frame == NULL) {
      for(; iDir != NULL; iDir = iDir->next) {
        if(frame != NULL) break; /*- do the check here to have the next iDir-*/
        frame  = FdIDirProcessOne(iDir, FdIn->action->debugLvl); 
        sender = iDir->dirName;}}

    framePart = frame;
    if(FdIn->frMrgr  != NULL)
         frame = FdFrMrgrProc    (FdIn->frMrgr,     frame, sender);
    else frame = FdGetFirstFrProc(FdIn->getFirstFr, frame, sender);
    if(frame     != NULL) break;
    if(framePart == NULL) break;}

  /*--------------------------------------------prepare to output the frame---*/
  if(FdIn->frMrgr != NULL) {
    if(FdIn->frMrgr->nMissing == 0) FdIn->action->state = CfgServerGolden;
    else                            FdIn->action->state = CfgServerActive;
    snprintf(FdIn->action->userInfo, FdIn->action->userInfoSize,
	"%s;",FdIn->frMrgr->status);
    snprintf(FdIn->action->serData, FdIn->action->serDataSize, 
	FdIn->frMrgr->serData);}
  else if(FdIn->getFirstFr != NULL) {
    snprintf(FdIn->action->userInfo, FdIn->action->userInfoSize,
	"%s;",FdIn->getFirstFr->status);
    snprintf(FdIn->action->serData, FdIn->action->serDataSize, 
	FdIn->getFirstFr->serData);}

  if(frame == NULL) return;
 
  /*-----------------------------------------------remove unwanted channels---*/
  if(FdIn->action->tag != NULL) {
    FrameTag(frame, FdIn->action->tag);
    FrameRemoveUntagged(frame);}
 
  /*----------------------------------------------record last frame arrival---*/
  if(frame != NULL) {
    FdIn->action->lastTime = time(NULL);
    FdIn->action->frameDt = frame->dt;}

  /*---------------------------------------------call the next ouput action---*/
  next = FdIn->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*----------------------------------------------------------------------------*/
int FdICmConnectNew(void *arg, CfgDataType_t *data )
/*----------------------------------------------------------------------------*/
{
  FdAction **root;
  int irc, queueSize, nRetry;
  FdIO* fdIO;

  fdIO = FdIOGetRoot();
  FrStrCpy(&fdIO->dynamicSrc, CfgParseGetNextString(data));
  queueSize = CfgParseGetNextDec(data);
  nRetry    = CfgParseGetNextDec(data);
  CfgMsgAddInfo("Request frames from %s; input frame buffer size:%d nRetry:%d", 
		fdIO->dynamicSrc, queueSize, nRetry);

  root = arg;
  if((*root)->tag != NULL) 
    irc = FdRequestFrames(fdIO->dynamicSrc, CfgGetCmName(),(*root)->tag, 
		       queueSize, nRetry);
  else 
    irc = FdRequestFrames(fdIO->dynamicSrc, CfgGetCmName(), "*", 
		       queueSize, nRetry);

  if(irc == 0) CfgMsgAddFatal(" could not complete frame request");
  
  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
int FdIDataSenderNew(void *arg, CfgDataType_t *data )
/*----------------------------------------------------------------------------*/
/* This function first creates the FDIN_CM object which has only one source 
   and needs only a few frames in the input buffer                            */
/*----------------------------------------------------------------------------*/
{
  FdIO *fdIO;

  if(*((FdAction**) arg) != NULL) 
    CfgMsgAddFatal("FDIN_DATA_SENDER must be the first input key");


  FdICmAdd((FdAction**) arg, 5, 1);

  FdIn->DsName        = CfgParseGetNextString(data);
  FdIn->DsiFileStart  = CfgParseGetNextDec(data);
  FdIn->DsiFileLength = CfgParseGetNextDec(data);

  /*------------------------------------send the request to the data sender---*/
  FdDataSenderRequest(FdIn->DsName, CfgGetCmName(), 
	              FdIn->action->tag, FdIn->DsiFileStart, 
		      FdIn->DsiFileLength, 2);

  fdIO = FdIOGetRoot();
  FrStrCpy(&fdIO->dynamicSrc, CfgParseGetNextString(data));

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
void FdDataSenderRequest (char *dsName, char *destName, char *tag, 
	         	  int gpsStart, int nSeconds, int queueSize)
/*----------------------------------------------------------------------------*/
{
  CmMessage message;

  CfgMsgAddInfo("Sending Data request to %s:  dest=%s"  
		" tag=%s  gpsStart=%d  nSeconds=%d queueSize=%d\n",
		dsName, destName, tag, gpsStart, nSeconds, queueSize);

  message = CmMessageNew();
  CmMessagePutText (message,destName);
  CmMessagePutText (message,tag);
  CmMessagePutInt  (message,gpsStart);
  CmMessagePutInt  (message,nSeconds);
  CmMessagePutInt  (message,queueSize);

  CmMessageSetType (message,"FdGetFrameSet");
  CmMessageSend    (message,dsName);

  CmMessageDelete (message);

  return;
}
/*---------------------------------------------------------------------------*/
CmMessageStatus FdFrMrgrHandlerAdd (CmMessage message, 
				    char* from, char* server)
/*---------------------------------------------------------------------------*/
{
  char *name;
 
  name = CmMessageGetText(message);
  CfgMsgAddInfo ("FdFrMrgHandler: %s asked to add %s to the source list",
		 from, name);

  if(FdIn->frMrgr == NULL) { 
    CfgMsgAddError("This process is not doing frame merging");
    return(CmMessageOk);}

  if(FdFrMrgrAddSource(FdIn->frMrgr, name) != 0) return(CmMessageOk);

  //-----------------------------------------------------if source is from shm
  if(strstr(name, "/") != NULL) {
    FdIDir* iDir = FdIDirNew(name, -1);
    iDir->next = FdIn->iDir;
    FdIn->iDir = iDir;}

  return(CmMessageOk);
}
/*---------------------------------------------------------------------------*/
CmMessageStatus FdFrMrgrHandlerRm(CmMessage message, 
				  char* from, char* server)
/*---------------------------------------------------------------------------*/
{
  char *name;
 
  name = CmMessageGetText(message);
  CfgMsgAddInfo ("FdFrMrgHandler: %s asked to remove %s from the source list",
		 from, name);

  if(FdIn->frMrgr == NULL) { 
    CfgMsgAddError("This process is not doing frame merging");
    return(CmMessageOk);}

  //----------------------------------------------remove it to the frame merger
  if(FdFrMrgrRemoveSource(FdIn->frMrgr, name) != 0) return(CmMessageOk);

  //---------------------------------if source is from shm, remove frame reading
  if(strstr(name, "/") != NULL) {
    FdIDir** iDir = &(FdIn->iDir); 
    while(*iDir != NULL) {
      if(strcmp(name, (*iDir)->dirName) != 0) iDir = &((*iDir)->next);
      else {
          CfgMsgAddInfo(" Remove %s reading", name);
          FdIDir* next = (*iDir)->next; 
          FdIDirFree(*iDir);
          *iDir = next;}}}

  return(CmMessageOk);
}
/*----------------------------------------------------------------------------*/
int FdIFrMrgrNew(void *arg, CfgDataType_t *data )
/*----------------------------------------------------------------------------*/
{
  char *sourcesList;
  int nFrames, nSources, i;
  FdFrMrgr *frMrgr;
  FdIDir *iDir;

  if(*((FdAction**) arg) != NULL) 
    CfgMsgAddFatal("FDIN_FRAME_MERGER must be the first input key and not %s",
                   (*((FdAction**) arg))->type);

  nFrames     = CfgParseGetNextDec(data);
  sourcesList = CfgParseGetNextString(data);

  /*---------------------get the frame merger to know the number of sources---*/
  frMrgr = FdFrMrgrNew(nFrames, sourcesList);
  nSources = frMrgr->nSources;

  /*----create the FDIN_CM object, reserving extra space for future sources---*/
  FdICmAdd((FdAction**) arg, nFrames*nSources, nSources+50);
  FdIn->frMrgr = frMrgr;

  frMrgr->debug = (*((FdAction**) arg))->debugLvl;

  /*--------------look if some input are from /dev/shm or a local directory---*/
  for(i = 0; i<nSources; i++) {
    if(strstr(frMrgr->sNames[i], "/") == NULL) continue;
    iDir = FdIDirNew(frMrgr->sNames[i], -1);
    iDir->next = FdIn->iDir;
    FdIn->iDir = iDir;}

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
int FdIGetFirstFrNew(void *arg, CfgDataType_t *data )
/*----------------------------------------------------------------------------*/
{
  if(*((FdAction**) arg) != NULL) 
    CfgMsgAddFatal("FDIN_GET_FIRST_FRAM must be the first input key and not %s",
                   (*((FdAction**) arg))->type);

  int latencyMax  = CfgParseGetNextDec(data);
  char* sourcesList = CfgParseGetNextString(data);

  /*---------------------get the frame merger to know the number of sources---*/
  FdGetFirstFr* getFirstFr = FdGetFirstFrNew(latencyMax, sourcesList);
  int nSources = getFirstFr->nSources;

  /*----create the FDIN_CM object, reserving extra space for future sources---*/
  FdICmAdd((FdAction**) arg, latencyMax*nSources, nSources+50);
  FdIn->getFirstFr = getFirstFr;

  /*--------------look if some input are from /dev/shm or a local directory---*/
  for(int i = 0; i<nSources; i++) {
    if(strstr(getFirstFr->sNames[i], "/") == NULL) continue;
    FdIDir* iDir = FdIDirNew(getFirstFr->sNames[i], -1);
    iDir->next = FdIn->iDir;
    FdIn->iDir = iDir;}

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
void FdSleepProcess(FdSleep *s, FrameH *frame)
/*----------------------------------------------------------------------------*/
{
  FdAction *next;

  usleep(s->sleepTime);

  /*---------------------------------------------call the next ouput action---*/
  next = s->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*----------------------------------------------------------------------------*/
int FdSleepNew(void *arg, CfgDataType_t *data )
/*----------------------------------------------------------------------------*/
{
  FdIO *fdIO;
  FdSleep *s;

  s = (FdSleep*) calloc(1,sizeof(FdSleep));
  
  s->sleepTime = CfgParseGetNextDec(data);

  fdIO = FdIOGetRoot();
  s->action = FdActionNew(&(fdIO->actionsOut), (void*) s,
                          FdSleepProcess, "Sleep");
 
  CfgMsgAddInfo("Sleep %d us",s->sleepTime);

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
FdAction*  FdOCmCAdd (char *name, char *tag, int queueSize, int maxTry, int compChksum)
/*----------------------------------------------------------------------------*/
{
  FdOCmC *out;
  FdAction *action;

  if(queueSize < 0) queueSize = 0;

  /*----------------if the consumer is already there just update parameters---*/
  action = FdOCmFindAction(name);

  if (action != NULL) {
    CfgMsgAddWarning( "FdOCmCAdd> %s already known; update it (%s %d %d)",
		      name, tag, queueSize, maxTry);
    out = (FdOCmC*) action->data;
    free(out->tag);
    if(FrStrCpy (&(out->tag), tag) == NULL) CfgMsgAddFatal("FdOut malloc");
    out->maxTry = maxTry;
    out->queueSize = queueSize;
    return(NULL);}

  /*---------------------------------------------------- add a new consumer---*/
  out = calloc(1, sizeof(struct FdOCmC));
  if (out == NULL)  CfgMsgAddFatal("FdOutAdd> malloc failed");

  if (FrStrCpy (&(out->name), name) == NULL) CfgMsgAddFatal("FdOut malloc");
  if (FrStrCpy (&(out->tag),  tag)  == NULL) CfgMsgAddFatal("FdOut malloc");

  out->nTry       = 0;
  out->maxTry     = maxTry;
  out->queueSize  = queueSize;
  out->lastSize   = 65536;
  out->flushing   = FR_NO;
  out->compChksum = compChksum;

  if(queueSize <= 0) queueSize = 1;/*--we need to store at least one message--*/
  out->messages  = calloc(queueSize, sizeof(CmMessage)); 
  out->iterators = calloc(queueSize, sizeof(CmMessageIterator));
  out->buf       = calloc(queueSize, sizeof(char*));

  out->action = FdActionNewRaw((void*) out, FdOCmCProcess, "CmOutput"); 

  CfgMsgAddInfo("Add cm output to %s tag:%s maxTry:%d queueSize:%d "
                "checkSumFlag:%d",out->name, out->tag, out->maxTry,
                out->queueSize, out->compChksum);
  return(out->action);
}
/*----------------------------------------------------------------------------*/
int FdOCmCNew(void *arg, CfgDataType_t *data )
/*----------------------------------------------------------------------------*/
{
  char *name, *tag;
  int maxTry, queueSize, compChksum;

  /*---check that we are not in a thread (Cm calls should be from the main)---*/
  FdIO* fdIO = FdIOGetRoot();
  FdAction *a = fdIO->actionsOut;
  for(;a != NULL; a = a->next) {
    if(strcmp((a)->type,"Thread") != 0) continue;
    CfgMsgAddFatal(
        "The FDOUT_CM key must not be after a FDOUT_THREAD key");}
  
  /*-------------------------------------------------then create the object---*/
  name       = CfgParseGetNextString( data );
  tag        = CfgParseGetNextString( data );
  queueSize  = CfgParseGetNextDec( data );
  maxTry     = CfgParseGetNextDec( data );
  compChksum = CfgParseGetNextDec( data );

  a = FdOCmCAdd(name, tag, queueSize, maxTry, compChksum);

  /*---------------------------put the action at the end of the linked list---*/
  FdAction **r = &(fdIO->actionsOut);
  while(*r != NULL) r = &((*r)->next);
  *r = a;

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
void FdOCmCProcess(FdOCmC *out, FrameH *frame)
/*----------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL &&
     out->action->toBeRemoved == FR_NO) FdOCmCProcessOne(out, frame);

  /*---------------------------------------------call the next ouput action---*/
  next = out->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*----------------------------------------------------------------------------*/
void FdOCmCProcessOne(FdOCmC *out, FrameH *frame)
/*----------------------------------------------------------------------------*/
/* This function send or post a frame to a consumer                           */
/*----------------------------------------------------------------------------*/
{
  long iQueue, status, nBytes, nQueue, i, irc;
  struct timeval tS, tE;
  double outputTime;

  gettimeofday(&tS, NULL);

  out->action->state = CfgServerGolden;

  /*---------------remove consumers if too many unsuccesful sending attempt---*/
  if((out->nTry > out->maxTry) && out->maxTry >= 0) {
    CfgMsgAddWarning("Remove %s from the Cm destinations after %d send errors", 
		     out->name, out->nTry);
    out->action->toBeRemoved = FR_YES;
    return;}

  /*-----------------------check the status of the already posted messages---*/
  for(i=0; i<out->queueSize; i++) {
    if(out->messages[i] == NULL) continue;

    irc = CmMessageIteratorFinished(out->iterators[i]);
    if(irc != 0) {
      if(out->nTry > 0) {
	CfgMsgAddInfo("could send again frame to %s (%d)",
		      out->name, frame->GTimeS);
  	out->nTry = 0;}

      CmMessageIteratorDelete (out->iterators[i]);
      CmMessageDelete(out->messages[i]);
      free(out->buf[i]);
      out->messages[i] = NULL;
      out->iterators[i] = NULL;
      continue;}
    }  

  /*-------------------------find a free slot for the message and create it---*/
  nQueue = 0;
  for(iQueue = 0; iQueue< out->queueSize; iQueue++) {
    if(out->messages[iQueue] != NULL) nQueue++;}

  if(out->flushing == FR_YES) {
    if(nQueue == 0) {
      CfgMsgAddInfo("output queue to %s has been flushed; Sending frame %d",
		    out->name, frame->GTimeS);
      out->flushing = FR_NO;}
    else {
      snprintf(out->action->userInfo, out->action->userInfoSize,
	      "(flushing output queue to %s queue=%ld)",out->name, nQueue);
      return;}}

  if(nQueue == out->queueSize && out->queueSize > 0) {
    CfgMsgAddError("frame %d not send to %s: sending queue is full",
		   frame->GTimeS, out->name);
    snprintf(out->action->userInfo, out->action->userInfoSize,
	    "(flushing output queue to %s queue=%ld)",out->name, nQueue);
    out->flushing = FR_YES;
    return;}
 
  /*--------------------------------------------------------get a free slot---*/
  for(iQueue = 0; iQueue< out->queueSize; iQueue++) {
    if(out->messages[iQueue] == NULL) break;}

  /*---------------------------------------------create the output message---*/
  out->messages[iQueue] = CmMessageNew();
  if(out->messages[iQueue] == NULL) CfgMsgAddFatal("malloc CmMessage failed"); 

  /*---------------------------------------------------put frame in message---*/
  if(strcmp(out->tag,"*") != 0) FrameTag(frame, out->tag);
  nBytes = FdOCmCPutFrameInMessage (out, iQueue, frame, -1);
  FrameUntag(frame);

  /*---------------------------------------regular send if no queue is used---*/
  if(out->queueSize <= 0) {
    status = CmMessageSend(out->messages[0],out->name);
    CmMessageDelete(out->messages[0]);
    out->messages[0] = NULL;
    free(out->buf[0]);

    gettimeofday(&tE, NULL);
    outputTime = tE.tv_sec -tS.tv_sec + 1.e-6 * (tE.tv_usec - tS.tv_usec);

    if(status == 0) {
      out->action->state = CfgServerActive;
      if(out->nTry == 0) CfgMsgAddError("%d: could not send frame to %s",
		       			frame->GTimeS, out->name);
      out->nTry++;
      snprintf(out->action->userInfo, out->action->userInfoSize,
 	"(%.1f kB to %s FAILED)", nBytes/1024., out->name);}
    else {
      if(out->nTry > 0) {
	CfgMsgAddInfo("%d: could send again frame to %s",
		      frame->GTimeS, out->name);
	out->nTry = 0;}
      snprintf(out->action->userInfo, out->action->userInfoSize,
 	"(%.1f kB to %s outTime: %.4f)", nBytes/1024., out->name, outputTime);}

    return;}

  /*----------------------------------send message using the post mechanism---*/
  CmMessageIteratorDelete(out->iterators[iQueue]);
  out->iterators[iQueue] = CmMessagePost(out->messages[iQueue], out->name, 0);

  if (out->iterators[iQueue] == NULL) {
    out->action->state = CfgServerActive;
    CmMessageDelete(out->messages[iQueue]);
    out->messages[iQueue] = NULL;
    free(out->buf[iQueue]);
    out->buf[iQueue] = NULL;
    if(out->nTry == 0) CfgMsgAddError("could not post frame %d to %s", 
				frame->GTimeS, out->name);
    out->nTry++;

    snprintf(out->action->userInfo, out->action->userInfoSize,
 	      "(%.1f kB to %s FAILED)", nBytes/1024., out->name);
    return;}

  /*-----------------------------------------------print info if using post---*/
  gettimeofday(&tE, NULL);
  outputTime = tE.tv_sec -tS.tv_sec + 1.e-6 * (tE.tv_usec - tS.tv_usec);
  snprintf(out->action->userInfo, out->action->userInfoSize,
	"(%.1f kB to %s outTime: %.4f qSize:%ld/%d)",
	nBytes/1024., out->name, outputTime, nQueue, out->queueSize);

  return;
}
/*----------------------------------------------------------------------------*/
int FdOCmCPutFrameInMessage (FdOCmC *out, int i, FrameH *frame, int compress)
/*----------------------------------------------------------------------------*/
{
  int nBytes, bufSize;

  /*----------guess the buffer size taking a little more than the last time---*/
  bufSize = 1.5*out->lastSize;

  /*--------------------------------Allocate the buffer and write the frame---*/
  while(1) {
    out->buf[i] = malloc(bufSize);
    if(out->buf[i] == NULL) return(0);

    nBytes = FrameWriteToBuf(frame, compress, out->buf[i], bufSize, out->compChksum);

    if (nBytes != 0) break;

    /*-----the buffer is too small, increases its size until the frame fits---*/
    bufSize = 2 * bufSize;
    free(out->buf[i]);
  }

  out->lastSize = nBytes;

  /*------------------------------------------------- build the Cm message ---*/
  CmMessagePutText     (out->messages[i], frame->name);
  CmMessagePutInt      (out->messages[i], frame->run);
  CmMessagePutInt      (out->messages[i], frame->frame);
  CmMessagePutInt      (out->messages[i], frame->GTimeS);
  CmMessagePutInt      (out->messages[i], frame->GTimeN);
  CmMessagePutInt      (out->messages[i], 0);
  CmMessagePutInt      (out->messages[i], nBytes);
  CmMessagePutExtArray (out->messages[i], CmMessageChar, nBytes, out->buf[i]);
  CmMessagePutDouble   (out->messages[i], frame->dt);

  CmMessageSetType     (out->messages[i], "FdFrame");

  return(nBytes);
}
/*----------------------------------------------------------------------------*/
FdAction* FdOCmFindAction (char *name)
/*----------------------------------------------------------------------------*/
{
  FdAction *action;
  FdOCmC *out;
  FdIO *fdIO;

  fdIO = FdIOGetRoot();
  for(action = fdIO->actionsOut; action != NULL; action = action->next) {
    if(strcmp(action->type, "CmOutput") != 0) continue;
    out = action->data;
    if (strcmp(out->name, name) == 0) break;}

  return(action);
}
/*----------------------------------------------------------------------------*/
void FdOCmCleanup ()
/*----------------------------------------------------------------------------*/
{
  FdAction *action, **r;
  FdOCmC *out;
  FdIO *fdIO;
  int i;

  fdIO = FdIOGetRoot();
  for(r = &(fdIO->actionsOut); *r != NULL;) {
    action = *r;
    if((strcmp(action->type, "CmOutput") != 0) ||
       (action->toBeRemoved != FR_YES)) {
      r  = &(action->next);
      continue;}

    /*--------------------------------remove this Cm output from the list---*/
    out = (FdOCmC*) action->data; 
    CfgMsgAddInfo ("FdOCmCleanup: remove %s",out->name);

    /*--------------------------------------------stop all pending messages---*/
    CmConnectKill(CmConnectGetReference(out->name));

    /*-----------------------------------------cleanup the list of messages---*/
    for(i=0; i<out->queueSize; i++) {
      if(out->messages[i] == NULL) continue;
      CmMessageDelete(out->messages[i]);
      CmMessageIteratorDelete(out->iterators[i]);
      free(out->buf[i]);}

    free(out->messages);
    free(out->iterators);
    free(out->buf);
    free(out->name);
    free(out->tag);

    free(out);

    *r = action->next;
    FdActionRemove(action);
    }

  return;
}
/*----------------------------------------------------------------------------*/
int FdOCmGetNCsmr ()
/*----------------------------------------------------------------------------*/
{
  FdAction *action;
  FdIO *fdIO;
  int nCsmr;
 
  nCsmr = 0;
  fdIO = FdIOGetRoot();

  for(action = fdIO->actionsOut; action != NULL; action = action->next) {
    if(strcmp(action->type, "CmOutput") == 0) nCsmr++;}

  return(nCsmr);
}
/*----------------------------------------------------------------------------*/
CmMessageStatus FdOCmHandlerAdd (CmMessage message, char* from, char* server)
/*----------------------------------------------------------------------------*/
{
  char *name, *tag;
  int maxTry, queueSize, compChksum;
  FdAction *a;

  name      = CmMessageGetText(message);
  tag       = CmMessageGetText(message);
  queueSize = CmMessageGetInt(message);
  maxTry    = CmMessageGetInt(message);
  compChksum= CmMessageGetInt(message);

  CfgMsgAddInfo ("FdOCmHandlerAdd: %s asked to send frames to %s tag=%s"
                 " qSize=%d maxTry=%d", from, name, tag, queueSize, maxTry);

  /*----------------------------- there is always a FdOCmSrvr server object---*/
  if(FdOCmSrvr == NULL) CfgMsgAddFatal(" FdOCmSrvr does not exist");

  /*----------------------check parameter against common setting if present---*/
  int nConsumers = FdOCmGetNCsmr();
  if(nConsumers >= FdOCmSrvr->maxConsumer) {
    CfgMsgAddError(" Could not add it; mximum number of consumers (%d) reached",
  		        FdOCmSrvr->maxConsumer);
    return(CmMessageOk);}

  if(FdOCmSrvr->queueSize >= 0) {
    queueSize = FdOCmSrvr->queueSize;
    CfgMsgAddInfo(" queue size set to the preset value:%d", queueSize);}
   
  a = FdOCmCAdd (name, tag, queueSize, maxTry, compChksum);
  if(a == NULL) return(CmMessageOk);

  /*------------------------- add the action just after the FdOCmSrvr action--*/
  a->next = FdOCmSrvr->action->next;
  FdOCmSrvr->action->next = a;

  return(CmMessageOk);
}
/*----------------------------------------------------------------------------*/
CmMessageStatus FdOCmHandlerCList (CmMessage message, char* from, char* server)
/*----------------------------------------------------------------------------*/
{
  FdOCmCLRequest *r;

  CfgMsgAddInfo("%s asked for the list of channels",from);

  /*---------------------if the list of channel is available, send it back---*/
  if(FdOCmSrvr->chListTimeSpam > 0) {
    FdOCmSendCList(from);
    return(CmMessageOk);}

  /*---------------check if a request is already pending for this receiver---*/
  for(r = FdOCmSrvr->requests; r != NULL; r = r->next) {
    if(strcmp(r->name, from) == 0) break;}
  if(r != NULL) return(CmMessageOk);

  /*---the list is not ready, fill the request to send it at the next frame---*/
  r = (FdOCmCLRequest*) calloc(1,sizeof(FdOCmCLRequest));
  if(r == NULL) CfgMsgAddFatal("FdOCmHanlerCList malloc failed ");
  r->next = FdOCmSrvr->requests;
  FdOCmSrvr->requests = r;

  FrStrCpy(&r->name, from);
  if(r->name == NULL) CfgMsgAddFatal("FdOCmHandlerCList malloc name");

  return(CmMessageOk);
}
/*---------------------------------------------------------------------------*/
void FdOCmDisconnectionHandler (CmConnect connect)
/*---------------------------------------------------------------------------*/
{
  char *name;
  FdAction *action;
  FdOCmC *out;
 
  name = CmConnectGetName(connect);
  if(name == NULL) {
    CfgMsgAddError("FdOCmDisconnectionHandler with no name");
    return;}

  action = FdOCmFindAction(name);
  if(action == NULL) return;
  
  CfgMsgAddInfo ("FdOCmDisconnectionHandler: remove %s",name);
  if( (out = (FdOCmC*)action->data) && out->maxTry >= 0 ) {
    action->toBeRemoved = FR_YES; }

  return;
}
/*---------------------------------------------------------------------------*/
CmMessageStatus FdOCmHandlerRemove (CmMessage message, 
				    char* from, char* server)
/*---------------------------------------------------------------------------*/
{
  char *name;
  FdAction *action;
 
  name = CmMessageGetText(message);
  CfgMsgAddInfo ("FdOCmHandlerRemove> %s asked to remove %s from the cm output",
		 from, name);

  action = FdOCmFindAction(name);
  if(action == NULL) {
    CfgMsgAddWarning(" This consumer was not in the list");
    return(CmMessageOk);}

  action->toBeRemoved = FR_YES;

  return(CmMessageOk);
}
/*----------------------------------------------------------------------------*/
void FdOCmServerAdd(int maxConsumer, int chListTimeSpam, int queueSize)
/*----------------------------------------------------------------------------*/
{
  if(FdOCmSrvr != NULL) return;

  /*---check that we are not in a thread (Cm calls should be from the main)---*/
  FdIO* fdIO = FdIOGetRoot();
  FdAction *a = fdIO->actionsOut;
  for(;a != NULL; a = a->next) {
    if(strcmp((a)->type,"Thread") != 0) continue;
    CfgMsgAddFatal(
        "The FDOUT_CM_SERVER key must not be after a FDOUT_THREAD key");}
 
  /*---------------------------------------------------------create object---*/ 
  CfgMsgAddInfo("FdOCm parameters: maxConsumer%d chListTimeSpam%d queueSize=%d",
		maxConsumer, chListTimeSpam, queueSize);

  FdOCmSrvr = (FdOCmServer*) calloc(1,sizeof(FdOCmServer));
  if(FdOCmSrvr == NULL) CfgMsgAddFatal("FdOCmSrvr malloc failed");

  FdOCmSrvr->maxConsumer    = maxConsumer;  
  FdOCmSrvr->chListTimeSpam = chListTimeSpam;
  FdOCmSrvr->queueSize      = queueSize,

  FdOCmSrvr->listAdc  = FdCListNew();
  FdOCmSrvr->listProc = FdCListNew();
  FdOCmSrvr->listSer  = FdCListNew();
  FdOCmSrvr->listSim  = FdCListNew();

  /*-------------create the action and put it at the end of the linked list---*/
  FdOCmSrvr->action = FdActionNewRaw((void*) FdOCmSrvr, FdOCmServerProcess, "OCmServeur");

  FdAction **r = &(fdIO->actionsOut);
  while(*r != NULL) r = &((*r)->next);
  *r = FdOCmSrvr->action;

  /*--------------------------------------------install associated handlers---*/
  CmMessageInstallHandler (FdOCmHandlerAdd,    "FdAddCmOutput"); 
  CmMessageInstallHandler (FdOCmHandlerRemove, "FdRemoveCmOutput");
  CmMessageInstallHandler (FdOCmHandlerCList,  "FdGetChannelsList");
  CmMessageInstallHandler (FdFrMrgrHandlerAdd, "FdAddFrameMergerSource");
  CmMessageInstallHandler (FdFrMrgrHandlerRm,  "FdRemoveFrameMergerSource");

  CmConnectInstallDisconnectionHandler(FdOCmDisconnectionHandler);

  /*-----------the next lines are for backward compatibility---*/
  CmMessageInstallHandler (FdOCmHandlerAdd,    "FdAddFrame"); 
  CmMessageInstallHandler (FdOCmHandlerRemove, "FdRemoveFrame");

  return;
}
/*----------------------------------------------------------------------------*/
int FdOCmServerNew(void *arg, CfgDataType_t *data )
/*----------------------------------------------------------------------------*/
{
  int maxConsumer, chListTimeSpam, queueSize;

  if(FdOCmSrvr != NULL) 
    CfgMsgAddFatal("Duplicated FDOUT_CM_SERVER key; please keep only one");

  maxConsumer    = CfgParseGetNextDec(data);  
  chListTimeSpam = CfgParseGetNextDec(data);
  queueSize      = CfgParseGetNextDec(data);

  FdOCmServerAdd(maxConsumer, chListTimeSpam, queueSize);

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
void FdOCmServerProcessOne(FdOCmServer *oCm, FrameH *frame)
/*----------------------------------------------------------------------------*/
{
  unsigned int gpsMin, nChannels, nRemoved;
  FdOCmCLRequest *r, *nextCLR;
  char *p, *pMax;
 
  FdOCmSrvr->action->userInfo[0] = '\0';
  pMax = FdOCmSrvr->action->userInfo + FdOCmSrvr->action->userInfoSize;

  /*-------------------------------update the list of channels if requested---*/
  if (oCm->chListTimeSpam > 0) {
    gpsMin = frame->GTimeS - oCm->chListTimeSpam;
    nRemoved  = FdCListBuildAdc (FdOCmSrvr->listAdc,  frame, gpsMin);
    nRemoved += FdCListBuildProc(FdOCmSrvr->listProc, frame, gpsMin);
    nRemoved += FdCListBuildSer (FdOCmSrvr->listSer,  frame, gpsMin);
    nRemoved += FdCListBuildSim (FdOCmSrvr->listSim,  frame, gpsMin);

    /*------------------------if there are duplicated channel add user info---*/
    p = FdOCmSrvr->action->userInfo;
    FdOCmSrvr->action->state = CfgServerGolden;

    nChannels = FdOCmSrvr->listAdc->nChnl + FdOCmSrvr->listProc->nChnl +
                FdOCmSrvr->listSer->nChnl + FdOCmSrvr->listSim->nChnl;
    p += snprintf(p, pMax-p, " nChannels=%d; ",nChannels);

    if(FdOCmSrvr->listAdc->nDuplicated > 0) {
      p += snprintf(p, pMax-p, "%d duplicated ADC, first:%s;",
		   FdOCmSrvr->listAdc->nDuplicated, 
		   FdOCmSrvr->listAdc->firstDup);
      FdOCmSrvr->action->state = CfgServerActive;
      FdOCmSrvr->listAdc->nDuplicated = 0;}

    if(FdOCmSrvr->listProc->nDuplicated > 0) {
      p += snprintf(p, pMax-p, "%d duplicated Proc, first:%s;",
		   FdOCmSrvr->listProc->nDuplicated, 
		   FdOCmSrvr->listProc->firstDup);
      FdOCmSrvr->action->state = CfgServerActive;
      FdOCmSrvr->listProc->nDuplicated = 0;}

    if(FdOCmSrvr->listSer->nDuplicated > 0) { 
      p += snprintf(p, pMax-p, "%d duplicated Ser, first:%s;",
		   FdOCmSrvr->listSer->nDuplicated, 
		   FdOCmSrvr->listSer->firstDup);
      FdOCmSrvr->action->state = CfgServerActive;
      FdOCmSrvr->listSer->nDuplicated = 0;}

    if(FdOCmSrvr->listSim->nDuplicated > 0) {
      p += snprintf(p, pMax-p, "%d duplicated Sim, first:%s;",
		   FdOCmSrvr->listSim->nDuplicated, 
		   FdOCmSrvr->listSim->firstDup);
      FdOCmSrvr->action->state = CfgServerActive;
      FdOCmSrvr->listSim->nDuplicated = 0;}

    /*-----------------------------print some statistic in case of change---*/
    if((nRemoved != 0) && (FdOCmSrvr->nRemoved ==0)) {
      CfgMsgAddInfo("FdOCmServerProcess: %d Start to have duplicated channels "
                    " nRemoved=%d %s",
                     frame->GTimeS, nRemoved, FdOCmSrvr->action->userInfo);}
    if((nRemoved == 0) && (FdOCmSrvr->nRemoved !=0)) {
      CfgMsgAddInfo("FdOCmServerProcess: %d no more duplicated channels",
                     frame->GTimeS);}
    FdOCmSrvr->nRemoved = nRemoved;

    snprintf(FdOCmSrvr->action->serData, FdOCmSrvr->action->serDataSize, 
	"listSize %d nRemovedList %d", nChannels, nRemoved);
  }
  /*-----------------------check if there are pending channel list requests---*/
  else {
    for(r = FdOCmSrvr->requests; r != NULL; r = nextCLR) {

      CfgMsgAddInfo("Build and send channel list to %s",r->name);

      gpsMin = frame->GTimeS - oCm->chListTimeSpam;
      FdCListBuildAdc (FdOCmSrvr->listAdc,  frame, gpsMin);
      FdCListBuildProc(FdOCmSrvr->listProc, frame, gpsMin);
      FdCListBuildSer (FdOCmSrvr->listSer,  frame, gpsMin);
      FdCListBuildSim (FdOCmSrvr->listSim,  frame, gpsMin);

      FdOCmSendCList(r->name);

      nextCLR = r->next;
      free(r->name);
      free(r);}
    FdOCmSrvr->requests = NULL;}

  return;
}
/*----------------------------------------------------------------------------*/
void FdOCmServerProcess(FdOCmServer *oCm, FrameH *frame)
/*----------------------------------------------------------------------------*/
{
  FdAction *nextAction;

  if(frame != NULL) FdOCmServerProcessOne(oCm, frame); 

  /*---------------------------------------------call the next ouput action---*/
  nextAction = oCm->action->next;
  if(nextAction != NULL) nextAction->action(nextAction->data, frame);

  return;
}
/*----------------------------------------------------------------------------*/
void FdCmParserAdd(FdIO* fdIO)
/*----------------------------------------------------------------------------*/
{
  /*-----------------------------------------------------------FDIN actions---*/
  CfgParseGetFunctionAdd(fdIO->parser,"FDIN_CM",
			 FdICmWaitNew, (void *) &(fdIO->actionsIn),
			 2, CfgDec,  CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser,"FDIN_CM_CONNECT",
			 FdICmConnectNew, (void *) &(fdIO->actionsIn),
			 3, CfgString, CfgDec, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser,"FDIN_CM_THREAD",
			 FdICmThreadNew, (void *) &(fdIO->actionsIn), 
			 1, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser,"FDIN_FRAME_MERGER",
			 FdIFrMrgrNew, (void *) &(fdIO->actionsIn),
			 2, CfgDec,  CfgString);
  CfgParseGetFunctionAdd(fdIO->parser,"FDIN_GET_FIRST_FRAME",
			 FdIGetFirstFrNew, (void *) &(fdIO->actionsIn),
			 2, CfgDec,  CfgString);
  CfgParseGetFunctionAdd(fdIO->parser,"FDIN_DATA_SENDER",
			 FdIDataSenderNew, (void *) &(fdIO->actionsIn),
			 3, CfgString, CfgDec,  CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser,"FDIN_SLEEP",
			 FdSleepNew, (void *) &(fdIO->actionsIn),
			 1, CfgDec);

  /*---------------------------------------------------------FDOUT actions---*/
  CfgParseGetFunctionAdd(fdIO->parser,"FDOUT_CM_SERVER",
			 FdOCmServerNew, (void *) &(fdIO->actionsOut),
			 3, CfgDec,  CfgDec, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_CM", 
			 FdOCmCNew, (void *) &(fdIO->actionsOut),
			 5, CfgString, CfgString, CfgDec, CfgDec, CfgDec);

  return;
}
/*----------------------------------------------------------------------------*/
void FdOCmSendCList (char* from)
/*----------------------------------------------------------------------------*/
{
  char *listAdc, *listProc, *listSer, *listSim;
  CmMessage message;
  int irc; 
  
  CfgMsgAddInfo ("Send lists of channels (nChnl=%d %d %d %d) to %s",
                 FdOCmSrvr->listAdc->nChnl, FdOCmSrvr->listProc->nChnl,
                 FdOCmSrvr->listSer->nChnl, FdOCmSrvr->listSer->nChnl, from);

  message = CmMessageNew();

  listAdc  = FdCListDump(FdOCmSrvr->listAdc);
  listProc = FdCListDump(FdOCmSrvr->listProc);
  listSer  = FdCListDump(FdOCmSrvr->listSer);
  listSim  = FdCListDump(FdOCmSrvr->listSim);

  CmMessagePutInt (message, FdOCmSrvr->listAdc->gps_ms/1000);
  CmMessagePutText(message, "ADC");
  CmMessagePutInt (message, FdOCmSrvr->listAdc->nChnl);
  CmMessagePutText(message, listAdc);
  CmMessagePutText(message, "SER");
  CmMessagePutInt (message, FdOCmSrvr->listProc->nChnl);
  CmMessagePutText(message, listProc);
  CmMessagePutText(message, "PROC");
  CmMessagePutInt (message, FdOCmSrvr->listSer->nChnl);
  CmMessagePutText(message, listSer);
  CmMessagePutText(message, "SIM");
  CmMessagePutInt (message, FdOCmSrvr->listSim->nChnl);
  CmMessagePutText(message, listSim);

  free(listAdc);
  free(listProc);
  free(listSer);
  free(listSim);
    
  CmMessageSetType (message, "FdChannelsList");
  irc = CmMessageSend (message, from);
  if(irc == 0)
    CfgMsgAddError("could not send the list of channels to %s",from);

  CmMessageDelete (message);

  return;
}
