/*----------------------------------------------------------------------*/
#include <stdlib.h>
#include "FrameL.h"
#include "Fd.h"
#include "CfgVirgo.h"
#include "FdChList.h"

int FdCListCmp (const void * a, const void * b);
int FdCListSort(FdCList* list, int gpsLimit);

/*---------------------------------------------------------------------------*/
void FdCListAdd(FdCList* list, char* name, double sampleRate, long gps_ms)
/*---------------------------------------------------------------------------*/
/* This function add an element to the list.
   The gps is in millisecond to support sub-second frames.
   It first check if the name is already in the list. 
   If not it add it at the end in the unsorted part of the list.             */
/*---------------------------------------------------------------------------*/
{
  double *sampleRateList;
  char *base, *nameL;
  long **gL, len, *baseI, *gpsL;
  
  list->gps_ms = gps_ms;

  if(list->nSorted > 0) {
    base = name-sizeof(double)-sizeof(long);
    baseI = (long*) &base;
    gL = bsearch(baseI, list->chnl, list->nSorted,sizeof(int *),FdCListCmp);}
  else {
    gL = NULL;}

  /*-------------update parameters if the channel is already in the list---*/
  if(gL != NULL) {

  /*------ check if it is a duplicated channel---*/
    if(**gL ==  gps_ms) {
      list->nDuplicated++;
      if(list->nDuplicated == 1) sprintf(list->firstDup, name);}
    else {
      **gL =  gps_ms;}

    sampleRateList = (double*)(*gL + 1);
    if(*sampleRateList != sampleRate) {
      CfgMsgAddWarning("%ld: sampling rate for %s is changed from %g to %g",
		       gps_ms/1000,name,*sampleRateList, sampleRate);
      *sampleRateList = sampleRate;}
    return;}

  /*---------------------------add the channel at the end of the list---*/
  len = strlen(name)+sizeof(long)+sizeof(double)+2;
  if((list->chnl[list->nChnl] = malloc(len)) == NULL)
    CfgMsgAddFatal("FdCListBuild malloc failed at %ld, len=%ld"
				,gps_ms/1000, len);
  gpsL = (long*) list->chnl[list->nChnl];
  *gpsL = gps_ms/1000;
  sampleRateList = (double*) (gpsL + 1);
  *sampleRateList = sampleRate;
  nameL = (char*)list->chnl[list->nChnl] + sizeof(long) + sizeof(double);
  sprintf(nameL, name);
  list->nChnl++;

  /*------------------------if we reach the limit of the list, extend it---*/
  if(list->nChnl < list->maxChnl) return;

  list->maxChnl *= 2;
  list->chnl = realloc(list->chnl, list->maxChnl * sizeof(char *));
  if(list->chnl == NULL) 
    CfgMsgAddFatal("FdCListBuild malloc failed at %ld",gps_ms/1000);

  return;
}
/*---------------------------------------------------------------------------*/
int FdCListBuildAdc(FdCList* list, FrameH *frame, int gpsLimit)
/*---------------------------------------------------------------------------*/
/* This function update the sorted list of ADC channels                      */
/*---------------------------------------------------------------------------*/
{
  FrAdcData *adc;
  int nRemoved;

  if(frame->rawData != NULL) {
    for (adc = frame->rawData->firstAdc; adc != NULL; adc = adc->next) {
        long gps_ms = (long) frame->GTimeS * 1000 + frame->GTimeN/1e6;
	FdCListAdd(list, adc->name, adc->sampleRate, gps_ms);}}

  nRemoved = FdCListSort(list, gpsLimit);

  return(nRemoved);
}
/*---------------------------------------------------------------------------*/
int FdCListBuildSer(FdCList* list, FrameH *frame, int gpsLimit)
/*---------------------------------------------------------------------------*/
/* This function update the sorted list of Ser channels                      */
/*---------------------------------------------------------------------------*/
{
  FrSerData *ser;
  char *data, word1[512], word2[512], word3[512], *name;
  int nRemoved, nRead, len;
 
  if (frame->rawData != NULL) {

    for (ser = frame->rawData->firstSer; ser != NULL; ser = ser->next) {

      data = ser->data;
      while ((nRead = sscanf(data,"%s %s %s",word1, word2, word3)) >= 2) {

        if(strncmp(word1,"units",5) != 0) {
	  len = strlen(ser->name)+strlen(word1)+30;
  	  if((name = malloc(len)) == NULL)
	    CfgMsgAddFatal("FdCListBuild malloc failed at %d",frame->GTimeS);
	  sprintf(name,"%s_%s", ser->name, word1);
          long gps_ms = (long) frame->GTimeS * 1000 + frame->GTimeN/1e6;
 	  FdCListAdd(list, name, ser->sampleRate, gps_ms);
          free(name);}

        if (nRead != 3) break;
        data = strstr(data+strlen(word1)+strlen(word2), word3);}}}

  nRemoved = FdCListSort(list, gpsLimit);

  return(nRemoved);
}
/*---------------------------------------------------------------------------*/
int FdCListBuildSim(FdCList* list, FrameH *frame, int gpsLimit)
/*---------------------------------------------------------------------------*/
/* This function update the sorted list of Sim channels                      */
/*---------------------------------------------------------------------------*/
{
  FrSimData *sim;
  int nRemoved;

  for (sim = frame->simData; sim != NULL; sim = sim->next) {
    long gps_ms = (long) frame->GTimeS * 1000 + frame->GTimeN/1e6;
    FdCListAdd(list, sim->name, sim->sampleRate, gps_ms);}

  nRemoved = FdCListSort(list, gpsLimit);

  return(nRemoved);
}

/*---------------------------------------------------------------------------*/
int FdCListBuildProc(FdCList* list, FrameH *frame, int gpsLimit)
/*---------------------------------------------------------------------------*/
/* This function build a sorted list of the PROC channels name, sampling rate.
   The GPS time is added at the end of each channel info string.             */
/*---------------------------------------------------------------------------*/
{
  FrProcData *proc;
  double sampleRate;
  int nRemoved;

  for (proc = frame->procData; proc != NULL; proc = proc->next) {
    sampleRate = 0.;
    if(proc->data != NULL) {
      if(proc->data->dx[0] != 0.) {
	sampleRate = 1./proc->data->dx[0];}}
    long gps_ms = (long) frame->GTimeS * 1000 + frame->GTimeN/1e6;
    FdCListAdd(list, proc->name, sampleRate, gps_ms);}

  nRemoved = FdCListSort(list, gpsLimit);

  return(nRemoved);
}
/*---------------------------------------------------------------------------*/
int FdCListCmp (const void * a, const void * b)
/*---------------------------------------------------------------------------*/
/* this function compares two elements of the list                           */
/*---------------------------------------------------------------------------*/
{
  void **addressA, **addressB;
  char *nameA, *nameB;

  addressA = (void **) a;
  addressB = (void **) b;
  nameA = (char*)*addressA + sizeof(long) + sizeof(double);
  nameB = (char*)*addressB + sizeof(long) + sizeof(double);
 
  int ierr = strcmp (nameA, nameB);

  return (ierr);
}
/*---------------------------------------------------------------------------*/
char* FdCListDump(FdCList* list)
/*---------------------------------------------------------------------------*/
{
  char *buffer, *p, *name;
  int i, size, sizeLD;
  double *sampleRate;

  if(list == NULL) return(NULL);

  size = 20;
  sizeLD = sizeof(long)+sizeof(double);

  /*---------------------------------------------- first compute the size---*/
  for(i=0; i<list->nChnl; i++) {
    size += strlen((char*)list->chnl[i]+sizeLD) + 20;}

  /*------------------------------------------------get the output buffer---*/
  buffer = malloc(size);
  if(buffer == NULL) CfgMsgAddFatal("FdCListDump malloc failed (%d %d)",
				    size, list->nChnl);

  /*---------------------------------------------fill the list of names---*/
  p = buffer;
  for(i=0; i<list->nChnl; i++) {
    name = (char*) list->chnl[i] + sizeLD;
    sampleRate = (double*) (((long*) list->chnl[i]) + 1);
    p += sprintf(p,"%s %g\n", name, *sampleRate);}
  if(list->nChnl == 0) sprintf(buffer," ");

  return(buffer);
}
/*---------------------------------------------------------------------------*/
void FdCListFree(FdCList *list)
/*---------------------------------------------------------------------------*/
{
  int i;

  if(list == NULL) return;

  for(i = 0; i<list->nChnl; i++) {
    if(list->chnl[i] != NULL) free(list->chnl[i]);}

  if(list->chnl != NULL) free(list->chnl);

  free(list);

  return;

}
/*---------------------------------------------------------------------------*/
FdCList* FdCListMerge(FdCList *listADC, 
		      FdCList *listSER,
		      FdCList *listPROC, 
		      FdCList *listSIM)
/*---------------------------------------------------------------------------*/
/* This function produces a single list from the four input lists            */
/*---------------------------------------------------------------------------*/
{
  char *name;
  long i, sizeLD, gps_ms;
  double *sampleRate;
  FdCList *list;

  gps_ms = 0;
  list = FdCListNew();

  sizeLD = sizeof(long)+sizeof(double);

  if(listADC != NULL) {
    gps_ms = listADC->gps_ms;
    for(i=0; i<listADC->nChnl; i++) {
      name = (char*) listADC->chnl[i] + sizeLD;
      sampleRate = (double*) listADC->chnl[i] + 1;
      FdCListAdd(list, name, *sampleRate, gps_ms);}}

  if(listSER != NULL) {
    gps_ms = listSER->gps_ms;
    for(i=0; i<listSER->nChnl; i++) {
      name = (char*) listSER->chnl[i] + sizeLD;
      sampleRate = (double*) listSER->chnl[i] + 1;
      FdCListAdd(list, name, *sampleRate, gps_ms);}}

  if(listPROC != NULL) {
    gps_ms = listPROC->gps_ms;
    for(i=0; i<listPROC->nChnl; i++) {
      name = (char*) listPROC->chnl[i] + sizeLD;
      sampleRate = (double*) listPROC->chnl[i] + 1;
      FdCListAdd(list, name, *sampleRate, gps_ms);}}

  if(listSIM != NULL) {
    gps_ms = listSIM->gps_ms;
    for(i=0; i<listSIM->nChnl; i++) {
      name = (char*) listSIM->chnl[i] + sizeLD;
      sampleRate = (double*) listSIM->chnl[i] + 1;
      FdCListAdd(list, name, *sampleRate, gps_ms);}}

  if(gps_ms > 0) FdCListSort(list, gps_ms);

  return(list);
}
/*---------------------------------------------------------------------------*/
FdCList* FdCListNew()
/*---------------------------------------------------------------------------*/
/* This function creates a new list of channel  */
/*---------------------------------------------------------------------------*/
{
  FdCList *list;

  list = (FdCList *) calloc(1, sizeof(FdCList));
  if(list == NULL) return (NULL);

  list->nChnl = 0;
  list->nSorted = 0; 

  list->maxChnl = 512;
  if((list->chnl = malloc(list->maxChnl * sizeof(char *))) == NULL)
    CfgMsgAddFatal("FdCListNew malloc failed");

  return(list);}

/*---------------------------------------------------------------------------*/
int FdCListSort(FdCList* list, int gpsLimit)
/*---------------------------------------------------------------------------*/
/* This function remove the channels which are too old and sort them         */
/*---------------------------------------------------------------------------*/
{
  long nRemoved, *gpsList, i;
  char *name;

  nRemoved = 0;
  for (i = 0; i<list->nChnl; i++) {
    gpsList = (long*) list->chnl[i];
    if(*gpsList >= gpsLimit) continue;

    /*-the channel is too old, set its name to be at the end of the list---*/
    name = (char*) list->chnl[i] + sizeof(long) + sizeof(double);
    name[0] = 0x7f;
    nRemoved++;}

  /*-------------------------- if there are no changes in the list return---*/
  if(list->nSorted == list->nChnl && nRemoved == 0) return(nRemoved);

  /*----------------------------------- sort the list by alphabetic order---*/
  qsort(list->chnl, list->nChnl, sizeof(char *), FdCListCmp);
  if(nRemoved == 0) {
    list->nSorted = list->nChnl;
    return(nRemoved);}

  /*---------------------------------------------free the end of the list---*/
  for (i = list->nChnl-nRemoved; i<list->nChnl; i++) {
    free(list->chnl[i]);}

  list->nChnl -= nRemoved;
  list->nSorted = list->nChnl;

  return(nRemoved);
}

