/*---------------------------------------------------------------------------*/
/* FdTools.c    Set of function to be moved to the Fr and Frv packages       */
/*---------------------------------------------------------------------------*/

#include "FdTools.h"
#include "math.h"

/*---------------------------------------------------------FrSerDataAddData--*/
void FrSerDataAddData(FrSerData *ser, 
		      char* data) 
/*---------------------------------------------------------------------------*/
{
  char *newData;

  if(ser  == NULL) return;
  if(data == NULL) return;

  newData = malloc(strlen(ser->data)+strlen(data)+2);
  if(newData == NULL) return;

  sprintf(newData,"%s %s",ser->data, data);
  free(ser->data);
  ser->data = newData;

  return;}  

/*-----------------------------------------------------------------FrStrcmp--*/
int FrStrcmp(char *c1, 
               char* c2) 
/*---------------------------------------------------------------------------*/
{
  if(c1 == NULL && c2 == NULL) return(0);
  if(c1 == NULL && c2 != NULL) return(-1);
  if(c1 != NULL && c2 == NULL) return(-1);

  return(strcmp(c1, c2));
}
/*---------------------------------------------------------------------------*/
void FrvFDFilterFreeTFBin(FrvFDFilter *fdf)
/*---------------------------------------------------------------------------*/
{FrvFDFTFBin *tfBin, *next;

 if(fdf == NULL) return;

 for(tfBin = fdf->tfBins; tfBin != NULL; tfBin = next) {
   next = tfBin->next;
   free(tfBin);}

 fdf->tfBins = NULL;

 return;
}
/*---------------------------------------------------------------------------*/
void FrVectFixName(FrVect *vect)
/*---------------------------------------------------------------------------*/
{
  int len, i;

  len = strlen(vect->name);

  for(i=0; i<len; i++) {if(vect->name[i] == ' ') vect->name[i] = '_';}

  return;
}

/*-------------------------------------------------------------FrFrameResize---*/
FrameH* FrFrameEnlarge(FrameH* frame, 
                       FrameH** resizedFrame, 
                       double newFrameLength)
/*-----------------------------------------------------------------------------*/
/* Resize a frame and returns NULL if the resized frame is incomplet           */
/*-----------------------------------------------------------------------------*/
{
  FrameH* output;
  int position, nFrames, resizedEnd;

  if(frame == NULL) return(NULL);

  output   = NULL;
  position = fmod(frame->GTimeS+1.e-9*frame->GTimeN+1.e-6, newFrameLength)/
                                    frame->dt;
  nFrames  = (newFrameLength+1.e-6)/frame->dt;

  /*------- first case: create a new output frame which could be the last... --*/
  if(*resizedFrame == NULL) {
    *resizedFrame = FrameReshapeNew(frame, nFrames, position);
    if(position == nFrames-1) {
      FrameReshapeEnd(*resizedFrame);
      output = *resizedFrame;
      *resizedFrame = NULL;}
    return(output);}
     
  /*-------------- second case, add the frame to the existing resized frame ---*/
  resizedEnd = newFrameLength + (*resizedFrame)->GTimeS;
  if(frame->GTimeS <= resizedEnd) {
    FrameReshapeAdd(*resizedFrame, frame);
    if(position == nFrames-1) {
      FrameReshapeEnd(*resizedFrame);
      output = *resizedFrame;
      *resizedFrame = NULL;}
    return(output);}
   
  /*----------------------- last case, add a frame to the next resized frame --*/
  FrameReshapeEnd(*resizedFrame);
  output = *resizedFrame;
  *resizedFrame = FrameReshapeNew(frame, nFrames, position);

  return(output);}

/*---------------------------------------------------------------------------*/
void FrameFreqCut(FrameH *frame, double fCut)
/*---------------------------------------------------------------------------*/
/* This function keep only the adc and proc channels with a frequency <= fCut*/
/*---------------------------------------------------------------------------*/
{
  FrAdcData *adc,   *nextAdc;
  FrProcData *proc, *nextProc;
  double dt;

  if(frame == NULL) return; 
  FrameUntag(frame);
  dt = 1./fCut;

  /*------------------------------------------------------------- procData---*/
  proc = frame->procData;
  frame->procData = NULL; 
  for(; proc != NULL; proc = nextProc) {
    nextProc = proc->next;
    if(proc->type != 1 || proc->data->dx[0] < dt) {
      proc->next = NULL;
      FrProcDataFree(proc);}
    else {
      proc->next = frame->procData;
      frame->procData = proc;}}

  /*--------------------------------------------------------------- ADC ---*/
  if(frame->rawData == NULL) return;
 
  adc = frame->rawData->firstAdc;
  frame->rawData->firstAdc = NULL;
  for(; adc != NULL; adc = nextAdc) {
    nextAdc = adc->next;
    if(adc->sampleRate > fCut || adc->data->nDim > 1) {
      FrAdcDataFreeOne(adc);}
    else {
      adc->next =  frame->rawData->firstAdc;
      frame->rawData->firstAdc = adc;}}
 
  return;
}
/*----------------------------------------------------- FrRandGNew -----*/
FrRandomG* FrRandGNew(long seed)
/*----------------------------------------------------------------------*/
/* Create a new structure for gaussian random generator                 */
/*----------------------------------------------------------------------*/
{
  FrRandomG* ran;

  ran = (FrRandomG*) calloc(1,sizeof(FrRandomG));
  if(ran == NULL) return NULL;

  if(seed == 0) ran->seed = 2147483646;
  else          ran->seed = seed;

  return(ran);
}

/*------------------------------------------------------- FrRandG-----*/
/* This function generates a pseudo random number (double) with normal*/
/* (Gaussian) distribution with zero mean and unit variance.          */
/* Function derived from the SIESTA code */
/*--------------------------------------------------------------------*/
double FrRandG(FrRandomG* ran)
{
  static FrRandomG* random = NULL;
  double fac,r,x,y;
  long a = 16807;
  long m = 2147483647;
  long q = 127773;
  long p = 2836;
  long hi, lo;

  /*------------ this is to allow the function to work in any case---*/
  if (ran == NULL) {
    if(random == NULL) {
      random = FrRandGNew(0);
      if(random == NULL) return(0);}
    ran = random;}

  /*--- since the function generate two values, return one if available--*/
  if (ran->iset == 1 ){
    ran->iset = 0;
    return(ran->gset);}
 
  /*--- generate x and y uniformly distributed between -1 and 1 with
    the constrain that they are within a cercle of radiuis unity ---*/
  ran->iset = 1;
  do {
    hi   = ran->seed / q;
    lo   = ran->seed - hi * q;
    ran->seed = a*lo - p*hi;
    if (ran->seed <= 0) ran->seed = ran->seed + m;
    x    = -1. + 2. * (double)ran->seed / (double)m;
    hi   = ran->seed / q;
    lo   = ran->seed - hi * q;
    ran->seed = a*lo - p*hi;
    if (ran->seed <= 0) ran->seed = ran->seed + m;
    y    = -1. + 2. * (double)ran->seed / (double)m;
    r    = x*x + y*y;
  }
  while (r >= 1.0);
 
  /*--- invert the random distribution---*/

  if (r == 0.0)  r = 0.000001;
  fac  = sqrt(-2.*log(r)/r);
  ran->gset = x * fac;

  return(y*fac);
}

/*---------------------------------------------------------------------------*/
double FrRand1(long *seed)
/*---------------------------------------------------------------------------*/
/* This function generate a random number between 0 and 1 (Lehmer generator) */
/* See Stephen K. Park and Keith W Miller in                                 */
/* Comunications of the ACM October 1988 Volume 31  Number 10)               */
/*---------------------------------------------------------------------------*/
{
  long a = 16807;
  long m = 2147483647;
  long q = 127773;
  long p = 2836;
  static long localSeed = 2147483646;
  long hi, lo;
 
  if(seed == NULL) seed = &localSeed; 
 
  /*-----------------------------------------seed should be in [1, 2^31 - 2]*/
  if(*seed < 1 || *seed > 2147483646) *seed = 2147483646;

  hi   = *seed / q;
  lo   = *seed - hi * q;
  *seed = a*lo - p*hi;
  if (*seed <= 0) *seed = *seed + m;

  return (double) ((double)*seed / (double)m);
}
/*--------------------------------------------------- FrUMrand2 ------*/
/*  Taken from the UMrand2 SIESTA function:                           */
/*  Adapted from TRandom2 of ROOT by Julien Ramonet (LAPP Annecy)     */
/*  Minor Changes by A.Vicere' (Urbino)                               */
/*  Machine independent random number generator.                      */
/*  Produces uniformly-distributed floating points between 0 and 1.   */
/*  Identical sequence on all machines of >= 32 bits.                 */
/*  Periodicity 10**14. Note: TRandom3 is faster                      */
/*  Note: needs two seeds! To initialize, call twice passing the two  */
/*        seeds in succession.                                        */
/*--------------------------------------------------------------------*/
double FrUMrand2(double *seed1, double *seed2)
{
  static double fSeed1 = 9876.;
  static double fSeed2 = 54321.;

  double iz, r;
  int k;

  /*--------------------------------------------get the seed values---*/
  if(seed1 != NULL) {
      if(*seed1 >= 2147483647) *seed1 = 0;
      if(*seed2 >= 2147483647) *seed2 = 0;
      if(*seed1 == 0) *seed1 = 9876;
      if(*seed2 == 0) *seed2 = 54321;}
  else {
    seed1 = &fSeed1;
    seed2 = &fSeed2;}
 
  /*-------------------------------------------------------- compute */
  k = (int)(*seed1/53668.);
  *seed1 = 40014.*(*seed1 - k*53668.) - k*12211.;
  if (*seed1 < 0) *seed1 += 2147483563;
  k = (int)(*seed2/52774.);
  *seed2 = 40692.*(*seed2 - k*52774.) - k*3791.;
  if (*seed2 < 0) *seed2 += 2147483399.;
  iz = *seed1 - *seed2;
  if (iz <= 0) iz += 2147483562.;
  r = iz*4.6566128e-10;

  return(r);
}

