/*****************************************************************************/
/*  FdIO.h                                                                   */
/*   This library contains basic API's for I/O interfacing                   */
/*   on disk, ethernet and shared memory                                     */
/*  By A. Masserot, B. Mours, D Verkindt  (LAPP Annecy)                      */
/*****************************************************************************/
#if !defined(FDIOLIB_DEFINED)
#define FDIOLIB_DEFINED DONE
/*---------------------------------------------------------------------------*/

typedef struct FdIO FdIO;

/*---- the next three lines are for the usleep define */
#undef _XOPEN_SOURCE
#define _XOPEN_SOURCE 500
#include <unistd.h>

#undef _GNU_SOURCE
#define _GNU_SOURCE
   
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/time.h>
#include <sys/timeb.h>

#ifdef NOCM
#include <CbfHeader.h>
#include <CbfHash.h>
#include <CbfClean.h>
#include <CfgParse.h>
#else
#include <FdUtil.h>
#include <CfgVirgo.h>
#include <FdCm.h>
#endif


#include "FdTools.h"

#ifdef __cplusplus
extern "C" {
#endif

  typedef struct FdAction FdAction;

  struct FdAction {        /* contains the pointers for the various actions         */
    void*      data;       /* hold the data associated to this action               */
    void     (*action)();  /* this is the function for this action                  */
    char*      type;       /* describe the type of action                           */
    FRBOOL     toBeRemoved;/* tell if the action must be removed from the list      */
    int        debugLvl;   /* debug level for this action                           */
    char*      serData;    /* records the info for the FrSerData of this process    */
    int        serDataSize;/* size of the previous string                           */
    char*      userInfo;   /* records the info for the userInfo msg of this process */
    int        userInfoSize;/*size of the previous string                           */
    int        state;      /* state (active/golden) of this action                  */
    FdAction*  next;       /* next action in the linked list                        */
    /*---the following parameters are used for the first action in the linked list  */
    int        lastTime;   /* C time of arrival of the last frame                   */
    int        frameDt;    /* length of the last frame                              */
    char*      tag;        /* tag for the output file                               */
    FRBOOL     terminated; /* tell if the last call has already been made           */
  };

  /*----remarks: it is up to the process function to call the next one 
    a process function is call with frame=NULL to terminate the processing
    and close all open files ---------------------------*/

  FdAction* FdActionNewRaw(void *data, void (*action)(), char *type);
  FdAction* FdActionNew(FdAction** root, void *data, 
			void (*action)(), char *type);
  void      FdActionRemove(FdAction* action);
  FdAction* FdActionGetLast(FdAction* root);

struct FdIO                    /* main structure                                */
{
  FdAction*   actionsIn;       /* linked list of action performed on input       */
  FdAction*   actionsOut;      /* linked list of action performed on output      */
  FRBOOL      deleteMsg;       /* flag to remove all FrMsg from the input frames */
  FRBOOL      reportFrErrors;  /* tell if error message from Fr are reported     */
  FRBOOL      noMissFrameMsg;  /* tell if missing frame are reported             */
  int         frameFormat;     /* use to set the output frame format version     */
  double      getFrTimeOut;    /* Cm timeout used at the end of FrGetFrame call  */
  double      putFrTimeOut;    /* Cm timeout used at the end of FrPutFrame call  */
  int         noFrMsgTimeOut;  /* timeout to print the no frame received messages*/
  FRBOOL      getFrStatus;     /* status of FdIOGetFrame: receiving frame or not */
  FrameH**    iFrameList;      /* list of input frame used by FdActionLastIn     */
  int         iFrameListIndex; /* index position in the input list of frames     */
  int         iFrameListSize;  /* size (number of frames) of iFrameList          */
  char*       dynamicSrc;      /* name of the frame source if dynamic connection */
  char*       inputTag;        /* tag to apply on the input frame                */
  char*       userInfo;        /* last userInfo message                          */
  char*       moreUserInfo;    /* userInfo provided by the application linking Fd*/
  char*       serData;         /* last data block for FrSerData                  */
  double      lastIFrameDt;    /* length of the last input  frame received       */
  double      lastOFrameDt;    /* length of the last output frame received       */
  double      lastIFrameGTime; /* GPS time of the last input  frame processed    */
  double      lastOFrameGTime; /* GPS time of the last output frame processed    */
  int         lastFrameTime;   /* Unix time for the last received frame          */
  FRBOOL      terminating;     /* tell if we are terminating the processing      */
  FRBOOL      terminated;      /* tell if all output process have been terminated*/
  double      maxSpeed;        /* maximum input speed (frame/sec)                */
  int         minLatency;      /* minimum latency to accept a frame              */
  int         compressIn;      /* compression type for input frames              */
  int         nFramesGet;      /* number of frame processed (FdIOGetFrame)       */
  int         nFramesPut;      /* number of output frames processed              */
  int         nextStop;        /* next GPS time or modulo to stop processing     */
  int         nextStopD;       /* next GPS time or modulo to stop (default value)*/
  double      inputTime;       /* time to input a frame (spend in FdIOGetFrame)  */
  double      outputTime;      /* time to output a frame (spend in FdIOPutFrame) */
  CfgParse_t* parser;          /* parseur                                        */
  CfgBool_t   noConfigCheck;   /* if = CfgTrue do not check unused config. keys  */ 
  int         tStart;          /* program start time (Unix time)                 */
  CfgBool_t   autoState;       /* if = CfgFalse; do not update program state     */
  char*       chPrefix;        /* channel name prefix; by default "V1:Daq"       */
  CmConnectPrinter cmPrinter;  /* printer used by Cm                             */
};

FdIO* FdIOGetRoot();
int   FdIOGetCompressIn();
FdIO* FdIONew(int  argc, char *argv[]);
FdIO* FdIONewNoCfgIdle();
void  FdIOParseAndIni(FdIO* fdIO);
void FdIODataSenderRequest (char *DsName, char *destName, char *inputfile, char *tag, int gpsStart, int nSeconds);
void FdIODataSenderWaitingTime (char *DsName, char *destName, double waiting_time);
void    FdIOStat (struct FdIO *fdIO, FrameH* frame);
FrameH* FdIOGetFrame(FdIO* fdIO);
void    FdIOPutFrame(FdIO* fdIO, FrameH* frame);
void    FdUpdateState(FdIO *fdIO, int state);
void    FdSetUserState(FdIO *fdIO, int state);
int     FdResizeAdd(FdAction** actionRoot,   double newDuration, double firstGPS);
void    FdIOSetInputFrameDuration(FdIO* fdIO, int length);
void    FdOFileCheckPath(char *name);

#ifdef __cplusplus
}
#endif

#endif
