/*****************************************************************************/
/*  FdIO.c                                                                   */
/*   This library contains basic API's for I/O interfacing                   */
/*   on disk, ethernet and shared memory                                     */
/*  By A. Masserot, B. Mours, D Verkindt  (LAPP Annecy)                      */
/*****************************************************************************/

#include <FdIO.h>
#include "FdCm.h"
#include <errno.h>

/*--------declare external function which enrich the parser---*/

void FdBufferParserAdd  (FdIO* fdIO);
void FdChkDupParserAdd  (FdIO* fdIO);
void FdClnDirParserAdd  (FdIO* fdIO);
void FdCombineParserAdd (FdIO* fdIO);
void FdCompressParserAdd(FdIO* fdIO);
void FdDemodParserAdd   (FdIO* fdIO);
void FdEvtParserAdd     (FdIO* fdIO);
void FdIDirParserAdd    (FdIO* fdIO);
void FdFbtParserAdd     (FdIO* fdIO);
void FdFileParserAdd    (FdIO* fdIO);
void FdFilterParserAdd  (FdIO* fdIO);
void FdNewFrameParserAdd(FdIO* fdIO);
void FdPlaybackParserAdd(FdIO* fdIO);
void FdRangeGatingAdd   (FdIO* fdIO);
void FdRecastParserAdd  (FdIO* fdIO);
void FdRenameParserAdd  (FdIO* fdIO);
void FdResizeParserAdd  (FdIO* fdIO);
void FdRescaleParserAdd (FdIO* fdIO);
void FdShmParserAdd     (FdIO* fdIO);
void FdSiParserAdd      (FdIO* fdIO);
void FdStatParserAdd    (FdIO* fdIO);
void FdThreadParserAdd  (FdIO* fdIO);
void FdToAclParserAdd   (FdIO* fdIO);
void FdFreqAnaParserAdd (FdIO* fdIO);

void MbThreadWaitForAllDone(FdAction *action);

void FdChkDupIni        (FdIO *fdIO);

FdIO* MyFdIO;    /*--- global variable needed by the Cm Frame Handler ------*/
int   FdDebugLvl;/*--- gloal variable to record the current debug level------*/
/*----------------------------------------- Declare private functions -------*/

CmMessageStatus FdIOSmsHandler (CmMessage message, char* from, char* server);
CmMessageStatus FdIONextStopHandler(CmMessage message, char* from, char* server);
CmMessageStatus FdIOCmFrameHandler(CmMessage message, char* sender, char* serverName);
char* FdSerDataBuild (FdIO *fdIO);

/*--------------------------------------------------------------------------*/
FdAction* FdActionGetLast(FdAction* root)
/*--------------------------------------------------------------------------*/
/* Thist function returns the last created action                           */
/*--------------------------------------------------------------------------*/
{
  FdAction *action;

  if(root == NULL) return(NULL);

  action = root;
  while(action->next != NULL) action = action->next;

  return(action);
}
/*--------------------------------------------------------------------------*/
FdAction* FdActionNewRaw(void *data,
		      void (*action)(),
		      char *type)
/*--------------------------------------------------------------------------*/
/* This function creates an FdAction structure                              */
/*--------------------------------------------------------------------------*/
{
  FdAction *this;

  this = (FdAction*) calloc(1,sizeof(FdAction));
  if(this == NULL) CfgMsgAddFatal("malloc FdAction failed");
  this->data   = data;
  this->action = action;
  FrStrCpy(&this->type, type);
  this->tag    = NULL;
  this->toBeRemoved = FR_NO;
  this->debugLvl    = FdDebugLvl;
  this->state = CfgServerGolden;

  this->serDataSize = 256;
  this->serData     = malloc(this->serDataSize);
  this->serData[0]  ='\0';
  this->userInfoSize = 2048;
  this->userInfo     = malloc(this->userInfoSize);
  this->userInfo[0]  ='\0';
  this->terminated   = FR_NO;

  return(this);
}
/*--------------------------------------------------------------------------*/
FdAction* FdActionNew(FdAction** root, 
		      void *data,
		      void (*action)(),
		      char *type)
/*--------------------------------------------------------------------------*/
/* This function creates an FdAction and add it at the end of the linked 
   list starting from root. It returns the created FdAction.               */
/* Each "action" is supposed to call the next one in the linked list, for 
   each frame produced. */
/* If the action is call with a NULL frame, this is to close all open files*/
/*--------------------------------------------------------------------------*/
{
  FdAction *this, **r;

  this = FdActionNewRaw(data, action, type);

  r = root;
  while(*r != NULL) r = &((*r)->next);
  *r = this;

  return(this);
}
/*--------------------------------------------------------------------------*/
void FdActionRemove(FdAction* action)
/*--------------------------------------------------------------------------*/
/* Thist function removes an FdAction from the linked list                  */
/*--------------------------------------------------------------------------*/
{
  free(action->type);
  if(action->serData  != NULL) free(action->serData);
  if(action->userInfo != NULL) free(action->userInfo);

  free(action);

  return;
}
/*---------------------------------------------------------------------------*/
void FdActionLastIn(FdIO *fdIO, 
		    FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  if(frame == NULL) return;

  if(fdIO->iFrameListIndex+1 >= fdIO->iFrameListSize) {
    fdIO->iFrameListSize *= 2;
    fdIO->iFrameList = realloc(fdIO->iFrameList, 
			       fdIO->iFrameListSize * sizeof(FrameH*));}

  fdIO->iFrameListIndex++;
  fdIO->iFrameList[fdIO->iFrameListIndex] = frame;

  return;
}
/*---------------------------------------------------------------------------*/
void FdActionLastOut(FdIO *fdIO,
		     FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  if(frame != NULL) {
    fdIO->nFramesPut++;
    FrameFree(frame);}
  else if(fdIO->terminating == FR_YES) {// all output objects have been flushed
    CfgMsgAddInfo("          %d frames processed (out)", fdIO->nFramesPut);
    fdIO->terminated = FR_YES;}

  return;
}
/*--------------------------------------------------------------------------*/
void FdFrErrorHandler(int level, char *message) 
/*--------------------------------------------------------------------------*/
{

  if(MyFdIO->reportFrErrors == FR_NO) return;

  if     (level > 2) CfgMsgAddError  ("%s", message);
  else if(level > 1) CfgMsgAddWarning("%s", message);
  else               CfgMsgAddInfo   ("%s", message);

  return;
}
/*--------------------------------------------------------------------------*/
int FdICompressNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{        
  if(*((FdAction**) arg) != NULL) 
    CfgMsgAddFatal("FDIN_COMPRESS must be declare before the %s key",
                   (*((FdAction**) arg))->type);

  MyFdIO->compressIn  =  CfgParseGetNextDec(data);

  return(CFG_OK);
}
/*-------------------------------------------------------------FdIOExitFnt---*/
void FdIOExitFnt (void)
/*---------------------------------------------------------------------------*/
{
  return;}

/*-------------------------------------------------------------FdIOGetRoot---*/
FdIO* FdIOGetRoot()
/*---------------------------------------------------------------------------*/
{
  return(MyFdIO);}

/*-------------------------------------------------------FdIOGetCompressIn---*/
int FdIOGetCompressIn()
/*---------------------------------------------------------------------------*/
{
  return(MyFdIO->compressIn);}

/*---------------------------------------------------------FdIOQuitHandler---*/
void *FdIOQuitHandler (void *dummy) 
/*---------------------------------------------------------------------------*/
{
  FdIO* fdIO;

  fdIO = dummy;

  if( !CbfCleanFind( "FdIOQuitHandler", fdIO) ) return(fdIO);
  CfgMsgAddInfo("FdIOQuit: %d frames processed (get); CPU= %ld s for all threads",
    fdIO->nFramesGet, clock()/CLOCKS_PER_SEC);

  /*---------in case of dynamic connectection to the source, disconnect it---*/
  if(fdIO->dynamicSrc != NULL) 
    FdRemoveFramesRequest(fdIO->dynamicSrc,CfgGetCmName());

  /*-------------- call the output actions to tell then to close all files---*/
  fdIO->terminating = FR_YES;
  if(fdIO->actionsOut != NULL) {
    while(fdIO->terminated == FR_NO) {
      fdIO->actionsOut->action(fdIO->actionsOut->data, NULL);}}

  CbfCleanDone("FdIOQuitHandler", fdIO);

  return(NULL);
}
/*---------------------------------------------------------------- FdIOIni---*/
FdIO* FdIOIni(int  argc, char *argv[], char *appName)
/*---------------------------------------------------------------------------*/
/* This obsolete function is provided for backward compabitlity 
   but must not be used since it might be removed in a futur version */
/*---------------------------------------------------------------------------*/
{
  FdIO* fdIO;

  fdIO = FdIONew(argc, argv);

  return(fdIO);
}
/*---------------------------------------------------------------- FdIONew---*/
FdIO* FdIONew(int  argc, char *argv[])
/*---------------------------------------------------------------------------*/
{
  FdIO* fdIO;

  if(CfgIdle( argc, argv, 1000) == CFG_FAIL) exit(EXIT_FAILURE);

  fdIO = FdIONewNoCfgIdle();

  FrErrorSetHandler(FdFrErrorHandler);

  return(fdIO);
}
/*------------------------------------------------------ FdIONewNoCfgIddle---*/
FdIO* FdIONewNoCfgIdle()
/*---------------------------------------------------------------------------*/
{
  FdIO* fdIO;

  CfgMsgAddInfo("Fd version %s (compiled: %s %s)", FD_PATH, __DATE__, __TIME__);
 
  fdIO = (FdIO*) calloc(1, sizeof(FdIO));
  if(fdIO == NULL) CfgMsgAddFatal("FdIOIni malloc failed");
  MyFdIO = fdIO;

  fdIO->actionsIn  = NULL;
  fdIO->actionsOut = NULL;
  fdIO->reportFrErrors = FR_NO;
  fdIO->lastIFrameGTime = 0;
  fdIO->lastOFrameGTime = 0;
  fdIO->getFrTimeOut = .001;
  fdIO->putFrTimeOut = .0001;
  fdIO->noFrMsgTimeOut = 20;
  fdIO->getFrStatus = FR_NO;
  fdIO->lastFrameTime = time(NULL);
  fdIO->frameFormat = 8;
  fdIO->deleteMsg   = FR_NO;
  fdIO->inputTag  = NULL;
  fdIO->tStart    = 0;
  fdIO->maxSpeed = 0;
  fdIO->minLatency = -10;
  fdIO->inputTime = 0;
  fdIO->outputTime = 0;
  fdIO->noConfigCheck = CfgFalse;
  fdIO->autoState = CfgFalse;
  fdIO->userInfo = NULL;
  fdIO->serData  = NULL;
  fdIO->compressIn = -1;
  fdIO->nextStop  = 0;
  fdIO->nextStopD = 1;
  fdIO->terminating = FR_NO;
  fdIO->terminated  = FR_NO;
  fdIO->noMissFrameMsg = FR_NO;

  fdIO->iFrameListIndex = -1;
  fdIO->iFrameListSize  = 2;
  fdIO->iFrameList = calloc(fdIO->iFrameListSize, sizeof(FrameH*));
  FrStrCpy(&fdIO->chPrefix, "V1:Daq_");

  /*---------------------------------------------------------get Cfg printer---*/
  if(CmConnectGetDomain() != NULL) {
    fdIO->cmPrinter = CmMessageInstallPrinter((CmConnectPrinter)NULL);
    CmMessageInstallPrinter(fdIO->cmPrinter);}

  /*---------------------------------------declare parser and few basic keys---*/
  fdIO->parser = CfgParseNew();

  CfgParseAdd(fdIO->parser,"FDIN_NO_FR_MSG_TIMEOUT",   
              1, CfgDec,   &fdIO->noFrMsgTimeOut);
  CfgParseAdd(fdIO->parser,"FDIN_CM_TIMEOUT",   1,CfgReal,  &fdIO->getFrTimeOut);
  CfgParseAdd(fdIO->parser,"FDIN_MAX_SPEED",    1,CfgReal,  &fdIO->maxSpeed);
  CfgParseAdd(fdIO->parser,"FDIN_MIN_LATENCY",  1,CfgDec,   &fdIO->minLatency);
  CfgParseAdd(fdIO->parser,"FDIN_NOMSG",        1,CfgFlag,  &fdIO->deleteMsg);
  CfgParseAdd(fdIO->parser,"FDIN_TAG",          1,CfgString,&fdIO->inputTag);

  CfgParseGetFunctionAdd(fdIO->parser,"FDIN_COMPRESSION",
                         FdICompressNew, (void *) &(fdIO->actionsIn),
                         1, CfgDec);

  CfgParseAdd(fdIO->parser,"FDOUT_FRAME_FORMAT",1,CfgDec,   &fdIO->frameFormat);
  CfgParseAdd(fdIO->parser,"FDOUT_CM_TIMEOUT",  1,CfgReal,  &fdIO->putFrTimeOut);
  CfgParseAdd(fdIO->parser,"FDOUT_SET_NEXT_STOP",1,CfgDec,  &fdIO->nextStopD);

  CfgParseAdd(fdIO->parser,"FD_DEBUGLVL",       1,CfgDec,   &FdDebugLvl); 
  CfgParseAdd(fdIO->parser,"FD_CH_PREFIX",      1,CfgString,&fdIO->chPrefix);
  CfgParseAdd(fdIO->parser,"FD_REPORT_FR_ERRORS",1,CfgFlag, &fdIO->reportFrErrors);
  CfgParseAdd(fdIO->parser,"FD_NO_MISS_FRAME_MSG",1,CfgFlag,&fdIO->noMissFrameMsg);

  /*----------------------------------------------------------enrich parser---*/
  FdBufferParserAdd  (fdIO);
  FdChkDupParserAdd  (fdIO);
  FdClnDirParserAdd  (fdIO);
  FdCmParserAdd      (fdIO);
  FdCombineParserAdd (fdIO);
  FdCompressParserAdd(fdIO);
  FdDemodParserAdd   (fdIO);
  FdEvtParserAdd     (fdIO);
  FdFbtParserAdd     (fdIO);
  FdFileParserAdd    (fdIO);
  FdFilterParserAdd  (fdIO);
  FdIDirParserAdd    (fdIO);
  FdNewFrameParserAdd(fdIO);
  FdPlaybackParserAdd(fdIO);
  FdRangeGatingAdd   (fdIO);
  FdRecastParserAdd  (fdIO);
  FdRenameParserAdd  (fdIO);
  FdRescaleParserAdd (fdIO);
  FdResizeParserAdd  (fdIO);
  FdSiParserAdd      (fdIO);
  FdStatParserAdd    (fdIO);
  FdThreadParserAdd  (fdIO);
  FdToAclParserAdd   (fdIO);
  FdFreqAnaParserAdd (fdIO);

  CfgReachState(CfgServerIdle);

  return(fdIO);
}
/*----------------------------------------------------------------------------*/
void FdIOParse(FdIO* fdIO)
/*----------------------------------------------------------------------------*/
/* This obsolete function is just provided for backward compatibility
   it must not be used since it might be deleted in a futur version. */
{
  FdIOParseAndIni(fdIO);

  return;
}
/*----------------------------------------------------------------------------*/
void FdIOParseAndIni(FdIO* fdIO)
/*----------------------------------------------------------------------------*/
{
  char *unused;
  CfgParse_t *parser;
  int len, i;

  if (CfgGetConfig() == NULL) CfgMsgAddFatal("Could not get Config" );

  /*-----first a simple parser to know if we should check for unused words---*/
  parser = CfgParseNew();
  CfgParseAdd(parser,"FD_NO_CONFIG_CHECK",1,CfgFlag,  &fdIO->noConfigCheck);
  CfgParseBuffer(parser, CfgGetConfig());

  /*----------------------------if requested; check that all keys are known---*/
  if(fdIO->noConfigCheck == CfgFalse) {
    CfgParseIdleAddNoAction(fdIO->parser);   /*--------- to ignore Cfg keys---*/
    unused = CfgParseExtract(fdIO->parser, CfgGetConfig(), CfgFalse );
    if(unused != NULL) {
      len = strlen(unused);  /*------Cfg returns "\n" for C style comments---*/
      for(i=0; i<len; i++) {if(unused[i] !=  '\n') break;}
      if(i<len) {
        if(unused[len-1] ==  '\n') unused[len-1] = '\0';
        CfgMsgAddFatal("Unused word(s) in config: \"%s\"",unused);}}}
  else
    CfgMsgAddInfo("The configuration is not checked for unsed keys");

  /*----------------------------------------------------parse configuration---*/
  if (CfgParseBuffer(fdIO->parser, CfgGetConfig() ) == CFG_FAIL) 
    CfgMsgAddFatal("Could not parse Config" );
  CfgMsgAddInfo("Configuration successfully parsed");

  /*-----------add the last input action and initialize the associated list---*/
  FdActionNew(&(fdIO->actionsIn), (void*) fdIO, FdActionLastIn, "InputList");

  /*------make sure there is at an FdOCm server in the output actions---*/
  FdOCmServerAdd(2, 0, -1);

  /*--------------------------------- add the last output action: FrameFree---*/
  FdActionNew(&(fdIO->actionsOut), (void*) fdIO, FdActionLastOut, "FrameFree");

  /*--------------- check that the input tag is a real tag and not just "*"---*/
   if(fdIO->inputTag != NULL) {
     len = strlen(fdIO->inputTag);
     for(i=0; i<len; i++) {
       if(fdIO->inputTag[i] != ' ' &&
          fdIO->inputTag[i] != '*') break;}
     if(i == len) {
       free(fdIO->inputTag);
       fdIO->inputTag = NULL;}}

  /*---------------------------------copy the input tag to the first action---*/
  if(fdIO->inputTag != NULL) {
    CfgMsgAddInfo("Channel(s) selection on the input file: %s",fdIO->inputTag);
    if(fdIO->actionsIn == NULL) 
      CfgMsgAddFatal("FDIN_TAG asked but no frame sources defined");
    fdIO->actionsIn->tag = fdIO->inputTag;}

  /*---------------------------------------------- set frame format version---*/
  FrLibSetVersion(fdIO->frameFormat);

  /*--------------------------------------------------do some specific init---*/
  FdChkDupIni(fdIO);

  /*---------------------------------------------- install some Cm handler ---*/
  CmMessageInstallHandler ((CmMessageHandler)FdIOSmsHandler, "FbGetAllSmsData");
  CmMessageInstallHandler ((CmMessageHandler)FdIONextStopHandler, 
                           "FdSetNextStop");

  /*--------------------install quit handler for actions like closing files---*/ 
  CbfCleanAdd(NULL,"FdIOQuitHandler",FdIOQuitHandler, (void *)fdIO);
  CbfCleanSetExitFnt( (CbfCleanExitFnt_t) FdIOExitFnt);

  /*--------------------------------------------- process is now configured---*/
  CfgReachState(CfgServerConfigured);

  CfgMsgSendWithTimeout(0.01);

  return;
}
/*----------------------------------------------------------------------------*/
CmMessageStatus FdIOSmsHandler (CmMessage message, char* from, char* server)
/*----------------------------------------------------------------------------*/
{
  CmMessage answer;
  int frameN;

  CmMessageGetText(message);     /* text to tell which channel to provide   */
  CmMessageGetText(message);     /* text to get the GPS time of the request */
  frameN = CmMessageGetInt(message);

  if (MyFdIO == NULL) return(CmMessageOk);

  if(MyFdIO->serData == NULL) {
    CfgMsgAddWarning("No SMS data could be provided to %s",from);
    return(CmMessageOk);}

  answer = CmMessageNew();
  CmMessageSetType(answer, "FbSmsData");
  CmMessagePutInt (answer, frameN);
  CmMessagePutText(answer, MyFdIO->serData);

  CmMessageSend(answer, from);

  CmMessageDelete(answer);

  return(CmMessageOk);
}
/*----------------------------------------------------------------------------*/
CmMessageStatus FdIONextStopHandler (CmMessage message, char* from, char* server)
/*----------------------------------------------------------------------------*/
{
  int lastGTime;

  MyFdIO->nextStop = CmMessageGetInt(message);

  if(MyFdIO->lastIFrameGTime > 0) lastGTime = MyFdIO->lastIFrameGTime;
  else                            lastGTime = 1000000000;

  if(MyFdIO->nextStop < 0) {
    CfgMsgAddInfo("%s asked to stop the processing when GPS time will"
                  " be a multiple of %d (predefined)",from, MyFdIO->nextStopD);
    MyFdIO->nextStop = MyFdIO->nextStopD;}
  else if(MyFdIO->nextStop == 0) {
    CfgMsgAddInfo("%s asked to remove the stop processing request",from);}
  else if(MyFdIO->nextStop < lastGTime) {
    CfgMsgAddInfo("%s asked to stop the processing when GPS time will"
                  " be a multiple of %d",from, MyFdIO->nextStop);}
  else {
    CfgMsgAddInfo("%s asked to stop the processing when gps time will"
                  " reaches %d",from, MyFdIO->nextStop);}

  return(CmMessageOk);
}
/*----------------------------------------------------------------------------*/
FrameH *FdIOGetFrameDoIt(FdIO* fdIO)
/*----------------------------------------------------------------------------*/
{
  FrameH *frame;
  int i, wait, latency;
  double frameGTime, delta;
  struct timeval tNow;

  if(fdIO->actionsIn == NULL) 
    CfgMsgAddFatal("FdIOGetFrame called but not FDIN actions requested");

  /*--------------no frame available; go througth the input list of actions---*/
  if(fdIO->iFrameListIndex < 0) 
    fdIO->actionsIn->action(fdIO->actionsIn->data, NULL);

  /*--do we have a new frame? if yes, take the first one and shift the list---*/
  if(fdIO->iFrameListIndex >= 0) {
    frame = fdIO->iFrameList[0];
    for(i = 0; i< fdIO->iFrameListIndex; i++) {
      fdIO->iFrameList[i] = fdIO->iFrameList[i+1];}
    fdIO->iFrameListIndex--;}
  else {
    return(NULL);}

  /*--------------------------------------- reject frame with no GPS time ---*/
  if (frame->GTimeS == 0) {
    CfgMsgAddError("FdIOGetFrame: Reject frame %s without GPS",frame->name);
    FrameFree(frame);
    return(NULL);}

  /*--------------------------------------------------reject futur frames ---*/
  if((latency = FrameLatency(frame)) < fdIO->minLatency) {
    CfgMsgAddError("FdIOGetFrame: Reject futur frame: GPS=%d latency=%d",
		   frame->GTimeS, latency);
    FrameFree(frame);
    return(NULL);}

  /*---------------------------------------------------reject late frames ---*/
  frameGTime = frame->GTimeS + 1.e-9*frame->GTimeN;
  if (frameGTime <= fdIO->lastIFrameGTime && fdIO->lastIFrameGTime > 0) {
    CfgMsgAddError( "FdIOGetFrame: Reject late frame: GPS=%.1f (last=%.1f)",
		    frameGTime, fdIO->lastIFrameGTime);
    FrameFree(frame);
    return(NULL);}

  /*-----------------------------------------warning for missing frames ---*/
  delta =fabs(fdIO->lastIFrameGTime + fdIO->lastIFrameDt - frameGTime); 
  if ((delta > 0.001*fdIO->lastIFrameDt) && 
      (fdIO->lastIFrameGTime > 0) &&
      (fdIO->noMissFrameMsg == FR_NO)) {
    CfgMsgAddWarning ( "FdIOGetFrame: miss %g seconds between %.1f and %.1f",
		       delta,      fdIO->lastIFrameGTime+fdIO->lastIFrameDt, 
		       frameGTime);}

  fdIO->lastIFrameGTime = frameGTime;
  fdIO->lastIFrameDt    = frame->dt;

  /*--------------------------------------------remove FrMsg if requested ---*/
  if(fdIO->deleteMsg == FR_YES && frame->rawData != NULL) {
    FrMsgFree(frame->rawData->logMsg);
    frame->rawData->logMsg = NULL;}

  /*------------------------------------ check reading speed if requested ---*/
  if (fdIO->maxSpeed > 0 && fdIO->nFramesGet > 0) {
    gettimeofday(&tNow, NULL);
    if(fdIO->tStart == 0) fdIO->tStart = tNow.tv_sec;
    wait =  fdIO->nFramesGet/fdIO->maxSpeed + (fdIO->tStart - tNow.tv_sec);
    wait  = 1.e6*wait - tNow.tv_usec;
    if(wait > 0) usleep(wait);}
      
  return(frame);}

/*----------------------------------------------------------------------------*/
FrameH *FdIOGetFrame(FdIO* fdIO)
/*----------------------------------------------------------------------------*/
{
  FrameH *frame;
  int dt, state;
  struct timeval tS, tE;

  gettimeofday(&tS, NULL);
  state = CfgServerGolden;

  frame = FdIOGetFrameDoIt(fdIO);

  /*------------if no frame update the program state and messages if needed--*/
  if (frame == NULL) {
    dt = time(NULL)-fdIO->lastFrameTime;
    if (dt > fdIO->noFrMsgTimeOut + 2*fdIO->lastIFrameDt) {
      if (fdIO->getFrStatus == FR_YES)
	CfgMsgAddWarning("No input frame since at least %d seconds", dt);
      fdIO->getFrStatus = FR_NO;
      state = CfgServerActive;
      CfgMsgAddUserInfo("Warning: no input frame since %d s", dt);}}

  /*----------- a frame is available; update messages and last frame info ---*/
  else {
    if (fdIO->getFrStatus == FR_NO) 
      CfgMsgAddInfo("Input frames are back; gps=%d latency=%.1f",
                    frame->GTimeS, FrameLatency(frame));
    fdIO->getFrStatus = FR_YES;
    fdIO->lastFrameTime = time(NULL);
    fdIO->nFramesGet ++;

    /*-------------------------------------------- update input statistics ---*/
    gettimeofday(&tE, NULL);
    fdIO->inputTime = tE.tv_sec -tS.tv_sec + 1.e-6 * (tE.tv_usec - tS.tv_usec);}

  /*--------------------------------------------------update program state---*/
  FdUpdateState(fdIO, state);

  /*----------------------------------------------------flush Cfg messages---*/
  if(fdIO->getFrTimeOut >= 0) CfgMsgSendWithTimeout(fdIO->getFrTimeOut);

  return(frame);
}
/*---------------------------------------------------------------------------*/
void FdIOPutFrame(FdIO* fdIO, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  struct timeval tS, tE;
  int endFrame, cmState;
  FdAction *action;
  double delta, frameGTime;

  if(frame == NULL) return;
 
  gettimeofday(&tS, NULL);

  /*-----------------------------------------warning for missing frames ---*/
  frameGTime = frame->GTimeS + 1.e-9*frame->GTimeN;
  delta =fabs(fdIO->lastOFrameGTime + fdIO->lastOFrameDt - frameGTime); 
  if ((delta > 0.001*fdIO->lastOFrameDt) && 
      (fdIO->lastOFrameGTime > 0) &&
      (fdIO->noMissFrameMsg == FR_NO)) {
    CfgMsgAddWarning ( "FdIOPutFrame: miss %g seconds between %.1f and %.1f",
		       delta,      fdIO->lastOFrameGTime+fdIO->lastOFrameDt, 
		       frameGTime);}

  fdIO->lastOFrameGTime = frameGTime;
  fdIO->lastOFrameDt    = frame->dt;

  /*------------------------------check if we should stop after this frame---*/
  if(fdIO->nextStop > 0) {
    endFrame = frame->GTimeS + frame->dt;
    if(endFrame == fdIO->nextStop ||
       endFrame  % fdIO->nextStop == 0) {
      CfgMsgAddInfo("GPS time reaches the requested stop: %d", endFrame);
      CfgTerminate();}}


  /*---performes the output actions; each actions should call the next one---*/
  if(fdIO->actionsOut != NULL) 
    fdIO->actionsOut->action(fdIO->actionsOut->data, frame);

  /*--------------------------if needed remove the Cm output from the list---*/
  FdOCmCleanup();

  /*-------------------disable/enable Cm printer in case of Cm output error---*/
  if(CmConnectGetDomain() != NULL) {
    cmState = CfgServerGolden;
    for(action = fdIO->actionsOut; action != NULL; action = action->next) {
      if(action->state == CfgServerGolden) continue;
      if(strstr(action->type,"CmOutput") == NULL) continue;
      cmState = action->state;
      break;}
   
    if(cmState == CfgServerGolden) 
         CmMessageInstallPrinter(fdIO->cmPrinter);
    else CmMessageInstallPrinter((CmConnectPrinter)NULL);}

  /*---------------------------------------------------compute output time---*/
  gettimeofday(&tE, NULL);
  fdIO->outputTime = tE.tv_sec -tS.tv_sec + 1.e-6 * (tE.tv_usec - tS.tv_usec);
 
  /*--------------------------------------------------update program state---*/
  FdUpdateState(fdIO, CfgServerGolden);

  /*----------------------------------------------------flush Cfg messages---*/
  if(fdIO->putFrTimeOut >= 0) CfgMsgSendWithTimeout(fdIO->putFrTimeOut);

  return;
}
/*---------------------------------------------------------------------------*/
void FdUpdateState(FdIO *fdIO, int stateIn)
/*---------------------------------------------------------------------------*/
{
  FdAction *action;
  int state;

  if(fdIO == NULL) return;
  if(stateIn < 0) fdIO->autoState = CfgFalse;
  if(fdIO->autoState == CfgFalse) return;

  if(stateIn == CfgServerError  ||
     stateIn == CfgServerActive ||
     stateIn == CfgServerGolden) state = stateIn;
  else                           state = CfgServerGolden;

  for(action = fdIO->actionsIn; action != NULL; action = action->next) {
    if(action->state == CfgServerGolden) continue;
    state = action->state;
    if(action->state == CfgServerError) break;}

  if(state != CfgServerError) {
    for(action = fdIO->actionsOut; action != NULL; action = action->next) {
      if(action->state == CfgServerGolden) continue;
      state = action->state;
      if(action->state == CfgServerError) break;}}

  if(stateIn > 0) CfgReachState(state);
 
  return;
}

/*---------------------------------------------------------------------------*/
void FdSetUserState(FdIO *fdIO, int stateIn)
/*---------------------------------------------------------------------------*/
{
  FdAction *action;
  int state;

  if(fdIO == NULL) return;
  fdIO->autoState = CfgFalse;

  if(stateIn == CfgServerError  ||
     stateIn == CfgServerActive ||
     stateIn == CfgServerGolden) state = stateIn;
  else                           state = CfgServerGolden;

  for(action = fdIO->actionsIn; action != NULL; action = action->next) {
    if(action->state == CfgServerGolden) continue;
    state = action->state;
    if(action->state == CfgServerError) break;}

  if(state != CfgServerError) {
    for(action = fdIO->actionsOut; action != NULL; action = action->next) {
      if(action->state == CfgServerGolden) continue;
      state = action->state;
      if(action->state == CfgServerError) break;}}

  if(stateIn > 0) CfgReachState(state);

  return;
}
