#include "FdIO.h"

typedef struct FdFreqAna  FdFreqAna;
typedef struct FdGetBrms  FdGetBrms;
typedef struct FdGetLineA FdGetLineA;
typedef struct FdGetLines FdGetLines;

struct FdFreqAna         // common structure to analyse a channel
{
  char*       chName;    // name of the channel to process
  FRBOOL      chMissing; // tell if the main channel is missing
  double      period;    // period to search for lines
  FdGetBrms*  brms;      // linked list of Brms objects
  FdGetLineA* lineA;     // linked list of lineA objects
  FdGetLines* lines;     // linked list of lines objects
  FrVect*     vectIn;    // input vector converted to float
  FrvRFFT*    fft;       // fft object
  FdAction*   action;    // action object which hold this freqAna object
  };

struct FdGetBrms         // to search for lines in the frequency band
{
  double      fMin;      // start of the frequency band
  double      fMax;      // end of the frequency band
  int         iFirst;    // index of fMin in the FFT vector
  int         iLast;     // index of fMax in the FFT vector
  char*       outName;   // output channel name
  FrVect*     vectOut;   // ouput vector to be added to the frame
  FdGetBrms*  next;      // next object in the linked list
  };

struct FdGetLineA        // to search for lines in the frequency band
{
  double      freq;      // start of the frequency band
  int         iLine;     // index of freq in the FFT vector
  char*       outNameA;  // output channel name (amplitude)
  char*       outNameP;  // output channel name (phase)
  FrVect*     vectOutA;  // ouput vector to be added to the frame
  FrVect*     vectOutP;  // ouput vector to be added to the frame
  FdGetLineA* next;      // next object in the linked list
  };

struct FdGetLines        // to search for lines in the frequency band
{
  double      fMin;      // start of the frequency band
  double      fMax;      // end of the frequency band
  int         iFirst;    // index of fMin in the FFT vector
  int         iLast;     // index of fMax in the FFT vector
  int         nLines;    // number of lines to search
  char**      outNameF;  // output channel names (frequency) size: nLines
  char**      outNameA;  // output channel names (amplitude) size: nLines
  FrVect**    vectOutF;  // ouput vectors to be added to the frame, size nLines
  FrVect**    vectOutA;  // ouput vectors to be added to the frame, size nLines
  FdGetLines* next;      // next object in the linked list
  };

void FdFreqAnaProcess(FdFreqAna *fAna, FrameH *frame);

//---------------------------------------------------------------------------
FdFreqAna* FdFreqAnaNew(void *arg, CfgDataType_t *data)
//---------------------------------------------------------------------------
{
  char *chName;

  FrStrCpy(&chName, CfgParseGetNextString(data));
  double period = CfgParseGetNextReal(data);
 
  //-----------is the last action on the same channel with the same period?
  FdFreqAna* fAna = NULL;
  FdAction *lastAction = FdActionGetLast(*((FdAction**) arg));
  if(lastAction != NULL) {
    if(strcmp(lastAction->type,"FdFreqAna") == 0) {
      fAna = (FdFreqAna*) lastAction->data;
      if((strcmp(fAna->chName, chName) != 0) ||
         (fAna->period != period))  fAna = NULL;}}

  //-------------------------------------if not, create a freqAna structure
  if(fAna == NULL) {
    fAna = calloc(1, sizeof(FdFreqAna));
    if(fAna == NULL) CfgMsgAddFatal("FdFreqAnaNew: malloc failed");
 
    fAna->action = FdActionNew((FdAction**) arg, (void*) fAna, 
				FdFreqAnaProcess, "FdFreqAna");
    fAna->chName = chName;
    fAna->period = period;
    CfgMsgAddInfo("Channel %s will be analyzed every %g s.", chName, period);}
  else {
    free(chName);}

  //---------------------------------------------------------default state
  fAna->chMissing = FR_NO;
  fAna->action->state = CfgServerGolden;

  return(fAna);
}
//---------------------------------------------------------------------------
int FdGetLineANew(void *arg, CfgDataType_t *data)
//---------------------------------------------------------------------------
{
  FdFreqAna* fAna = FdFreqAnaNew(arg, data);

  FdGetLineA* getA= calloc(1, sizeof(FdGetLineA));
  if(getA == NULL) CfgMsgAddFatal("FdGetLineANew: malloc failed");
  getA->next = fAna->lineA;
  fAna->lineA = getA;

  getA->freq = CfgParseGetNextReal(data);
  CfgMsgAddInfo("  will search the line amplitude at %g Hz", getA->freq);

  getA->outNameA = malloc(50+strlen(fAna->chName));
  getA->outNameP = malloc(50+strlen(fAna->chName));
  if(getA->outNameP == NULL) CfgMsgAddFatal("FdGetLineANew: malloc failed");
  if(getA->freq > 1e5) {
    sprintf(getA->outNameA,"%s_%.0fkHz_mag", fAna->chName, getA->freq/1000.);
    sprintf(getA->outNameP,"%s_%.0fkHz_phi", fAna->chName, getA->freq/1000.);}
  else {
    sprintf(getA->outNameA,"%s_%.0fHz_mag", fAna->chName, getA->freq);
    sprintf(getA->outNameP,"%s_%.0fHz_phi", fAna->chName, getA->freq);}

  return(CFG_OK);
}
//---------------------------------------------------------------------------
int FdGetLinesNew(void *arg, CfgDataType_t *data)
//---------------------------------------------------------------------------
{
  int i;

  FdFreqAna* fAna = FdFreqAnaNew(arg, data);

  FdGetLines* getL= calloc(1, sizeof(FdGetLines));
  if(getL == NULL) CfgMsgAddFatal("FdGetLinesNew: malloc failed");
  getL->next = fAna->lines;
  fAna->lines = getL;

  getL->fMin   = CfgParseGetNextReal(data);
  getL->fMax   = CfgParseGetNextReal(data);
  getL->nLines = CfgParseGetNextDec(data);
  if(getL->nLines < 1) getL->nLines = 1;
  CfgMsgAddInfo("  will search the loudest %d line(s) in the %g-%g Hz range", 
	getL->nLines, getL->fMin, getL->fMax);

  getL->outNameF = calloc(getL->nLines, sizeof(char*));
  getL->outNameA = calloc(getL->nLines, sizeof(char*));
  getL->vectOutF = calloc(getL->nLines, sizeof(FrVect*));
  getL->vectOutA = calloc(getL->nLines, sizeof(FrVect*));
  if(getL->vectOutA == NULL) CfgMsgAddFatal("FdGetLinesNew: malloc failed");

  for(i=0; i<getL->nLines; i++) {
    getL->outNameF[i] = malloc(50+strlen(fAna->chName));
    getL->outNameA[i] = malloc(50+strlen(fAna->chName));
    if(getL->outNameA[i] == NULL) CfgMsgAddFatal("FdGetLinesNew: malloc failed");
  sprintf(getL->outNameF[i],"%s_line%d_freq", fAna->chName, i);
  sprintf(getL->outNameA[i],"%s_line%d_mag" , fAna->chName, i);}

  return(CFG_OK);
}
//---------------------------------------------------------------------------
int FdGetBrmsNew(void *arg, CfgDataType_t *data)
//---------------------------------------------------------------------------
{
  FdFreqAna* fAna = FdFreqAnaNew(arg, data);

  FdGetBrms* getB= calloc(1, sizeof(FdGetBrms));
  if(getB == NULL) CfgMsgAddFatal("FdGetBrmsNew: malloc failed");
  getB->next = fAna->brms;
  fAna->brms = getB;

  getB->fMin = CfgParseGetNextReal(data);
  getB->fMax = CfgParseGetNextReal(data);
  CfgMsgAddInfo("  will compute the BRMS from %g to %g kHz", 
		getB->fMin, getB->fMax);

  getB->outName = malloc(50+strlen(fAna->chName));
  if(getB->outName == NULL) CfgMsgAddFatal("FdGetBrmsNew: malloc failed");
  if(getB->fMax > 1.e5) { 
    sprintf(getB->outName,"%s_BRMS_%.0f_%.0fkHz",fAna->chName, getB->fMin/1000., getB->fMax/1000.);}
  else {
    sprintf(getB->outName,"%s_BRMS_%.0f_%.0fHz", fAna->chName, getB->fMin, getB->fMax);}

  return(CFG_OK);
}
//---------------------------------------------------------------------------
void FdGetLineAProcess(FdGetLineA *getA, FrameH *frame, FdFreqAna *fAna, 
			int iAna, int nAna)
//---------------------------------------------------------------------------
{

  if(getA->vectOutA == NULL) {
    getA->vectOutA = FrVectNewTS(getA->outNameA, 1./fAna->period, nAna, -32);
    getA->vectOutP = FrVectNewTS(getA->outNameP, 1./fAna->period, nAna, -32);
    if(getA->vectOutP == NULL) 
	CfgMsgAddFatal("FdGetLineAProcess: malloc %d failed", nAna);
    FrVectSetUnitY(getA->vectOutA, fAna->vectIn->unitY);
    FrVectSetUnitY(getA->vectOutP, "rad");
    getA->iLine = getA->freq/fAna->fft->amplitude->dx[0]-1;
    if(getA->iLine < 0) getA->iLine = 0;
    if(getA->iLine >= fAna->fft->amplitude->nx[0]) 
       getA->iLine =  fAna->fft->amplitude->nx[0]-1;}

  int iLine = getA->iLine;
  double real = fAna->fft->output->dataD[2*iLine];
  double imag = fAna->fft->output->dataD[2*iLine+1];
  getA->vectOutP->dataF[iAna] = atan2(imag, real);

  //-----due to the FFT window, the power is spread on three bins
  double* ampD = fAna->fft->amplitude->dataD;
  double power = ampD[iLine]*ampD[iLine];
  if(iLine != 0) power += ampD[iLine-1]*ampD[iLine-1];
  if(iLine+1 < fAna->fft->amplitude->nx[0]) power += ampD[iLine+1]*ampD[iLine+1];
  getA->vectOutA->dataF[iAna] = sqrt(2.*power);

  if(iAna != nAna-1) return;

  FrProcDataNewVT(frame, getA->vectOutA, 1);
  FrProcDataNewVT(frame, getA->vectOutP, 1);
  getA->vectOutA = NULL;
  getA->vectOutP = NULL;

  return;
}
//---------------------------------------------------------------------------
void FdGetLinesProcess(FdGetLines *getL, FrameH *frame, FdFreqAna *fAna, 
			int iAna, int nAna)
//---------------------------------------------------------------------------
{
  int iLine, i;

  if(getL->vectOutF[0] == NULL) {
    for(i = 0; i<getL->nLines; i++) {
      getL->vectOutF[i]= FrVectNewTS(getL->outNameF[i], 1./fAna->period, nAna, -32);
      getL->vectOutA[i]= FrVectNewTS(getL->outNameA[i], 1./fAna->period, nAna, -32);
      if(getL->vectOutA[i] == NULL) 
	CfgMsgAddFatal("FdGetLinesProcess: malloc %d failed", nAna);
    FrVectSetUnitY(getL->vectOutF[i], "Hz");
    FrVectSetUnitY(getL->vectOutA[i], fAna->vectIn->unitY);
    getL->iFirst = getL->fMin/fAna->fft->amplitude->dx[0]-1;
    getL->iLast  = getL->fMax/fAna->fft->amplitude->dx[0];
    if(getL->iFirst < 0) getL->iFirst = 0;
    if(getL->iLast >= fAna->fft->amplitude->nx[0]) 
       getL->iLast  = fAna->fft->amplitude->nx[0];}}

  FrVect* amplitude = FrVectCopy(fAna->fft->amplitude);
  if(amplitude == NULL) CfgMsgAddFatal("FdGetLinesProcess: VectCopy failed");

  double* ampD = amplitude->dataD;
  double dnu   = amplitude->dx[0];

  for(iLine = 0; iLine < getL->nLines; iLine++) {
    int iMax = getL->iFirst;
    for(i = getL->iFirst; i < getL->iLast; i++) {
      if(ampD[i] > ampD[iMax]) iMax = i;}

    if(iMax == getL->iFirst || iMax == getL->iLast) {
      //----------- no interpolation at the edge of the frequency range
      getL->vectOutF[iLine]->dataF[iAna] = iMax*dnu;
      getL->vectOutA[iLine]->dataF[iAna] = 2.*ampD[iMax];
      ampD[iMax] = 0;} // cancel this line

    else { 
      //---- do a quadratic interpolation: y = a*f^2 + b*f +c; f = +/-/0 dnu 
      double y0 = 2.*ampD[iMax-1]*ampD[iMax-1];
      double y1 = 2.*ampD[iMax  ]*ampD[iMax];
      double y2 = 2.*ampD[iMax+1]*ampD[iMax+1];
      double c = y1;
      double b = (y2 - y0)/(2.*dnu);
      double a = (y0 + b*dnu - c)/(dnu*dnu);
      double dfMax = -b/(2*a);
      getL->vectOutA[iLine]->dataF[iAna] = sqrt(a*dfMax*dfMax + b*dfMax + c);
      getL->vectOutA[iLine]->dataF[iAna] = sqrt(y0+y1+y2);
      getL->vectOutF[iLine]->dataF[iAna] = (iMax+1)*dnu + dfMax;

      ampD[iMax-1] = 0; // cancel this line and the adjacent bins
      ampD[iMax  ] = 0; 
      ampD[iMax+1] = 0;} 

    FrProcDataNewVT(frame, getL->vectOutF[iLine], 1);
    FrProcDataNewVT(frame, getL->vectOutA[iLine], 1);
    getL->vectOutF[iLine] = NULL;
    getL->vectOutA[iLine] = NULL;}

  FrVectFree(amplitude);

  return;
}
//---------------------------------------------------------------------------
void FdGetBrmsProcess(FdGetBrms *brms, FrameH *frame, FdFreqAna *fAna, 
			int iAna, int nAna)
//---------------------------------------------------------------------------
{
  int i;

  if(brms->vectOut == NULL) {
    brms->vectOut = FrVectNewTS(brms->outName, 1./fAna->period, nAna, -32);
    if(brms->vectOut == NULL) 
	CfgMsgAddFatal("FdGetBrmsProcess: malloc %d failed", nAna);
    FrVectSetUnitY(brms->vectOut, fAna->vectIn->unitY);
    brms->iFirst = brms->fMin/fAna->fft->amplitude->dx[0]-1;
    brms->iLast  = brms->fMax/fAna->fft->amplitude->dx[0];
    if(brms->iFirst < 0) brms->iFirst = 0;
    if(brms->iLast >= fAna->fft->amplitude->nx[0]) 
       brms->iLast  = fAna->fft->amplitude->nx[0];}


  double* ampD = fAna->fft->amplitude->dataD;
  double sum  = 0;

  for(i = brms->iFirst; i < brms->iLast; i++) {
     sum  += ampD[i]*ampD[i];}

  brms->vectOut->dataF[iAna] = sqrt(sum);

  if(iAna != nAna-1) return; 

  FrProcDataNewVT(frame, brms->vectOut, 1);
  brms->vectOut = NULL;

  return;
}
//---------------------------------------------------------------------------
void FdFreqAnaProcess(FdFreqAna *fAna, FrameH *frame)
//---------------------------------------------------------------------------
{
  FrVect *data;
  FrAdcData *adc;
  FrProcData *proc;
  double slope, dt;
  int iAna, j, nData;
  char *units;
  FdGetBrms* getB;
  FdGetLineA *getA;
  FdGetLines *getL;

  FdAction* next = fAna->action->next;
  if(frame == NULL) {
    if(next != NULL) next->action(next->data, frame);
    return;}

  //-------------------------------------get channel
  slope = 1.;
  adc = FrAdcDataFind(frame, fAna->chName);
  if(adc != NULL) {
     data  = adc->data;
     slope = adc->slope;
     units = adc->units;}
  else {
     proc = FrProcDataFind(frame, fAna->chName);
     if(proc != NULL) {
       data = proc->data;
       units = data->unitY;}
     else {
       data = NULL;}}
   
  if(data == NULL) {
    if(fAna->chMissing == FR_NO) {
      CfgMsgAddWarning("GPS=%d %s start to be missing", frame->GTimeS, fAna->chName);
      fAna->chMissing = FR_YES;
      fAna->action->state = CfgServerActive;}
    if(next != NULL) next->action(next->data, frame);
    return;}

  if(fAna->chMissing == FR_YES) {
    CfgMsgAddInfo("GPS=%d %s is back",frame->GTimeS, fAna->chName);
    fAna->chMissing = FR_NO;
    fAna->action->state = CfgServerGolden;}

  //--------------------------get the number of analysis per frame
  if(fAna->period > frame->dt) CfgMsgAddFatal("FdFreqAna: requested period for %s"
	" is larger than frame duration", fAna->chName);

  int nAna = frame->dt/fAna->period;
  if(frame->dt != nAna * fAna->period) CfgMsgAddFatal(
	"FdFreqAna: period (%g) is not a multiple (%d) of the frame duration(%g)",
	fAna->period, nAna, frame->dt);

  dt = data->dx[data->nDim-1];

  if(data->nDim > 2)  CfgMsgAddFatal("FdFreqAna: too large dimension");
  if(data->nDim == 1) nData = data->nData/nAna;
  else {  //-------------------------sample data                
    nData = data->nx[1]; 
    if(nAna != data->nx[0]) {
      nAna = data->nx[0];
      fAna->period = frame->dt/nAna;
      CfgMsgAddError("FdFreqAna: period set to %g for %s",
	fAna->period, fAna->chName);}}

  //----------------------- check that the data has not changed
  if(fAna->fft != NULL) {
    if(2*fAna->fft->output->nData*nAna != data->nData) {
      CfgMsgAddWarning("FdFreqAna: nData data changed for %s at %d",
		fAna->chName, frame->GTimeS);
      fAna->fft = NULL; }}
 
  //------------------------------------------ create FFT at the first call
  if(fAna->fft == NULL) {
    fAna->fft = FrvRFFTNew("HS", nData, 1);
    fAna->fft->nAverage = 1;}

  //----------------process smaller vector if requested
  if(fAna->vectIn == NULL) {
    fAna->vectIn = FrVectNewTS(fAna->chName, 1./dt, nData, -32);
    if(fAna->vectIn == NULL) CfgMsgAddFatal("FdFreqAna: malloc %d failed", nData);}
  FrVectSetUnitY(fAna->vectIn, units);

  float* dataF = fAna->vectIn->dataF;

  for(iAna = 0; iAna < nAna; iAna++) {
    if(data->type == FR_VECT_4R) {
      for(j=0; j<nData; j++) {dataF[j] = slope*data->dataF[iAna*nData + j];}}
    else if(data->type == FR_VECT_8R) { 
      for(j=0; j<nData; j++) {dataF[j] = slope*data->dataD[iAna*nData + j];}}
    else if(data->type == FR_VECT_2S) { 
      for(j=0; j<nData; j++) {dataF[j] = slope*data->dataS[iAna*nData + j];}}
    else if(data->type == FR_VECT_4S) { 
      for(j=0; j<nData; j++) {dataF[j] = slope*data->dataI[iAna*nData + j];}}
    else {
      CfgMsgAddFatal("FdFreqAna: unsuported type for %s",fAna->chName);}

    FrvRFFTFor(fAna->fft, fAna->vectIn);
  
    for(getB = fAna->brms; getB != NULL; getB = getB->next) {
      FdGetBrmsProcess(getB, frame, fAna, iAna, nAna);}

    for(getA = fAna->lineA; getA != NULL; getA = getA->next) {
      FdGetLineAProcess(getA, frame, fAna, iAna, nAna);}

    for(getL = fAna->lines; getL != NULL; getL = getL->next) {
      FdGetLinesProcess(getL, frame, fAna, iAna, nAna);}}

  //-------------------------------------------------  call next action
  if(next != NULL) next->action(next->data, frame);

  return;
}
//---------------------------------------------------------------------------
void FdFreqAnaParserAdd(FdIO* fdIO)
//---------------------------------------------------------------------------
{
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_GET_BRMS",  FdGetBrmsNew, 
	(void*) &(fdIO->actionsIn), 4, CfgString, CfgReal, CfgReal, CfgReal);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_GET_LINE",  FdGetLineANew, 
	(void*) &(fdIO->actionsIn), 3, CfgString, CfgReal, CfgReal);
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_GET_LINES",  FdGetLinesNew, 
	(void*) &(fdIO->actionsIn), 5, CfgString, CfgReal, CfgReal, CfgReal, CfgDec);

  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_GET_BRMS",  FdGetBrmsNew, 
	(void*) &(fdIO->actionsOut), 4, CfgString, CfgReal, CfgReal, CfgReal);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_GET_LINE",  FdGetLineANew, 
	(void*) &(fdIO->actionsOut), 3, CfgString, CfgReal, CfgReal);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_GET_LINES",  FdGetLinesNew, 
	(void*) &(fdIO->actionsOut), 5, CfgString, CfgReal, CfgReal, CfgReal, CfgDec);

  return;
}
