#include "FdIO.h"
#include <errno.h>
#include <sys/stat.h>

/*-------------------------------------------------------private functions---*/

typedef struct FdAddEventsFromFile FdAddEventsFromFile;
typedef struct FdAddFromFile FdAddFromFile;
typedef struct FdAddNoInjCh  FdAddNoInjCh;
typedef struct FdIFile       FdIFile;
typedef struct FdIMFile      FdIMFile;
typedef struct FdOFile       FdOFile;
typedef struct FdOShm        FdOShm;

struct FdAddEventsFromFile
{
  char*     name;          /* input file name                               */
  char*     tag;           /* tag to select channels                        */
  int       timeOffset;    /* time offset applied when reading frame part   */
  FrFile*   iFile;         /* file pointer                                  */
  FdAction* action;        /* this is the action structure                  */
};

struct FdAddFromFile
{
  char*     name;          /* input file name                               */
  char*     tag;           /* tag to select channels                        */
  int       timeOffset;    /* time offset applied when reading frame part   */
  int       printError;    /* if non zero, skip error message about gaps    */
  FRBOOL    readOk;        /* flag telling if a warning need to be printed  */
  FrFile*   iFile;         /* file pointer                                  */
  FrameH*   longFrame;     // to store a longer frame from call to call
  FdAction* action;        /* this is the action structure                  */
};

struct FdAddNoInjCh
{
  char*     chName;        /* channel name                                  */
  double    chRate;        /* channel sampling rate                         */
  double    marginB;       /* margin before the FrSimEvent time             */
  double    marginA;       /* margin after  the FrSimEvent time             */
  char*     fileName;      /* input file name                               */
  char*     tag;           /* tag to select channels                        */
  FrFile*   iFile;         /* file pointer                                  */
  FdAction* action;        /* this is the action structure                  */
};

struct FdIFile     
{
  char*     name;          /* input file name                               */
  int       start;         /* start GPS time; 0=file start; -1= now         */
  int       length;        /* duration to be process or -1 for the full file*/
  double    requestedTime; /* GPS time of the next frame to be read         */
  int       endTime;       /* end of the frame segment to be read           */
  FrFile*   iFile;         /* file pointer                                  */
  FdAction* action;        /* this is the action structure                  */
};

struct FdIMFile     
{
  char*     name;          /* input file names                              */
  int       start;         /* start GPS time; 0=file start; -1= now         */
  int       length;        /* duration to be process or -1 for the full file*/
  int       nFiles;        /* number of file to merge                       */
  FrFile**  iFiles;        /* list of file pointer                          */
  FrameH**  frames;        /* next frame for each files                     */
  double*   requestedTime; /* GPS time of the next frame to be read         */
  int*      endTime;       /* end of the frame segment to be read           */
  FdAction* action;        /* this is the action structure                  */
};

struct FdOFile     
{
  char*     name;          /* output file name                              */
  int       fLength;       /* number of frame per output file               */
  int       dirPeriod;     /* period cover by a directory or 0 if single dir*/
  char*     tag;           /* tag for the output file                       */
  char*     nBytesName;    /* FrSerData name for recording the frame size   */
  FrFile*   oFile;         /* output file pointer                           */
  FdAction* action;        /* this is the action structure                  */
};

struct FdOShm     
{
  char*     name;          /* output file name prefix                       */
  char*     tag;           /* tag for the output file                       */
  char*     nBytesName;    /* FrSerData name for recording the frame size   */
  char*     fullName;      /* full file name                                */
  char*     tempName;      /* file name + _NOT_YET_CLOSED                   */
  char*     dirName;       /* name of the directory                         */
  int       duration;      /* shm size in secondes                          */
  int       cleanupPeriod; /* period to call the cleanup function           */
  FdAction* action;        /* this is the action structure                  */
};

void FdAddEventsFromFileProcess(FdAddEventsFromFile* f, FrameH* frame);
void FdAddFromFileProcess(FdAddFromFile* f, FrameH* frame);
void FdAddNoInjChProcess (FdAddNoInjCh* n,  FrameH* frame);
void FdIFileProcess      (FdIFile* f,       FrameH* frame);
void FdIMFileProcess     (FdIMFile* f,      FrameH* frame);
void FdOFileProcess      (FdOFile* f,       FrameH* frame);
void FdOShmProcess       (FdOShm*  s,       FrameH* frame);
FdOFile* LastOFile = NULL;

/*--------------------------------------------------------------------------*/
int FdAddEventsFromFileNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdAddEventsFromFile *f;

  f = (FdAddEventsFromFile*) calloc(1,sizeof(FdAddEventsFromFile));
  if(f == NULL) CfgMsgAddFatal("malloc FdAddEventsFromFile failed");

  f->action = FdActionNew((FdAction**) arg,  (void*) f,
                    FdAddEventsFromFileProcess, "FDIN_ADD_EVENTS_FROM_FILE");

  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&(f->name), CfgParseGetNextString(data));
  FrStrCpy(&(f->tag),  CfgParseGetNextString(data));

  if(f->tag == NULL) FrStrCpy(&(f->tag), "*");
  CfgMsgAddInfo("Add events \"%s\" from %s", f->tag, f->name);

  f->timeOffset = CfgParseGetNextDec(data);
  if(f->timeOffset != 0) CfgMsgAddInfo(" A %d s. time offset will be applied"
       " when adding eventss", f->timeOffset);

  /*------------------------------------------------------open input file---*/
  f->iFile = FrFileINew(f->name);
  if(f->iFile == NULL) CfgMsgAddFatal("Cannot open input file");

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdAddEventsFromFileProcess(FdAddEventsFromFile* f, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  if(frame != NULL) {
    double tStart = frame->GTimeS + 1.e-9*frame->GTimeN + f->timeOffset;

    //---------------------------------------------------------------add events
    FrEvent* events = FrEventReadTF(f->iFile, f->tag, tStart, frame->dt, 1, 0);
    if(events != NULL) {
      FrEvent* lastE = events;
      for(; lastE->next != NULL; lastE = lastE->next) {
          lastE->GTimeS -= f->timeOffset;}
      lastE->next = frame->event;
      frame->event = events;}

    //---------------------------------------------------------add sim events
    FrSimEvent* simEvents = FrSimEventReadT(f->iFile, f->tag, tStart, frame->dt, 
                                     -1e-30, 1e30);
    if(simEvents != NULL) {
      FrSimEvent* lastS = simEvents;
      for(; lastS->next != NULL; lastS = lastS->next) {
        if(f->timeOffset != 0) FrSimEventUpdateTime(lastS, -f->timeOffset);}
      lastS->next = frame->simEvent;
      frame->simEvent = simEvents;}
    }

  /*------------------------------------------------------call next action---*/
  FdAction* next = f->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}

/*--------------------------------------------------------------------------*/
int FdAddFromFileNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdAddFromFile *f;

  f = (FdAddFromFile*) calloc(1,sizeof(FdAddFromFile));
  if(f == NULL) CfgMsgAddFatal("malloc FdAddFromFile failed");

  f->action = FdActionNew((FdAction**) arg,  (void*) f,
                        FdAddFromFileProcess, "FDIN_ADD_FROM_FILE");

  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&(f->name), CfgParseGetNextString(data));
  FrStrCpy(&(f->tag),  CfgParseGetNextString(data));

  if(f->tag == NULL) FrStrCpy(&(f->tag), "*");
  CfgMsgAddInfo("Add frame parts \"%s\" from %s", f->tag, f->name);

  f->timeOffset = CfgParseGetNextDec(data);
  if(f->timeOffset != 0) CfgMsgAddInfo(" A %d s. time offset will be applied"
       " when adding frame parts", f->timeOffset);
  f->printError = CfgParseGetNextDec(data);

  /*------------------------------------------------------open input file---*/
  f->iFile = FrFileINew(f->name);
  if(f->iFile == NULL) CfgMsgAddFatal("Cannot open input file");

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdAddFromFileProcess(FdAddFromFile* f, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;
  FrEvent *event;
  FrSimEvent *sim;

  /*----------------------------------------first read additional channels---*/
  if(frame != NULL) {
    FrameH *part = NULL;
    double tStart = frame->GTimeS + 1.e-9*frame->GTimeN + f->timeOffset;

    //------------- reuse a previous long frame if it match the requested time
    if(f->longFrame != NULL) { 
      double tStartL = f->longFrame->GTimeS + 1.e-9*f->longFrame->GTimeN
		     - f->timeOffset;
      double tEndL   = tStartL + f->longFrame->dt;
      double tEnd    = tStart  +        frame->dt;
      if(tStart >= tStartL && tEnd <= tEndL) part = f->longFrame;
      else {
        FrameFree(f->longFrame);
        f->longFrame = NULL;}}

    if(part == NULL) part = FrameReadTChnl(f->iFile, tStart +.001, f->tag);

    if(part != NULL) {
      if(f->readOk != FR_YES) {
        f->readOk = FR_YES;
        CfgMsgAddInfo("At %d frames are again available from %s",
               frame->GTimeS, f->name);}

      //------------------------------deal with frames with different duration
      if(frame->dt > part->dt) CfgMsgAddFatal("FdAddFromFile: "
        "Frame duration missmatch: %g / %g", frame->dt, part->dt);
      else if(part->dt > frame->dt) {
        f->longFrame = part;
        double tStartL = part->GTimeS + 1.e-9*part->GTimeN;
        part = FrameCopyPart(part, tStart - tStartL, frame->dt);
        if(part == NULL) CfgMsgAddFatal("FdAddFromFileProcess: "
		"FrameCopyParttStart:%.2f/%.2f", tStart, tStartL);}

      /*--------------------------------------------------------add events---*/
      part->event = FrEventReadTF(f->iFile, f->tag, tStart, frame->dt, 1, 0);
      if(f->timeOffset != 0) {
        for(event = part->event; event != NULL; event = event->next) {
          event->GTimeS -= f->timeOffset;}}

      /*--------------------------------------------------------add events---*/
      part->simEvent = FrSimEventReadT(f->iFile, f->tag, tStart, frame->dt, 
				-1e-30, 1e30);
      if(f->timeOffset != 0) {
        for(sim = part->simEvent; sim != NULL; sim = sim->next) {
          FrSimEventUpdateTime(sim, -f->timeOffset);}}

      FrameMerge(frame, part);}

    else {
      if(f->readOk == FR_YES && f->printError == 0) {
        f->readOk = FR_NO;
        CfgMsgAddWarning("At %d: no more frames from %s",
               frame->GTimeS, f->name);}}}

  /*------------------------------------------------------call next action---*/
  next = f->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdAddNoInjChNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdAddNoInjCh *n;

  n = (FdAddNoInjCh*) calloc(1,sizeof(FdAddNoInjCh));
  if(n == NULL) CfgMsgAddFatal("malloc FdAddNoInjCh failed");

  n->action = FdActionNew((FdAction**) arg,  (void*) n,
                        FdAddNoInjChProcess, "ADD_NO_INJECTION_CHANNEL");

  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&(n->chName),   CfgParseGetNextString(data));
  n->chRate              = CfgParseGetNextReal(data);
  n->marginB             = CfgParseGetNextReal(data);
  n->marginA             = CfgParseGetNextReal(data);
  FrStrCpy(&(n->fileName), CfgParseGetNextString(data));
  FrStrCpy(&(n->tag),      CfgParseGetNextString(data));

  if(n->tag == NULL) FrStrCpy(&(n->tag), "*");
  CfgMsgAddInfo("Add channel %s to monitor the injection %s from %s (%g %g)",
	n->chName, n->tag, n->fileName, n->marginB, n->marginA);

  /*------------------------------------------------------open input file---*/
  n->iFile = FrFileINew(n->fileName);
  if(n->iFile == NULL) CfgMsgAddFatal("Cannot open %s", n->fileName);

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdAddNoInjChProcess(FdAddNoInjCh* n, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;
  FrVect *noInjV;
  FrProcData *proc;
  FrSimEvent *sim, *simAll;
  double tStart, offset, dt;
  int i, first, last;

  if(frame != NULL) {

    //----------------------------------------create the no injection channel
    noInjV = FrVectNewTS(n->chName, n->chRate, frame->dt*n->chRate, 32);
    proc = FrProcDataNewVT(frame, noInjV, 1);
    if(proc == NULL || noInjV == NULL) CfgMsgAddFatal("FdAddNoInjChProcess malloc failed");
    for(i = 0; i < noInjV->nData; i++) {noInjV->dataI[i] = 1;}
  
    //----------------------get simEvent for this frame, within the margin
    tStart = frame->GTimeS + 1.e-9*frame->GTimeN - n->marginA;
    dt     = frame->dt + n->marginB + n->marginA;
    simAll = FrSimEventReadT(n->iFile, n->tag, tStart, dt, -1e-30, 1e30);
    for(sim = simAll; sim != NULL; sim = sim->next) {
      offset = (double) sim->GTimeS - (double) frame->GTimeS + 
              ((double) sim->GTimeN - (double) frame->GTimeN) * 1.e-9;
      first = (offset - n->marginB) * n->chRate;
      last  = (offset + n->marginA) * n->chRate + 1;
      if(first < 0) first = 0;
      if(last  > noInjV->nData) last = noInjV->nData;
      for(i = first; i < last; i++) {noInjV->dataI[i] = 0;}}

   FrSimEventFree(simAll);}

  /*------------------------------------------------------call next action---*/
  next = n->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
void FdOFileCheckPath(char *name)
/*--------------------------------------------------------------------------*/
{
  char *fProbe, *p, *b, *dir, *end;
  FILE *fp;
  struct stat st;

  /*--------------------------------------if needed create the directory---*/
  FrStrCpy(&dir, name);
  end = dir + strlen(dir);
  p = dir;
  while(p < end) {
    b = strchr(p,'/');
    if(b == NULL) break;
    if(b == p) {
      p++;
      continue;}
    b[0] = '\0';
    if(stat(dir, &st) != 0) {
      CfgMsgAddInfo("Create folder %s",dir);
      if(mkdir(dir,0777) != 0) CfgMsgAddFatal(" cannot create folder");}
    b[0] = '/';
    p = b+1;}

  /*-----------------------------------------------probe the write access---*/
  fProbe = malloc(strlen(name) + 20);
  sprintf(fProbe, "%s_Fd.probe", name);
  fp = fopen(fProbe,"w");
  if(fp == NULL) 
    CfgMsgAddFatal("Could not write output file(s): %s",strerror(errno));
  fclose(fp);
  remove(fProbe);
  free(fProbe);
  free(dir);

  return;
}
/*--------------------------------------------------------------------------*/
int FdIFileNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{                                        
  FdIFile *f;
  FdIO *fdIO;

  f = (FdIFile*) calloc(1,sizeof(FdIFile));
  if(f == NULL) CfgMsgAddFatal("malloc FdIFile failed");

  f->action = FdActionNew((FdAction**) arg, 
			  (void*) f, FdIFileProcess, "FDIN_FILE");

  if(f->action != *((FdAction**) arg)) 
    CfgMsgAddFatal("FDIN_FILE must be the first input key and not %s",
                   (*((FdAction**) arg))->type);

  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&(f->name), CfgParseGetNextString(data));
  f->start  =          CfgParseGetNextDec(data);
  f->length =          CfgParseGetNextDec(data);

  CfgMsgAddInfo("Open input file:%s start GPS=%d for %d seconds",
		f->name, f->start, f->length);

  /*------------------------------------get current GPS time if requested---*/
  if(f->start < 0) {
    f->start = FrGetCurrentGPS();
    CfgMsgAddInfo(" Set start GPS time to now:%d",f->start);}

  /*------------------------------------------------------open input file---*/
  f->iFile = FrFileINew(f->name);
  if(f->iFile == NULL) CfgMsgAddFatal("Cannot open input file");
  f->iFile->compress = FdIOGetCompressIn();

  /*--------------------------------get the time of the next frame to read---*/
  if(f->start > 0) {
    FrFileISetTime(f->iFile, f->start);
    f->requestedTime = f->start;}
  else  {
    f->requestedTime = FrFileITStart(f->iFile);}

  /*-------------------------------------get the end reading time if asked---*/
  if(f->length > 0) {f->endTime = f->requestedTime + f->length;}
  else              {f->endTime = 2e9;}

  /*-reduce the input Cm timeout to avoid wasting time when reading a file---*/
  fdIO = FdIOGetRoot();
  fdIO->getFrTimeOut = 1.e-8;

  /*-----------------------------activate file reading checksum by default---*/
  f->iFile->chkSumFrFlag = FR_YES;

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdIFileProcess(FdIFile* f, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;
  double nBytes;

  /*-------------------if only a limited time was requested, are we done?---*/
  if((f->requestedTime >= f->endTime) || 
     (f->requestedTime < 0)) {
    CfgTerminate();
    return;}

  /*-----------------------no tag requested: regular sequential file read---*/
  nBytes = f->iFile->nBytes;
      
  if(f->action->tag == NULL) {
    frame = FrameRead(f->iFile);
    if(frame != NULL) f->requestedTime = frame->GTimeS + 1.e-9*frame->GTimeN 
      + 1.001*frame->dt;}

  /*------------------------------------------------------read with a tag---*/
  else {
    frame = FrameReadTChnl(f->iFile, f->requestedTime+.001, f->action->tag);
    if(frame != NULL) {
      frame->event = FrEventReadTF(f->iFile, f->action->tag, 
                                   frame->GTimeS + 1.e-9*frame->GTimeN, 
                                   frame->dt, 1, 0);
      frame->simEvent = FrSimEventReadT(f->iFile, f->action->tag, 
                                   frame->GTimeS + 1.e-9*frame->GTimeN, 
				   frame->dt, -1e-30, 1e30);}

    f->requestedTime = FrFileITNextFrame(f->iFile, f->requestedTime+0.001);}

  /*---------------------------------------------------------record info---*/
  nBytes = f->iFile->nBytes - nBytes;
  snprintf(f->action->serData, f->action->serDataSize,
	"units MB nBytesIn %g units counts", nBytes/1024./1024.);
  if(nBytes < 1024*128) {
    snprintf(f->action->userInfo, f->action->userInfoSize,
	"nBytesIn %.2f kB", nBytes/1024);}
  else { 
    snprintf(f->action->userInfo, f->action->userInfoSize,
	 "nBytes %.2f MB", nBytes/1024/1024);}

  /*---------------------terminte the job if we are at the end of the file---*/
  if (frame == NULL) CfgTerminate();

  /*------------------------------------------------check input read error---*/
  if(f->iFile->error != FR_OK) CfgMsgAddFatal("Read error for %s: %s",
	f->name, FrErrorGetHistory());

  /*------------------------------------------------------call next action---*/
  next = f->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdIMFileNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{                                        
#define MAXNAMESIZE 512
  char fileName[MAXNAMESIZE];
  FdIMFile *f;
  FdIO *fdIO;
  FILE *fp;
  char *cmd;
  int i, start;

  f = (FdIMFile*) calloc(1,sizeof(FdIMFile));
  if(f == NULL) CfgMsgAddFatal("malloc FdIMFile failed");

  f->action = FdActionNew((FdAction**) arg, 
			  (void*) f, FdIMFileProcess, "FDIN_MERGE_FILES");

  if(f->action != *((FdAction**) arg)) 
    CfgMsgAddFatal("FDIN_MERGE_FILES must be the first input key and not %s",
                   (*((FdAction**) arg))->type);

  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&(f->name), CfgParseGetNextString(data));
  f->start  =          CfgParseGetNextDec(data);
  f->length =          CfgParseGetNextDec(data);

  CfgMsgAddInfo("Open input files:%s start GPS=%d for %d seconds",
		f->name, f->start, f->length);

  /*---------------- get the number of files and allocate associated space---*/
  cmd = malloc(10+strlen(f->name));
  sprintf(cmd,"ls %s",f->name);
  if((fp = popen(cmd, "r")) == NULL) CfgMsgAddFatal("FdIFMFileNew: ls failed");

  f->nFiles = 0;
  while (fgets(fileName, MAXNAMESIZE, fp) != NULL) {f->nFiles++;}
  pclose(fp);

  CfgMsgAddInfo("There are %d input files", f->nFiles);
  if(f->nFiles == 0) CfgMsgAddFatal("FdIFMFileNew: no file to merge");
  
  f->iFiles        = calloc(f->nFiles, sizeof(FrFile*));
  f->frames        = calloc(f->nFiles, sizeof(FrameH*));
  f->requestedTime = calloc(f->nFiles, sizeof(double));
  f->endTime       = calloc(f->nFiles, sizeof(int));

  /*-------------------------------------------------------open all files---*/
  if((fp = popen(cmd, "r")) == NULL) CfgMsgAddFatal("FdIFMFileNew: ls failed");

  start = 2e9;
  for(i=0; i<f->nFiles; i++) {
    fgets(fileName, MAXNAMESIZE, fp);
    fileName[strlen(fileName)-1] = '\0'; // to remove a '\n'

    f->iFiles[i] = FrFileINew(fileName);
    if(f->iFiles[i] == NULL) CfgMsgAddFatal("Cannot open %s", fileName);
    f->iFiles[i]->compress = FdIOGetCompressIn();

    /*------------------------------get the time of the next frame to read---*/
    if(f->start > 0) {
         FrFileISetTime(f->iFiles[i], f->start);
         f->requestedTime[i] = f->start;}
    else f->requestedTime[i] = FrFileITStart(f->iFiles[i]);
    if(start > f->requestedTime[i]) start = f->requestedTime[i];

    CfgMsgAddInfo(" Open %s starting at %.0f", fileName, f->requestedTime[i]);}
 
  /*----------------------------------------------set the end reading time---*/
  for(i=0; i<f->nFiles; i++) {
    if(f->length > 0) f->endTime[i] = start + f->length;
    else              f->endTime[i] = 2e9;}

  CfgMsgAddInfo(" First frame to read is at %d", start);

  /*-reduce the input Cm timeout to avoid wasting time when reading a file---*/
  fdIO = FdIOGetRoot();
  fdIO->getFrTimeOut = 1.e-8;

  free(cmd);

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdIMFileProcess(FdIMFile* f, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;
  int i, nParts;
  double gps;

  /*--------------------first read all possible new frames from each files---*/
  for(i = 0; i<f->nFiles; i++) {
    if(f->frames[i] != NULL) continue;
    if(f->iFiles[i]  == NULL) continue;
    
    /*------------------if only a limited time was requested, are we done?---*/
    if((f->requestedTime[i] >= f->endTime[i]) || 
       (f->requestedTime[i] < 0)) {
      FrFileIEnd(f->iFiles[i]);
      f->iFiles[i] = NULL;
      continue;}

    /*----------------------no tag requested: regular sequential file read---*/
    if(f->action->tag == NULL) {
      f->frames[i] = FrameRead(f->iFiles[i]);
      if(f->frames[i] == NULL) {
        FrFileIEnd(f->iFiles[i]);
        f->iFiles[i] = NULL;
        continue;}
      f->requestedTime[i] = f->frames[i]->GTimeS + 1.e-9*f->frames[i]->GTimeN 
                    + 1.001*f->frames[i]->dt;}

    /*-----------------------------------------------------read with a tag---*/
    else {
      f->frames[i] = FrameReadTChnl(f->iFiles[i], f->requestedTime[i]+.001, 
					f->action->tag);
      if(f->frames[i] == NULL) {
        FrFileIEnd(f->iFiles[i]);
        f->iFiles[i] = NULL;
        continue;}

      gps = f->frames[i]->GTimeS + 1.e-9*f->frames[i]->GTimeN;
      f->frames[i]->event    = FrEventReadTF  (f->iFiles[i], f->action->tag, 
                                   gps, f->frames[i]->dt, 1, 0);
      f->frames[i]->simEvent = FrSimEventReadT(f->iFiles[i], f->action->tag, 
                                   gps, f->frames[i]->dt, -1e-30, 1e30);
      f->requestedTime[i] = FrFileITNextFrame(f->iFiles[i], 
                                f->requestedTime[i]+0.001);}}

  /*-------------------------------------------------- get the first frame---*/
  frame = NULL;
  for(i = 0; i<f->nFiles; i++) {
    if(f->frames[i] == NULL) continue;
    if(frame        == NULL) frame = f->frames[i];
    else if(frame->GTimeS > f->frames[i]->GTimeS) frame = f->frames[i];}

  /*-------------------------------merge all frames with the same GPS time---*/
  nParts = 0;
  for(i = 0; i<f->nFiles; i++) {
    if(f->frames[i] == NULL) continue;
    if(frame         == f->frames[i]) {
      f->frames[i] = NULL;
      continue;}
    if(frame->GTimeS != f->frames[i]->GTimeS) continue;
    if(frame->dt     != f->frames[i]->dt) 
	CfgMsgAddFatal("FdIMFileProcess: frame dt missmatch");
    FrameMerge(frame, f->frames[i]);
    f->frames[i] = NULL;
    nParts++;}

  /*-----------------------------------------------------------record info---*/
  snprintf(f->action->userInfo, f->action->userInfoSize, 
	"number of input streams: %d", nParts);

  /*-------------------terminate the job if we are at the end of the files---*/
  if (frame == NULL) CfgTerminate();

  /*------------------------------------------------------call next action---*/
  next = f->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdSetLocalCopyParameters(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{                                        
  if(*((FdAction**) arg) != NULL) 
    CfgMsgAddFatal("FDSET_LOCAL_COPY_PARAM must before any FDIN key");

  char* tag = strdup(CfgParseGetNextString(data));
  char* cmd = strdup(CfgParseGetNextString(data));
  FrSetLocalCopyParameters(tag, cmd);

  CfgMsgAddInfo("Set local copy parameters: tag=%s command=%s", tag, cmd);

  free(tag);
  free(cmd);

  return(CFG_OK);
}
/*--------------------------------------------------------------------------*/
int FdOFileChekSum(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{ /*--Define checksums computation; both checksums are computed if flag = 3 */
  int flag;
  FdOFile *f;

  if(LastOFile == NULL) 
    CfgMsgAddFatal("no FDOUT_FILE defined when using FDOUT_FILE_CHECKSUM");

  flag   =  CfgParseGetNextDec(data);

  f = LastOFile;
  if((flag & 1) == 1) {
    f->oFile->chkSumFiFlag = FR_YES;
    CfgMsgAddInfo("File  checksums will be computed for %s\n", f->name);}
  else {
    f->oFile->chkSumFiFlag = FR_NO;
    CfgMsgAddInfo("File  checksums will NOT be computed for %s\n", f->name);}

  if((flag & 2) == 2) {
    f->oFile->chkSumFrFlag = FR_YES;
    CfgMsgAddInfo("Frame checksums will be computed for %s\n", f->name);}
  else {
    f->oFile->chkSumFrFlag = FR_NO;
    CfgMsgAddInfo("Frame checksums will NOT be computed for %s\n", f->name);}

  return(CFG_OK);
}
/*--------------------------------------------------------------------------*/
int FdOFileNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{                                               /* Create a new output file */
  FdOFile *f;
  char *prefix, *cfgName;
  int maxLength;
  double fLength;
 
  f = (FdOFile*) calloc(1,sizeof(FdOFile));
  if(f == NULL) CfgMsgAddFatal("malloc FdOFile failed");
  LastOFile = f;

  f->action = FdActionNew((FdAction**) arg, 
			  (void*) f, FdOFileProcess, "OFile");

  FrStrCpy(&(f->name), CfgParseGetNextString(data));
  fLength           =  CfgParseGetNextReal(data);
  FrStrCpy(&(f->tag),  CfgParseGetNextString(data));
  f->dirPeriod      =  CfgParseGetNextDec(data);

  if((cfgName = strstr(f->name, "CFGNAME")) != NULL) {
    int l1 = cfgName - f->name;
    int l2 = strlen(CfgGetCmName());
    int l3 = strlen(f->name) - l1 - 7;
    char *fullName = malloc(l1 + l2 + l3 + 1);
    snprintf(fullName, l1+1, f->name);
    sprintf(fullName + l1, CfgGetCmName());
    if(l3 > 0) sprintf(fullName + l1 + l2, f->name + l1 + 7);
    CfgMsgAddInfo("Output file %s expanded to %s", f->name, fullName);
    free(f->name);
    f->name = fullName;}

  f->fLength = fLength;
  if(fLength != f->fLength) CfgMsgAddFatal(
	"Output file length for %s is not an interger:%g", f->name, fLength);

  maxLength = 10*365*24*3600; /* file duration is no more that 10 years*/
  if(f->fLength > maxLength) f->fLength = maxLength;

  if(f->tag != NULL && (strcmp(f->tag,"*") == 0)) {
    free(f->tag);
    f->tag = NULL;}
 
  /*-----------------------------------------------------------------------*/
  prefix = f->name;
  while(strstr(prefix,"/") != NULL) {prefix = strstr(prefix,"/")+1;}
  if(strlen(prefix) == 0) 
    CfgMsgAddFatal("Could not extract a prefix for the file name");
  f->nBytesName = malloc(20 + strlen(prefix));
  sprintf(f->nBytesName, "nBytes%s", prefix);
  CfgMsgAddInfo("  file prefix: %s  Record output frame size as %s", 
                prefix, f->nBytesName);

  /*-----------------------------------------------------------------------*/
  if(f->fLength <= 0) { 
    CfgMsgAddInfo("Open output file:%s; tag=%s", f->name, f->tag);
    f->oFile = FrFileONewH(f->name, -1, CfgGetCmName());}
  else {
    CfgMsgAddInfo("Open output file:%s-GPS with %d seconds per file; tag=%s",
		f->name, f->fLength, f->tag);
    if(f->dirPeriod == 0)  {
      f->oFile = FrFileONewM(f->name, -1, CfgGetCmName(), f->fLength);}

    else { 
      CfgMsgAddInfo(" new directory will be created every %d seconds",
		  f->dirPeriod);
      if(f->dirPeriod < f->fLength) 
        CfgMsgAddFatal("Requested directory spam smaller than file length");
      f->oFile = FrFileONewMD(f->name, -1, CfgGetCmName(), f->fLength,
			    prefix, f->dirPeriod);}}

  if(f->oFile == NULL) CfgMsgAddFatal("Cannot open output file");

  /*--------------------------------check write acces in output directory---*/
  FdOFileCheckPath(f->name);

  return(CFG_OK);
}
/*--------------------------------------------------------------------------*/
int FdOFileObsolete(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{                                             
  CfgMsgAddFatal("The key FDOUT_SHORT_AND_LONG_FILES is not any more "
		 "available Use the FDOUT_FILE keys");

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdOFileProcessOne(FdOFile *f,
		       FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  double size;

  if(f->tag != NULL) FrameTag(frame,f->tag);

  size = f->oFile->nBytes;

  FrameWrite(frame, f->oFile);

  if(f->oFile->error != FR_OK) { 
    CfgMsgAddError("%d: frame write error on %s..", frame->GTimeS, f->name);
    f->action->state = CfgServerActive;}
  else {
    f->action->state = CfgServerGolden;}

  if(f->tag != NULL) FrameUntag(frame);

  /*----be careful: when a new file is started, nBytes is reset to zero..---*/
  if(f->oFile->nFrames > 1) size = f->oFile->nBytes - size;
  else                      size = f->oFile->nBytes; 
  size = size/1024;
  if(f->action->state != CfgServerGolden)
       snprintf(f->action->userInfo,f->action->userInfoSize,"Write error");
  else if(size < 100) 
       snprintf(f->action->userInfo,f->action->userInfoSize,"Write %.2f kB",size);
  else snprintf(f->action->userInfo,f->action->userInfoSize,"Write %.2f MB",size/1024);

  snprintf(f->action->serData, f->action->serDataSize,
          "units MB %s %.3f units counts", f->nBytesName, size/1024);

  return;
}
/*---------------------------------------------------------------------------*/
void FdOFileProcess(FdOFile *f,
		    FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;
  long nBytes;
  char *fName;

  if(frame != NULL) FdOFileProcessOne(f, frame);

  /*--------------------------frame == NULL means we are closing the files---*/
  else if(f->oFile == NULL) {
    if(f->action->terminated == FR_NO) 
        CfgMsgAddInfo("No frame written on output file %s",f->name);
    f->action->terminated = FR_YES;}

  else if(f->oFile->frfd != NULL) {
    FrStrCpy(&fName, f->oFile->current->fileName);
    nBytes = FrFileOEnd(f->oFile);
    if(nBytes == 0) 
      CfgMsgAddError("Closing file error on %s ", fName);
    else
      CfgMsgAddInfo("Closing %s: %ld bytes written", fName, nBytes);
    free(fName);
    f->oFile = NULL;
    f->action->terminated = FR_YES;}

  /*-----------------------------------------------------call next actions---*/
  next = f->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*----------------------------------------------------------------FdClnDir---*/
void FdOShmClnDir(FdOShm *s, int gpsMin)
/*---------------------------------------------------------------------------*/
/* This function remove old files in a directory */
/*---------------------------------------------------------------------------*/
{
  DIR  *dirp;
  struct dirent *entry;
  int  gps, nameLen, i;

  /*-------------------------------------------------------------------------*/
  dirp  = opendir(s->dirName);
  if(dirp == NULL) CfgMsgAddFatal("FdClnDIr:Could not open folder %s",s->dirName);

  /*---------------do a first scan of the directory to get file statistics---*/
  while((entry = readdir(dirp)) != NULL) {

    /*---------------------------------first check that it is a frame file---*/
    nameLen = strlen(entry->d_name);
    if(strcmp(entry->d_name + nameLen - 4, ".gwf") != 0) continue;

    /*-------------------------------------get file duration from its name---*/
    for(i=nameLen-4; i>0; i--) {if(entry->d_name[i] == '-') break;}
    if(i == 0) continue;

    /*----------------------------------------get file start from its name---*/
    for(i--; i>0; i--) {if(entry->d_name[i] == '-') break;}
    if(i == 0) continue;
    sscanf(entry->d_name + i + 1, "%d", &gps);

    /*------------------remove file if too old---*/
    if(gps < gpsMin) {
      sprintf(s->fullName,"%s/%s",s->dirName,entry->d_name);
      remove(s->fullName);}
 } 

  closedir(dirp);

  return;
}
/*--------------------------------------------------------------------------*/
int FdOShmNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{                                               /* Create a new output file */
  FdOShm *s;
  char *prefix, *p;
 
  s = (FdOShm*) calloc(1,sizeof(FdOShm));
  if(s == NULL) CfgMsgAddFatal("malloc FdOShm failed");

  s->action = FdActionNew((FdAction**) arg, 
			  (void*) s, FdOShmProcess, "OShm");

  FrStrCpy(&(s->name), CfgParseGetNextString(data));
  FrStrCpy(&(s->tag),  CfgParseGetNextString(data));
  s->duration      = CfgParseGetNextDec(data);
  s->cleanupPeriod = CfgParseGetNextDec(data);

  if(s->cleanupPeriod == 0) s->cleanupPeriod = 1;
  FRBOOL cleanDir = FR_NO;
  if(s->cleanupPeriod < 0) {
    cleanDir = FR_YES;
    s->cleanupPeriod  *= -1;}

  CfgMsgAddInfo("Output shm:%s; tag=%s duration: %ds cleanup period: %ds", 
                 s->name, s->tag, s->duration, s->cleanupPeriod);

  if(strstr(s->name,"$AUTO_PREFIX$") != NULL) {
    s->name = realloc(s->name, strlen(s->name)+2*strlen(CfgGetCmName())+1);
    p = strstr(s->name,"$AUTO_PREFIX$");
    sprintf(p,"%s/%s", CfgGetCmName(), CfgGetCmName());
    CfgMsgAddInfo(" AUTO_PREFIX used, path expanded to %s",s->name);}

  if(strcmp(s->tag,"*") == 0) {
    free(s->tag);
    s->tag = NULL;}
 
  /*-----------------------------------------------------------------------*/
  if(strstr(s->name,"/") == NULL) 
    CfgMsgAddFatal("FdOShmNew: output name path does not contain a /");

  prefix = s->name;
  while(strstr(prefix,"/") != NULL) {prefix = strstr(prefix,"/")+1;}
  if(strlen(prefix) == 0) 
    CfgMsgAddFatal("Could not extract a prefix for the file name");
  s->nBytesName = malloc(20 + strlen(prefix));
  sprintf(s->nBytesName, "nBytes%s", prefix);
  FrStrCpy(&(s->dirName), s->name);
  s->dirName[prefix-s->name]='\0';
  CfgMsgAddInfo("  dirName: %s file prefix: %s  Record output frame size as %s", 
                s->dirName, prefix, s->nBytesName);

  /*-----------------------------------------------------------------------*/
  FdOFileCheckPath(s->name);

  //-------------------------------------------clean up directory if requested
  if(cleanDir == FR_YES) {
    CfgMsgAddInfo(" the third parameter asked for a directory cleanup");
    DIR* dirp  = opendir(s->dirName);
    if(dirp == NULL) CfgMsgAddFatal("Could not open dir. %s ", s->dirName);

    struct dirent* entry;
    while((entry = readdir(dirp)) != NULL) {
      int len = strlen(entry->d_name);
      if(entry->d_name[len-1] == '.') continue;
      char* fileName = malloc(strlen(s->dirName) + strlen(entry->d_name) + 2);
      sprintf(fileName,"%s/%s", s->dirName, entry->d_name);
      CfgMsgAddInfo(" remove %s ",fileName);
      remove(fileName);
      free(fileName);}

    closedir(dirp);}

  //-------------------------------------------------------------------------
  s->fullName = malloc(strlen(s->name)+40);
  s->tempName = malloc(strlen(s->name)+60);

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdOShmProcessOne(FdOShm *s,
	 	      FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  double size;
  FrFile *oFile;
  int index, nFramesPerSec;
 
  if(s->tag != NULL) FrameTag(frame,s->tag);

  /*------------------------------------------------------build file name---*/
  if(frame->dt >= 1) 
    sprintf(s->fullName,"%s-%d-%d.gwf",s->name, frame->GTimeS, (int) frame->dt);
  else {
    nFramesPerSec = 1;
    if     (frame->dt == 0.50) nFramesPerSec = 2;
    else if(frame->dt == 0.25) nFramesPerSec = 4; 
    else if(frame->dt == 0.20) nFramesPerSec = 5; 
    else if(frame->dt == 0.10) nFramesPerSec = 10; 
    else if(frame->dt == 0.05) nFramesPerSec = 20; 
    else CfgMsgAddFatal("FdOShm: unsupported frame length:%g",frame->dt);
    index = (frame->GTimeN*1.e-9)/frame->dt;
    sprintf(s->fullName,"%s-%d%c-%c.gwf",s->name, frame->GTimeS, 
	index+97, nFramesPerSec+96);}

  sprintf(s->tempName,"%s_NOT_YET_CLOSED",s->fullName);

  oFile = FrFileONew(s->tempName, -1);
  if(oFile == NULL) CfgMsgAddError("Could not open file:%s",s->tempName);

  if(oFile->historyMsg != NULL) free(oFile->historyMsg);
  oFile->historyMsg = malloc(strlen(CfgGetCmName()) + 16);
  if(oFile->historyMsg == NULL) CfgMsgAddFatal("Malloc history failed");
  sprintf(oFile->historyMsg, "FdIOServer:%s",CfgGetCmName());

  oFile->chkSumFiFlag = FR_NO; /*-- disable checksums when writing in shm---*/
  oFile->chkSumFrFlag = FR_NO;

  /*----------------------------------------------------------write frame---*/
  FrameWrite(frame, oFile);

  if(oFile->error != FR_OK) { 
    CfgMsgAddError("%d: frame write error on %s..", frame->GTimeS, s->name);
    s->action->state = CfgServerActive;}
  else {
    s->action->state = CfgServerGolden;}

  if(s->tag != NULL) FrameUntag(frame);

  size = FrFileOEnd(oFile)/1024; 
  rename(s->tempName, s->fullName);

  if(s->action->state != CfgServerGolden)
       snprintf(s->action->userInfo,s->action->userInfoSize,"Write error");
  else if(size < 100) 
       snprintf(s->action->userInfo,s->action->userInfoSize,"Write %.2f kB",size);
  else snprintf(s->action->userInfo,s->action->userInfoSize,"Write %.2f MB",size/1024);

  snprintf(s->action->serData, s->action->serDataSize,
               "units MB %s %.3f units counts", s->nBytesName, size/1024);

  if(frame->GTimeN == 0 && frame->GTimeS % s->cleanupPeriod == 0)
    FdOShmClnDir(s, frame->GTimeS - s->duration);

  return;
}
/*---------------------------------------------------------------------------*/
void FdOShmProcess(FdOShm *s,
		   FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdOShmProcessOne(s, frame);

  /*-----------------------------------------------------call next actions---*/
  next = s->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*---------------------------------------------------------------------------*/
void FdFileParserAdd(FdIO* fdIO)
/*---------------------------------------------------------------------------*/
{
 
  CfgParseGetFunctionAdd(fdIO->parser,"FDIN_ADD_EVENTS_FROM_FILE",
                         FdAddEventsFromFileNew, (void *) &(fdIO->actionsIn),
                         3, CfgString, CfgString, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser,"FDIN_ADD_FROM_FILE",
                         FdAddFromFileNew, (void *) &(fdIO->actionsIn),
                         4, CfgString, CfgString, CfgDec, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser,"FDIN_ADD_NO_INJECTION_CHANNEL",
                         FdAddNoInjChNew, (void *) &(fdIO->actionsIn),
                         6, CfgString, CfgReal, CfgReal, CfgReal, CfgString, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser,"FDIN_FILE",
                         FdIFileNew, (void *) &(fdIO->actionsIn),
                         3, CfgString, CfgDec,  CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser,"FDIN_MERGE_FILES",
                         FdIMFileNew, (void *) &(fdIO->actionsIn),
                         3, CfgString, CfgDec,  CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FD_SET_LOCAL_COPY_PARAMETERS",
                         FdSetLocalCopyParameters, (void *) &(fdIO->actionsIn),
                         2, CfgString, CfgString);

  CfgParseGetFunctionAdd(fdIO->parser,"FDOUT_FILE",
                         FdOFileNew, (void *) &(fdIO->actionsOut),
                         4, CfgString, CfgReal,  CfgString,  CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser,"FDOUT_SHORT_AND_LONG_FILES", 
                         FdOFileObsolete, NULL, 
                         7, CfgString, CfgString, CfgString, CfgDec, CfgDec,
                         CfgString, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser,"FDOUT_FILE_CHECKSUM",
                         FdOFileChekSum, NULL, 1, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser,"FDOUT_SHM", 
                         FdOShmNew, (void *) &(fdIO->actionsOut), 
                         4, CfgString, CfgString, CfgDec, CfgDec);

  return;
}
