#include <FdIO.h>
#include "FdCm.h"

typedef struct FdToAcl FdToAcl;

struct FdToAcl             // to send one data via Cm message to an Acl process
{
  char*      aclSrvName;   // Acl server name
  char*      aclChName;    // name seen by Acl
  char*      fdChName;     // channel name in the frame
  char*      loadOrApply;  // last parameter of the Cm message sent to Acl
  FRBOOL     errorPrinted; // tell if the error message has already been printed
  //------------------------- end of input parameters                       
  int        nTry;         // current number of send try                    
  double     lastValue;    // last received value
  double     chPeriod;     // channel (frame) period
  FdAction*  action;       // action for this structure      
};

void FdToAclProcess(FdToAcl *toAcl, FrameH *frame);

/*----------------------------------------------------------------------------*/
int FdToAclNew(void *arg, CfgDataType_t *data )
/*----------------------------------------------------------------------------*/
{
  FdToAcl* toAcl = calloc(1, sizeof(FdToAcl));

  FrStrCpy(&toAcl->aclSrvName, CfgParseGetNextString(data));
  FrStrCpy(&toAcl->aclChName,  CfgParseGetNextString(data));
  FrStrCpy(&toAcl->fdChName,   CfgParseGetNextString(data));
  FrStrCpy(&toAcl->loadOrApply,CfgParseGetNextString(data));

  CfgMsgAddInfo("Will send to Acl %s as %s the mean value of %s (%s)", 
     toAcl->aclSrvName, toAcl->aclChName, toAcl->fdChName, toAcl->loadOrApply);

  FdAction** root = arg;

  toAcl->action = FdActionNew(root, (void*) toAcl, FdToAclProcess, "FdToAcl");

  return(CFG_OK);
}
/*----------------------------------------------------------------------------*/
void FdToAclProcessOne(FdToAcl *toAcl, FrameH *frame)
/*----------------------------------------------------------------------------*/
/* This function send a message to an Acl process                             */
/*----------------------------------------------------------------------------*/
{
  long status;
  FrAdcData *adc;
  CmMessage  message; 

  /*---------------------------------------------create the output message---*/
  message = CmMessageNew();
  if(message == NULL) CfgMsgAddFatal("malloc CmMessage failed"); 

  CmMessageSetType(message, "AcConstChSet");

  /*----------------------------------------------------------get adc value---*/
  FrProcData* proc = FrProcDataFind (frame, toAcl->fdChName);
  if(proc != NULL) {
    toAcl->lastValue = FrVectMean(proc->data);
    toAcl->chPeriod  = proc->data->dx[0];
    if(toAcl->errorPrinted == FR_YES) {
      CfgMsgAddInfo("FdToAcl: %s start to be back at    %d",
                toAcl->fdChName, frame->GTimeS);
      toAcl->errorPrinted = FR_NO;}}
  else {
    adc = FrAdcDataFind (frame, toAcl->fdChName);
    if(adc != NULL) {
      toAcl->lastValue = adc->slope * FrVectMean(adc->data) + adc->bias;
      toAcl->chPeriod  = adc->data->dx[0];
      if(toAcl->errorPrinted == FR_YES) {
        CfgMsgAddInfo("FdToAcl: %s start to be back at    %d",
                  toAcl->fdChName, frame->GTimeS);
        toAcl->errorPrinted = FR_NO;}}
    else {
      if(toAcl->errorPrinted == FR_NO && toAcl->chPeriod <= 1) 
        CfgMsgAddError("FdToAcl: %s start to be missing at %d",
                       toAcl->fdChName, frame->GTimeS);
      toAcl->errorPrinted = FR_YES;}}

  /*------------------------------------------------- build the Cm message ---*/
  CmMessagePutText  (message, toAcl->aclChName);
  CmMessagePutDouble(message, toAcl->lastValue);
  CmMessagePutDouble(message, 0.);
  CmMessagePutText  (message, toAcl->loadOrApply);

  /*---------------------------------------regular send if no queue is used---*/
  status = CmMessageSend(message, toAcl->aclSrvName);
  if(status == 0) {
    if(toAcl->nTry == 0) CfgMsgAddError("%d: could not send data to %s",
		       frame->GTimeS, toAcl->aclSrvName);
      toAcl->nTry++;}
  else if(toAcl->nTry > 0) {
    CfgMsgAddInfo("%d: could send again data to %s",
		      frame->GTimeS, toAcl->aclSrvName);
    toAcl->nTry = 0;}

  CmMessageDelete (message);

  /*-----------------------------------------------print info if using post---*/
  snprintf(toAcl->action->userInfo, toAcl->action->userInfoSize, 
	  "(send %s=%g to_%s)",
	  toAcl->aclChName, toAcl->lastValue, toAcl->aclSrvName);

  return;
}
/*----------------------------------------------------------------------------*/
void FdToAclProcess(FdToAcl *toAcl, FrameH *frame)
/*----------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdToAclProcessOne(toAcl, frame);

  /*---------------------------------------------call the next ouput action---*/
  next = toAcl->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*----------------------------------------------------------------------------*/
void FdToAclParserAdd(FdIO* fdIO)
/*----------------------------------------------------------------------------*/
{
  CfgParseGetFunctionAdd(fdIO->parser,"FDOUT_TO_ACL",
			 FdToAclNew, (void *) &(fdIO->actionsOut),
			 4, CfgString, CfgString, CfgString, CfgString);

  return;
}
