
/**************************************************************/
/*                                                            */
/*                    FRAMES STATISTICS                       */
/*                                                            */
/* MODULE  : FdStat.c                                         */
/* AUTHOR  : B. Mours, D.Verkindt, Virgo-LAPP                 */
/* VERSION : v5r5p1                                           */
/* DATE    : 25 May 2009                                      */
/*                                                            */
/*  This program received frames and compute some statistics  */
/*  about the channels they contain.                          */
/*    Syntax:  FdStat  Target_CmName                          */
/*    Type just FdStat to get help.                           */
/*                                                            */
/************************ FdStat.c ****************************/

#include <math.h>
#include <signal.h>
#include "FdUtil.h"

struct FdsParam{
   char *myName;        /* Cm Name of this application                      */
   char *srcName;       /* Cm Name of the Frame source sending frames here  */
   char *tag;           /* tag to select frame's channels to receive        */
   long resetPeriod;    /* period of statistics reset (in nbr of frames)    */
   long refreshPeriod;  /* period of statistics print (in nbr of frames)    */
   long nFrames;        /* current frame index for nFrameTot                */
   long level;          /* level of stat display                            */
};
struct FdsParam Fds;

struct FdsStat{
  char  *name;
  double min;
  double max;
  double mean;
  struct FdsStat *next;
};
struct FdsStat *SFirst = NULL;


/*---------------------------------------------------------- FdsStatUpdate ---*/
void FdsStatUpdate (char* type, FrVect *vect)
/*---------------------------------------------------------- FdsStatUpdate ---*/
{
  struct FdsStat *sum;
  double min, mean, max, nAv, rate;

  if (vect == NULL) return;
  if (vect->compress != 0) FrVectExpand(vect);

  /*------------------------------------------------find the statistic bloc---*/
  for(sum  = SFirst; sum != NULL; sum = sum->next) {
    if (strcmp(sum->name, vect->name) == 0) break;}

  /*--------------------------------------if not existing, create a new one---*/
  if(sum == NULL) {
    sum = calloc(sizeof(struct FdsStat), 1); 
    FrStrCpy(&(sum->name), vect->name);}

  /*----------------------------------------------get min, max, mean values---*/
  mean = FrVectMean(vect);
  FrVectMinMax(vect, &min, &max);

  /*-------------------------reset values at the begining or after a period---*/
  if(Fds.nFrames%Fds.resetPeriod == 0) {
    sum->mean = 0;
    sum->min = min;  
    sum->max = max;}
   
  /*--------------------------------------------------------accumulate data---*/
  sum->mean += mean; 
  if(min < sum->min) sum->min = min;
  if(max > sum->max) sum->max = max;

  /*----------------------------- print statistic if it is the rigth time---*/
  if (Fds.nFrames%Fds.refreshPeriod == 0)  {
    nAv = ((Fds.nFrames-1)%Fds.resetPeriod) + 1;
    mean  = sum->mean  / nAv;
    if(vect->dx[0] != 0) rate = 1./vect->dx[0];
    else                 rate = -1.;
    printf(" %-40s nData=%7ld min= %9.3g  mean=%9.3g  max=%9.3g rate=%7gHz\n",
	   sum->name, vect->nData, sum->min, mean, sum->max, rate);}

  return;}

/*----------------------------------------------------------- FdsFrameStat ---*/
void FdsFrameStat (struct FrameH *frame, int nBytes)
/*----------------------------------------------------------- FdsFrameStat ---*/
{
  int nAdc, nSer, nProc, nAdcTot;
  char date_heure[24];
  FrSerData *ser;
  FrAdcData *adc;
  FrProcData *proc;
  time_t tt;

  Fds.nFrames++;

  tt = time(NULL);
  strftime(date_heure,24," %d %b %Y (%T)",localtime(&tt));
  printf("%5ld frames received; last: gps %d (%.3fMB) at %s\n", Fds.nFrames, 
	 frame->GTimeS, (double)nBytes/(1024.*1024.),  date_heure);

  if (Fds.level > 2) {
    printf("------------------------ Start Frame Dump ---------------------\n");
    FrameDump (frame,stdout,Fds.level-2);}


  if(Fds.nFrames%Fds.refreshPeriod == 0) {
    printf("====GTimeS: %d ================================================="
           "============================================\n", frame->GTimeS);}

  /*------------------------------------------------------update statistics---*/
  nProc = 0;
  nAdc  = 0;
  nSer  = 0;
  nAdcTot = 0;

  for (proc = frame->procData; proc != NULL; proc = proc->next) {
    FdsStatUpdate("PROC:",proc->data);
    nProc++;}

  if(frame->rawData != NULL) {
    for(adc = frame->rawData->firstAdc; adc != NULL; adc = adc->next) {
      nAdc++;}

    for(ser = frame->rawData->firstSer; ser != NULL; ser = ser->next) {
      nSer++;}

    if(nSer > 0) FrameReshapeConvertSer(frame, frame, frame->dt);

    for(adc = frame->rawData->firstAdc; adc != NULL; adc = adc->next) {
      FdsStatUpdate("ADC ",adc->data);
      nAdcTot++;}}

  /*----------------------------------------------------------------summary---*/
  if(Fds.nFrames%Fds.refreshPeriod == 0) {
    printf("====GTimeS: %d nProc=%5d nAdc=%5d nSer=%5d (%d channels) "
           "nAverage=%5ld =======\n", frame->GTimeS, 
           nProc, nAdc, nSer, nAdcTot-nAdc, 
           ((Fds.nFrames-1)%Fds.resetPeriod) + 1);}

  return;
}

/*----------------------------------------------------------------------------*/
CmMessageStatus FrameHandler (CmMessage message, char* sender, char* serverName)
/*----------------------------------------------------------------------------*/
{
  FrameH *frame;
  int nBytes;

  frame = FdGetFrameFromMessage (message, -1, &nBytes);

  if(frame == NULL) exit(0);

  FdsFrameStat (frame, nBytes);

  FrameFree(frame);

  return(CmMessageOk);
}
/*------------------------------------------------------------ quitHandler ---*/
void QuitHandler (int sig)
/*------------------------------------------------------------ quitHandler ---*/
{

  FdRemoveFramesRequest(Fds.srcName, Fds.myName);

  printf(" Statistics done over %ld frames.\n", Fds.nFrames);

  exit(sig);
}
/*----------------------------------------------------------------- FdsIni ---*/
void FdsIni (int argc, char *argv[])
/*----------------------------------------------------------------- FdsIni ---*/
{
  int i;

  if (argc == 1) {
    printf("\n");
    printf("--------------------------------------------------------------------------- \n");
    printf("  This utility gets frames through ethernet from a frame provider           \n");
    printf("  and dump information and statistics about the frames received.            \n");
    printf("--------------------------------------------------------------------------- \n");
    printf("  Syntax:  FdStat  target_CmName                                            \n\n");
    printf("  Putting \042-\042 as first letter of target_CmName names FdStat itself    \n");
    printf("                                                                            \n");
    printf("  Optional arguments:                                                       \n");
    printf("      -t <list of channels>                        (default=*=all)          \n");
    printf("         (list of channels = channels selection or keywords ADC, PROC or SMS) \n");
    printf("      -p <period of statistics display refresh>    (default=1)              \n");
    printf("      -r <period of statistics reset>              (default=-1=never)       \n");
    printf("      -d <level of stat>                           (default=2)              \n"); 
    printf("         (0=only_frame_header  1=list_of_channels  2=stat  above_2=stat+framedump(level-2) \n"); 
    printf("                                                                            \n");
    printf("  Example:  FdStat FbmMainUsers -t \"Bs_* Pr_*\" -p 2 -r -1                 \n");
    printf("  FdStat connects to the server FbmMainUsers and gets only the channels     \n");
    printf("  whose name begins with Bs or Pr, statistics display is refreshed every    \n");
    printf("  2 frames and the statistics are never reset (r=-1).                       \n");
    printf("--------------------------------------------------------------------------- \n\n");
    exit(0);
  }

  Fds.refreshPeriod =  1;
  Fds.resetPeriod   =  1;
  Fds.level         =  2;
  FrStrCpy (&(Fds.tag), "*");

  FrStrCpy (&(Fds.srcName), argv[1]);

  for (i=2; i<argc; i++) {
    if (strncmp(argv[i],"-t",2) == 0) {
      FrStrCpy (&(Fds.tag), argv[i+1]);
      i++;}
    if (strncmp(argv[i],"-p",2) == 0) {
      if (i+1< argc) Fds.refreshPeriod = atoi(argv[i+1]);
      i++;}
    else if (strncmp(argv[i],"-r",2) == 0) {
      if (i+1< argc) Fds.resetPeriod = atoi(argv[i+1]);
      i++;}
    else if (strncmp(argv[i],"-d",2) == 0) {
      if (i+1< argc) Fds.level = atoi(argv[i+1]);
      i++;}
  }

  if (Fds.resetPeriod < Fds.refreshPeriod) Fds.resetPeriod = Fds.refreshPeriod;

  printf("  refresh period=%ld   reset period=%ld  level=%ld  \n",
	 Fds.refreshPeriod, Fds.resetPeriod, Fds.level);

  return;
}

/*------------------------------------------------------------------- main ---*/
int main(int argc, char *argv[])  
/*------------------------------------------------------------------- main ---*/
{
  time_t tt;
  int irc;

  signal(SIGINT,QuitHandler);      /* Take care only of Ctrl-C */

  /*-------------------------------------------------------- Get parameters---*/
  FdsIni(argc, argv);

  /*-------------------------------------------------- start Cm Connection ---*/
  if(!CmMessageOpenMultipleServer("FdStat")) {
    printf("Cannot start Cm... NameServer problem?\n");
    exit(-1);}
  Fds.myName = CmConnectGetName(CmConnectWhoAmI());

  CmMessageInstallHandler (FrameHandler, "FdFrame");

  /*---------------------------------------------------------request frame ---*/
  printf(" Ask to %s the channels %s\n",Fds.srcName, Fds.tag);
  irc = FdRequestFrames (Fds.srcName, Fds.myName, Fds.tag, 0, 1);
  if(irc == 0) {
    printf("Could not contact the frame source.\n");
    exit(0);}

  /*------------------------------------------------------ Cm Waiting loop ---*/
  tt = time(NULL);
  fprintf(stderr,"\nFdStat: %s: Waiting for frames... %s \n", 
	  Fds.myName, ctime(&tt));

  while (1) {
    CmMessageWait();}

  QuitHandler(0);

  return(0);
}

