#include "FdIO.h"

/*-----------------------------------------------declare private functions---*/
typedef struct FdRecast  FdRecast;

struct FdRecast{
  char*      tag;          /* input channel definition    */
  int        type;         /* define the new vector type (from nBits)  */
  int        mode;         /* 0=replace,1=suffix on new; 2 suffix on old*/
  double     scale;        /* scaling factor when converting channel */
  FdAction*  action;      /* action object which hold this recast object */
};
 
typedef struct FdConvertSer  FdConvertSer;

struct FdConvertSer{
  FdAction*  action;      /* action object which hold this convert object */
};
 
int  FdRecastNew(void *arg, CfgDataType_t *data);
void FdRecastProcess(FdRecast* list, FrameH* frame);
void FdConvertSerProcess(FdConvertSer *convert, FrameH* frame);
 
/*---------------------------------------------------------------------------*/
int FdConvertSerNew(void *arg, CfgDataType_t *data)
/*---------------------------------------------------------------------------*/
{
  FdConvertSer *convert;

  convert = (FdConvertSer*) calloc(1,sizeof(FdConvertSer));
  if(convert == NULL) CfgMsgAddFatal("malloc FdConvertSer failed");

  convert->action = FdActionNew((FdAction**) arg, (void*) convert, 
                                FdConvertSerProcess, "ConvertSer");

  CfgMsgAddInfo("FrSerData will be converted to FrAdcData");

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdConvertSerProcessOne(FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrAdcData *adc;
  FrVect *isThere;

  //---------convert serData having a rate equal or larger than the frame rate
  FrameReshapeConvertSer(frame, frame, 1.1*frame->dt); 

  /*-------------------remove the auxiliary vectors created by ConvertSer---*/ 
  if(frame->rawData == NULL) return;
  for(adc = frame->rawData->firstAdc; adc != NULL; adc = adc->next) {
    if(adc->data == NULL) continue;

    isThere = adc->data->next;
    if(isThere == NULL)                                 continue;
    if(isThere->nData != 1)                             continue;
    if(isThere->type != FR_VECT_C)                      continue;
    if(strncmp(isThere->name,"Available_data",14) != 0) continue;
    if(isThere->data[0] != 1)                           continue;
    FrVectFree(isThere);
 
   adc->data->next = NULL;}
 
  return;
}
/*---------------------------------------------------------------------------*/
void FdConvertSerProcess(FdConvertSer *convert,
		         FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdConvertSerProcessOne(frame);

  next = convert->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*---------------------------------------------------------------------------*/
FrVect* FdRecastCopyToType(FrVect *vect, int type, double scale)
/*---------------------------------------------------------------------------*/
{
  FrVect *newVect = NULL;
  char *name;

  if((vect->type != FR_VECT_8R) &&
     (vect->type != FR_VECT_4R) &&
     (vect->type != FR_VECT_2S) &&
     (vect->type != FR_VECT_4S)) return(NULL);

  name = vect->name;
  if     (type == FR_VECT_8R) {newVect = FrVectCopyToD(vect, scale, name);}
  else if(type == FR_VECT_4R) {newVect = FrVectCopyToF(vect, scale, name);}
  else if(type == FR_VECT_2S) {newVect = FrVectCopyToS(vect, scale, name);}
  else if(type == FR_VECT_4S) {newVect = FrVectCopyToI(vect, scale, name);}
  else {CfgMsgAddFatal("FdRecastCopyToType: invalid type=%d",type);}

  if(newVect == NULL) CfgMsgAddFatal("FdRecast Malloc failed for %s",name);
 
  return(newVect);
}
/*---------------------------------------------------------------------------*/
int FdRecastNew(void *arg, CfgDataType_t *data)
/*---------------------------------------------------------------------------*/
{
  FdRecast *recast;
  int nBits;

  /*------------create the object and add it at the end of the linked list---*/
  recast = (FdRecast*) calloc(1,sizeof(FdRecast));
  if(recast == NULL) CfgMsgAddFatal("malloc FdRecast failed");

  recast->action = FdActionNew((FdAction**) arg, 
			       (void*) recast, FdRecastProcess, "Recast");
  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&recast->tag, CfgParseGetNextString(data));
  nBits         = CfgParseGetNextDec(data);
  recast->mode  = CfgParseGetNextDec(data);
  recast->scale = CfgParseGetNextReal(data);

  CfgMsgAddInfo("Recast output channel(s) %s to %d bits with scale factor=%g",
		recast->tag, nBits, recast->scale);

  /*-------------- define the type acordingly the number of requested bits---*/
  if     (nBits == -64) recast->type = FR_VECT_8R;
  else if(nBits == -32) recast->type = FR_VECT_4R;
  else if(nBits ==  16) recast->type = FR_VECT_2S;
  else if(nBits ==  32) recast->type = FR_VECT_4S;
  else CfgMsgAddFatal("FdRecast: nBit should be -64, -32, 16 or 32");

  if     (recast->mode == 0) 
    CfgMsgAddInfo("  mode = 0: Channels will be replaced");
  else if(recast->mode == 1) 
    CfgMsgAddInfo("  mode = 1: Create new channel with suffix");
  else if(recast->mode == 2) 
    CfgMsgAddInfo("  mode = 2: Create new channel; add suffix to old channel");
  else CfgMsgAddFatal(" Undefined cast mode: %d",recast->mode);
 
  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
char* FdRecastNewName(FrVect* vect)
/*---------------------------------------------------------------------------*/
{
  char *newName;
  
  newName = malloc(strlen(vect->name)+4);
  if(newName == NULL) CfgMsgAddFatal("FdRecast: malloc failed for newName");

  if     (vect->type == FR_VECT_4R) {sprintf(newName,"%s_4R",vect->name);}
  else if(vect->type == FR_VECT_8R) {sprintf(newName,"%s_8R",vect->name);}
  else if(vect->type == FR_VECT_2S) {sprintf(newName,"%s_2S",vect->name);}
  else if(vect->type == FR_VECT_4S) {sprintf(newName,"%s_4S",vect->name);}
  else {CfgMsgAddFatal("FdRecast: Unsupported type for %s",vect->name);} 

  FrVectSetName(vect, newName); 

  return(newName);
}
/*---------------------------------------------------------------------------*/
void FdRecastAdc(FdRecast* recast,
		 FrameH* frame,
		 FrAdcData* adc)
/*---------------------------------------------------------------------------*/
{
  FrAdcData *a;
  FrVect *newVect;
  char *newName;
  double scale;

  /*----------------------------------------create a new channel if needed---*/
  if(recast->mode != 0) {
    a = FrAdcDataCopy(adc, frame);  
    if(a == NULL) CfgMsgAddFatal("FdRecast malloc failed at %d for %s",
				 frame->GTimeS, adc->name);}
  else {a = adc;}

  /*------------------------------------------------replace the old vector---*/
  if(recast->scale != 0) {
    scale = recast->scale;
    a->slope *= 1./recast->scale;}
  else {
    scale = a->slope;
    a->slope = 1.;}
  newVect = FdRecastCopyToType(a->data, recast->type, scale);
  FrVectFree(a->data);
  a->data = newVect;

  /*-------------------------------------------------update name if needed---*/
  if(recast->mode == 0) return;
  if(recast->mode == 2) a = adc;
  newName = FdRecastNewName(a->data);
  free(a->name); 
  a->name = newName;

  return;
}
/*---------------------------------------------------------------------------*/
void FdRecastProc(FdRecast* recast,
		  FrameH* frame,
		  FrProcData* proc)
/*---------------------------------------------------------------------------*/
{
  FrProcData *p;
  FrVect *newVect;
  char *com;

  /*------------------------------------create a new channel if needed---*/
  if(recast->mode != 0) {
    p = FrProcDataCopy(proc, frame);
    if(p == NULL) CfgMsgAddFatal("FdRecast malloc failed at %d for %s",
				 frame->GTimeS, proc->name);}
  else {p = proc;}

  /*--------------------------------------------replace the old vector---*/
  newVect = FdRecastCopyToType(p->data, recast->type, recast->scale);
  FrVectFree(p->data);
  p->data = newVect;
  if(recast->scale != 1) {
    if(p->comment == NULL) {
      p->comment = malloc(50);
      if(p->comment == NULL) CfgMsgAddFatal("FdRecast malloc failed at %d",
					    frame->GTimeS);
      sprintf(p->comment,"FdRecast rescale factor: %g",recast->scale);}
    else {
      com = malloc(strlen(p->comment)+50);
      if(com == NULL) CfgMsgAddFatal("FdRecast malloc failed at %d",
				     frame->GTimeS);
      sprintf(com,"%s; FdRecast rescale factor: %g",p->comment,recast->scale);
      free(p->comment);
      p->comment = com;}}

  /*---------------------------------------------update name if needed---*/
  if(recast->mode == 0) return;
  if(recast->mode == 2) p = proc;
  free(p->name);
  p->name = FdRecastNewName(p->data);

  return;}

/*---------------------------------------------------------------------------*/
void FdRecastProcessOne(FdRecast* recast,
			FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FrProcData *proc;
  FrAdcData *adc;

  /*---------------------------------loop on all kind of recasting objects---*/
  FrameTag(frame, recast->tag);

  /*------------------------------------------ recast the procData channels--*/
  for(proc = frame->procData; proc != NULL; proc = proc->next) {
  
    /*------------------if the type and scale is already rigth: do nothing---*/
    if(proc->data == NULL) continue;
    if((recast->type == proc->data->type) && (recast->scale == 1)) continue; 

    /*--------------------------------------create a new channel if needed---*/
    FdRecastProc(recast, frame, proc);}

  /*----------------------------------------------- recast the ADC channels--*/
  if(frame->rawData == NULL) return;
  for(adc = frame->rawData->firstAdc; adc != NULL; adc = adc->next) {
 
    /*------------------if the type and scale is already rigth: do nothing---*/
    if((recast->type == adc->data->type) && (recast->scale == 1)) continue; 

    /*--------------------------------------create a new channel if needed---*/
    FdRecastAdc(recast, frame, adc);}

  FrameUntag(frame);

  return;
}

/*---------------------------------------------------------------------------*/
void FdRecastProcess(FdRecast *recast,
		     FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdRecastProcessOne(recast, frame);

  next = recast->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*---------------------------------------------------------------------------*/
void FdRecastParserAdd(FdIO* fdIO)
/*---------------------------------------------------------------------------*/
{
 
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_RECAST_CHANNELS",
			 FdRecastNew, (void *) &(fdIO->actionsIn), 4,
			 CfgString, CfgDec, CfgDec, CfgReal);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_RECAST_CHANNELS",
			 FdRecastNew, (void *) &(fdIO->actionsOut), 4,
			 CfgString, CfgDec, CfgDec, CfgReal);

  /*--remark: pass one unused parameter to FdConvertSerNew since
      CfgParseGetFunctionAdd crashes if the number of parameter is set to zero---*/
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_CONVERT_SERDATA",
			 FdConvertSerNew, (void *) &(fdIO->actionsIn), 1, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_CONVERT_SERDATA",
			 FdConvertSerNew, (void *) &(fdIO->actionsOut), 1, CfgDec);

  return;
}
