/*****************************************************************************/
/* FdDir */
/*****************************************************************************/
#include <FdIO.h>
#include <sys/stat.h>
#include <errno.h>

/*---------------------------------------------------------------------------*/
/* private definitions                                                      */
/*---------------------------------------------------------------------------*/

typedef struct FdFileStat FdFileStat;

struct FdFileStat{     /* a summary of the files included in a directory*/
  int    gps;          /* gps start */
  int    length;       /* file duration */
  double size;         /* file size */
  char  *fullName;     /* file name */
  FdFileStat *previous;/* previous object in the linked list */
  FdFileStat *next;    /* next object in the linked list */
};
typedef struct FdClnDir FdClnDir;

struct FdClnDir{       /* a summary of the files included in a directory*/
  char  *dirName;      /* directory name */
  int    period;       /* period check */
  int    nKeep;        /* nbr of file to keep */
  double sKeep;        /* size to keep */
  int    tKeep;        /* time to keep */
  char*  serId;        // optional id do identify multiple outputs
  char*  fflName;      /* ffl name */
  char*  fflInclude;   /* name of an ffl to include */
  int    overlap;      /* request max. time overlap with the included ffl*/ 
  int    eventMode;    /* tell how to fill the event part of the ffl 
			 ------------------------------------------------*/
  char*  fflTmp;       /* temporary name while building it*/
  int    lastCheck;    /* time of the directory check*/
  int    lastGps;      /* GPS time of the most recent file in the list */
  int    prefixLen;    /* length (number of characters) of the prefix */
  int    dirLen;       /* length (number of characters) of the directory name*/
  FdAction  *action;   /* root object in the action linked list */
};

FdClnDir* LastClnDir = NULL;

/*----------------------------------------------------------------FdClnDir---*/
void FdClnDirProcessOne(FdClnDir *clnDir, FrameH* frame)
/*---------------------------------------------------------------------------*/
/* This function build the list of file in a directory 
   removes from a directory the files which are too old.                     */
/*---------------------------------------------------------------------------*/
{
  DIR  *dirp;
  struct dirent *entry;
  struct stat sb;
  FdFileStat *files;   
  int  nameLen, len, gps, mostRecentGPS, mostRecentLen, nRemoved;
  int nFiles, nFilesAll, nFilesGwf, i, gpsNow, timeSpam, irc, pLen;
  int tStart, tEnd;
  char *prefix;
  FdFileStat *p, *next, *fStat, *pLimit, *previous, *last;
  double diskUsed, sizeRemoved;
  FRBOOL printError;
  FILE *ffl;
  FrFile* fflInclude;

  /*-------------------------------------enough time since the last check?---*/
  gpsNow = frame->GTimeS;

  if(gpsNow == 0) gpsNow = time(NULL); /*--if GPS is not provided, use local time--*/

  if(gpsNow < clnDir->lastCheck + clnDir->period) return;

  /*--------- align the check on a multiple of the period to do the check as
    soon a file is complited-------------------------------------------------*/
  if(clnDir->period > 0) 
    clnDir->lastCheck = clnDir->period * (gpsNow/clnDir->period);

  /*-------------------------------------------------------------------------*/
  dirp  = opendir(clnDir->dirName);
  if(dirp == NULL) CfgMsgAddFatal("FdClnDIr:Could not open folder %s",
				  clnDir->dirName);

  /*-----find the new files, based on their gps time taken from their name---*/
  mostRecentGPS = 0;
  mostRecentLen = 0;
  nFilesAll     = 0;
  nFilesGwf     = 0;
  pLen          = 0;
  files         = NULL;
  prefix        = NULL;
  printError    = FR_NO;

  /*---------------do a first scan of the directory to get file statistics---*/
  while((entry = readdir(dirp)) != NULL) {
    nFilesAll++;

    /*---------------------------------first check that it is a frame file---*/
    nameLen = strlen(entry->d_name);
    if(strcmp(entry->d_name + nameLen - 4, ".gwf") != 0) continue;

    /*-------------------------------------get file duration from its name---*/
    for(i=nameLen-4; i>0; i--) {if(entry->d_name[i] == '-') break;}
    if(i == 0) continue;
    if(sscanf(entry->d_name + i + 1, "%d", &len) != 1) continue;

    /*----------------------------------------get file start from its name---*/
    for(i--; i>0; i--) {if(entry->d_name[i] == '-') break;}
    if(i == 0) continue; 
    sscanf(entry->d_name + i + 1, "%d", &gps);

    /*------------------------------------------check prefix if available---*/
    if(prefix == NULL) {
      pLen = i+1;
      prefix = malloc(pLen+1);
      if(prefix == NULL) CfgMsgAddFatal("FdClnDir malloc prefix failed");
      strncpy(prefix, entry->d_name, i);
      prefix[i]   = '-';
      prefix[i+1] ='\0';}

    if(strncmp(entry->d_name, prefix, pLen) != 0) {
      if(printError == FR_NO) {
        printError = FR_YES;
        CfgMsgAddError("FdClnDir: more than one frame file type: %s and %s",
                        prefix, entry->d_name);}
      continue;}

    nFilesGwf++;

    /*-----------------------------------------------------------------------*/
    if(gps > mostRecentGPS) {
      mostRecentGPS = gps;
      mostRecentLen = len;}

    /*---------------------------create the FdFileStat linked list element---*/
    fStat = (FdFileStat *) calloc(1, sizeof(FdFileStat));
    if(fStat == NULL) CfgMsgAddFatal("FdDirCln: malloc FdFileStat failed");

    fStat->gps    = gps;
    fStat->length = len;
    fStat->size   = 0;

    fStat->fullName = malloc(clnDir->dirLen + nameLen + 2);
    if(fStat->fullName == NULL) CfgMsgAddFatal("FdDirCln: malloc fName failed");
    sprintf(fStat->fullName,"%s/%s", clnDir->dirName, entry->d_name);

    /*-------------------------------------------------------get file size---*/
    if(stat(fStat->fullName, &sb) == -1) 
	CfgMsgAddError("FdDirCln: could not get size for %s",fStat->fullName);
    fStat->size = sb.st_size;

    /*-------------------------------------------put it in the linked list---*/
    if(files == NULL) {
      files = fStat;
      continue;}

    /*--------------------make sure that the most recent file is first one---*/
    if(gps > files->gps) {
      fStat->next = files;
      files = fStat;
      continue;}

    for(p = files; p->next != NULL; p = p->next) {
      if(gps > p->next->gps) break;}
    fStat->next = p->next;
    p->next = fStat;

    } /*------------------------------------------end of the loop on files---*/
   
  closedir(dirp);

  if(prefix != NULL) free(prefix);

  if(clnDir->action->debugLvl > 0)
    CfgMsgAddInfo("FdClnDir: %d .gwf files out of %d files",nFilesGwf, nFilesAll);

  /*-----------------------------------------if no files found then return---*/
  if(files == NULL) return;
  clnDir->lastGps = mostRecentGPS;

  /*------------------------------------remove the to old files if needed----*/
  diskUsed = 0;
  nFiles = 0;
  mostRecentGPS += mostRecentLen;
  timeSpam = 0;
  nRemoved = 0;
  sizeRemoved = 0;

  /*---since the first files are the most recent, we loop over all files
     until we reach one of the request limits ------------------------------*/

  previous = NULL;
  last     = NULL;
  for(p = files; p != NULL; p = p->next) {
    p->previous = previous;
    previous = p;
 
    nFiles++;
    diskUsed += p->size/(1024*1024*1024);

    timeSpam = mostRecentGPS - p->gps;
    if(clnDir->nKeep != 0 && nFiles   > clnDir->nKeep) break;
    if(clnDir->sKeep != 0 && diskUsed > clnDir->sKeep) break;
    if(clnDir->tKeep != 0 && timeSpam > clnDir->tKeep) break;
    last = p;}

  pLimit = p;

  /*-------------------------------------remove the files after the limit---*/
  for(p = pLimit; p != NULL; p = p->next) {
      nRemoved++;
      sizeRemoved += p->size;
      if(clnDir->action->debugLvl > 0) 
        CfgMsgAddInfo("FdClnDir: Remove %s",p->fullName);
      remove(p->fullName);}
  sizeRemoved /= 1024*1024*1024;

  /*------------------------------------------------build ffl if requested---*/
  if(clnDir->fflName != NULL) {
    tEnd = 0.;
    ffl = fopen(clnDir->fflTmp,"w");
    if(ffl == NULL) CfgMsgAddError("FdClnDir: Could not open ffl %s (%s)",
				   clnDir->fflTmp, strerror(errno));
    else {
      /*---------------------------------add the included ffl is requested---*/
      if(clnDir->fflInclude != NULL) {
	fflInclude = FrFileINew(clnDir->fflInclude);
	if(fflInclude == NULL) {
	  CfgMsgAddError("Could not open %s",clnDir->fflInclude);}
	else {
          tStart = FrFileITStart(fflInclude);
          tEnd   = FrFileITEnd  (fflInclude);
          if(tEnd > tStart)  
	    fprintf(ffl,"%s %d %d\n",clnDir->fflInclude,tStart, tEnd);}
	FrFileIEnd(fflInclude);}

      /*--------------------------------------write the full list of file---*/
      for(p = last; p != NULL; p = p->previous) {
        if(p->gps < tEnd-clnDir->overlap) continue;
	fprintf(ffl,"%s %d %d",p->fullName,p->gps,p->length); 
        if     (clnDir->eventMode == 0) fprintf(ffl," %d %d",p->gps,p->gps+p->length); 
        else if(clnDir->eventMode == 1) fprintf(ffl," 0 0"); 
	fprintf(ffl,"\n");} 
      fclose(ffl);}

    irc = rename(clnDir->fflTmp, clnDir->fflName);
    if(irc != 0) CfgMsgAddError("ffl rename failed (%s) for %s",
			strerror(errno),clnDir->fflTmp);}

  /*------------------------------------------- free the linked list space---*/
  for(p = files; p != NULL; p = next) {
    next = p->next;
    free(p->fullName);
    free(p);}

  /*----------------------------------------------------------------debug---*/
  if(clnDir->action->debugLvl > 0)
    CfgMsgAddInfo("FdClnDir: nFiles=%d diskUsed=%g GB time spam=%d mostRecent=%d",
                   nFiles, diskUsed, timeSpam, mostRecentGPS);
  if(clnDir->action->debugLvl > 0) 
    CfgMsgAddInfo("FdClnDir: Removed %d files for %g GB", nRemoved, sizeRemoved);

   /*--------------------------fill info for user info message and SerData---*/
  if(clnDir->serId == NULL) {
    snprintf(clnDir->action->serData, clnDir->action->serDataSize,
	  " dirNfiles %d dirTimeSpam %d "
          "dirDiskUsedGB %.3f dirNremoved %d dirRemovedGB %g", 
           nFiles, timeSpam, diskUsed, nRemoved, sizeRemoved);
    snprintf(clnDir->action->userInfo, clnDir->action->userInfoSize,
	  "dir: nFiles=%d timeSpam=%d diskUsed=%.3f GB", 
	  nFiles, timeSpam, diskUsed);}
  else {
    snprintf(clnDir->action->serData, clnDir->action->serDataSize,
	  " %s_dirNfiles %d %s_dirTimeSpam %d "
          " %s_dirDiskUsedGB %.3f %s_dirNremoved %d %s_dirRemovedGB %g", 
           clnDir->serId, nFiles,   clnDir->serId, timeSpam, 
           clnDir->serId, diskUsed, clnDir->serId, nRemoved, 
           clnDir->serId, sizeRemoved);
    snprintf(clnDir->action->userInfo, clnDir->action->userInfoSize,
	  "dir %s: nFiles=%d timeSpam=%d diskUsed=%.3f GB", 
	  clnDir->serId, nFiles, timeSpam, diskUsed);}

  return;
}
/*---------------------------------------------------------------------------*/
void FdClnDirProcess(FdClnDir *clnDir,
		     FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdClnDirProcessOne(clnDir, frame);

  next = clnDir->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdClnDirNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
/* Create a FdClnDir object and put it in the linked list                   */
/*--------------------------------------------------------------------------*/
 {                                             
  FdClnDir *clnDir;

  clnDir = (FdClnDir*) calloc(1,sizeof(FdClnDir));
  if(clnDir == NULL) CfgMsgAddFatal("malloc FdDirCln failed");
 
  clnDir->action = FdActionNew((FdAction**) arg, (void*) clnDir, 
			       FdClnDirProcess, "ClnDir");

  LastClnDir = clnDir;

  FrStrCpy(&(clnDir->dirName), CfgParseGetNextString(data));
  clnDir->period  =  CfgParseGetNextDec(data);
  clnDir->nKeep   =  CfgParseGetNextDec(data);
  clnDir->sKeep   =  CfgParseGetNextReal(data);
  clnDir->tKeep   =  CfgParseGetNextDec(data);
  FrStrCpy(&(clnDir->serId), CfgParseGetNextString(data));

 /*------------------------------------Check that the file is a directory---*/
  if(opendir(clnDir->dirName) == NULL)
    CfgMsgAddFatal("FD_CLEAN_DIR: could not open: %s",clnDir->dirName);

  clnDir->dirLen = strlen(clnDir->dirName);

  /*-------------------------------------------------------------------------*/
  CfgMsgAddInfo("Add directory cleaning for:%s period:%d",
		clnDir->dirName, clnDir->period);
  if(clnDir->nKeep == 0 && clnDir->sKeep == 0 && clnDir->tKeep == 0)
    CfgMsgAddInfo(" No cleaning will be performed");
  if(clnDir->nKeep != 0) 
    CfgMsgAddInfo(" keep no more than %d most recent files",clnDir->nKeep);
  if(clnDir->sKeep != 0) 
    CfgMsgAddInfo(" keep no more than %g GBytes",clnDir->sKeep);
  if(clnDir->tKeep != 0) 
    CfgMsgAddInfo(" keep no more than %d s. of recent files",clnDir->tKeep);
  if(clnDir->serId != NULL) 
    CfgMsgAddInfo(" serData info names will contain %s", clnDir->serId);

  return(CFG_OK);
 }
/*--------------------------------------------------------------------------*/
int FdClnDirFflNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{         
  FILE *fp;
  
  if(LastClnDir == NULL) CfgMsgAddFatal("No FDOUT_CLEAN_DIR object found "
					  "before FDOUT_CLEAN_DIR_FFL");

  FrStrCpy(&(LastClnDir->fflName), CfgParseGetNextString(data));
  FrStrCpy(&(LastClnDir->fflInclude), CfgParseGetNextString(data));
  LastClnDir->overlap   = CfgParseGetNextDec(data);
  LastClnDir->eventMode = CfgParseGetNextDec(data);
  
  if(LastClnDir->fflInclude != NULL) {
    if(strcmp(LastClnDir->fflInclude,"NONE") == 0) {
      free(LastClnDir->fflInclude);
      LastClnDir->fflInclude = NULL;}}

  CfgMsgAddInfo(" FFL %s will be build for %s with events info mode:%d",
		LastClnDir->fflName, LastClnDir->dirName, LastClnDir->eventMode);
  if(LastClnDir->fflInclude != NULL) { 
    CfgMsgAddInfo(" It will include: %s as first element",
		  LastClnDir->fflInclude);
    CfgMsgAddInfo("   with a maximum time overlap of %d s",
                  LastClnDir->overlap);}

  LastClnDir->fflTmp = malloc(strlen(LastClnDir->fflName)+20);
  sprintf(LastClnDir->fflTmp, "%s_NOT_CLOSED", LastClnDir->fflName);
  fp = fopen(LastClnDir->fflTmp,"w");
  if(fp== NULL) 
    CfgMsgAddFatal("FdClnDirFFL: Could not write in ffl directory: %s",
                   strerror(errno));
  fclose(fp);
  remove(LastClnDir->fflTmp);

  return(CFG_OK);
}

/*--------------------------------------------------------------------------*/
int FdClnDirObsolete(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{                   
 
  CfgMsgAddFatal("FDOUT_CLEAN_DIRECTORY has been replaced by FDOUT_CLEAN_DIR"
                 " with new parameters");

  return(CFG_OK);
}
/*---------------------------------------------------------------- FdIOIni---*/
void FdClnDirParserAdd(FdIO* fdIO)
/*---------------------------------------------------------------------------*/
{ 
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_CLEAN_DIR", 
			 FdClnDirNew, (void *) &(fdIO->actionsOut), 6,
			 CfgString, CfgDec, CfgDec, CfgReal, CfgDec, CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_CLEAN_DIR_FFL", 
			 FdClnDirFflNew, NULL, 4, 
                         CfgString, CfgString, CfgDec, CfgDec);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_CLEAN_DIRECTORY", 
			 FdClnDirObsolete, NULL, 1, CfgString);

  return;
}

