#include "FdIO.h"

typedef struct FFLMoni FFLMoni;

struct FFLMoni{
  char*     fullName;
  char*     shortName;
  int       period;
  int       lastEndTime;
  double    fflTime;
  double    gwfTime;
  pthread_t thread_id;
  FFLMoni* next;
};

FFLMoni *Root;

//----------------------------------------------------------------------------
CmMessageStatus FMsmsHandler (CmMessage message, char* from, char* server)
//----------------------------------------------------------------------------
{
  FFLMoni *ffl;

  char serData[4096];
  char* p = serData;
  int now = FrGetCurrentGPS();

  p += sprintf(p, "units s");
  for(ffl = Root; ffl != NULL; ffl = ffl->next) {
    int latency = now - ffl->lastEndTime;
    if(ffl->fflTime < 0) latency = -1;
    p += sprintf(p," %s_ffl_latency %d %s_ffl_open_time %.3f %s_gwf_open_time %.3f",
	ffl->shortName, latency,
	ffl->shortName, ffl->fflTime,
	ffl->shortName, ffl->gwfTime);}

  CmMessageGetText(message);     
  CmMessageGetText(message);     
  int frameN = CmMessageGetInt(message);

  CmMessage answer = CmMessageNew();
  CmMessageSetType(answer, "FbSmsData");
  CmMessagePutInt (answer, frameN);
  CmMessagePutText(answer, serData);

  CmMessageSend(answer, from);

  CmMessageDelete(answer);

  return(CmMessageOk);
}
//----------------------------------------------------------------------------
int FMfflNew(void *arg,   CfgDataType_t *data)
//----------------------------------------------------------------------------
{        
  FFLMoni *ffl = (FFLMoni*) calloc(1, sizeof(FFLMoni));
  ffl->next = Root;
  Root = ffl;

  FrStrCpy(&ffl->fullName, CfgParseGetNextString(data));
  ffl->period = CfgParseGetNextDec(data);

  char *shortName = ffl->fullName;
  while(strstr(shortName,"/") != NULL) {
    shortName = strstr(shortName, "/")+1;}
  FrStrCpy(&ffl->shortName, shortName);
  ffl->shortName[strlen(shortName)-4] = '\0';

  ffl->fflTime = -1;
  ffl->gwfTime = -1;

  CfgMsgAddInfo("Will monitor %s (%s) every %d seconds", 
	ffl->fullName, ffl->shortName, ffl->period);

  return(CFG_OK);
}
//----------------------------------------------------------------------------
void* FMThreadMain(void *arg)
//----------------------------------------------------------------------------
{
  FFLMoni* ffl = arg; 
 
  CfgMsgAddInfo("Start thread to check %s", ffl->fullName);
 
  nice(10);  //-----------reduce the priority

  int lastTime = time(NULL);
  FRBOOL error = FR_YES;

  while(1) {
    usleep(10000);
    if(lastTime + ffl->period > time(NULL)) continue;
    lastTime = time(NULL);

    struct timeval tStart, tFFL, tEnd;
    gettimeofday(&tStart, NULL);

    //----------------------------------------------------------open the ffl
    FrFile *iFile = FrFileINew(ffl->fullName);
    if(iFile == NULL) {
      ffl->fflTime = -1;
      ffl->gwfTime = -1;
      if(error == FR_NO) {
        error = FR_YES;
        CfgMsgAddWarning("Open file error start for %s", ffl->shortName);}
      sleep(1); // this is to mitigate a memory leak....
      continue;}
  
    if(error == FR_YES) {
      error = FR_NO;
      CfgMsgAddInfo("Could read again %s",ffl->shortName);}

    //--------------------------------------get the end time of the last frame
    ffl->lastEndTime = FrFileITEnd(iFile);

    //--------------------------------------------measure the ffl opening time
    gettimeofday(&tFFL, NULL);
    ffl->fflTime = (tFFL.tv_sec  - tStart.tv_sec) + 1.e-6 *
                   (tFFL.tv_usec - tStart.tv_usec);

    //--------------------access the last gwf file and measure the access time 
    FrTOCFrameFindNT(iFile, ffl->lastEndTime-.1);
    if(iFile->toc == NULL) FrTOCReadFull(iFile);
    gettimeofday(&tEnd, NULL);
    ffl->gwfTime = (tEnd.tv_sec  - tFFL.tv_sec) + 1.e-6 *
                   (tEnd.tv_usec - tFFL.tv_usec);
    if(iFile->toc == NULL) ffl->gwfTime = -1;

    FrFileIEnd(iFile);}

  return(0);
}

/*------------------------------------------------------------- main ---------*/
int main(int argc, char *argv[])
/*------------------------------------------------------------- main ---------*/
{
  FdIO* fdIO = FdIONew(argc, argv);  //-------create the FdIO object; IDLE state

  //---------------------------------------------enrich parser and add cmHandler
  CfgParseGetFunctionAdd(fdIO->parser,"FFL", FMfflNew, NULL, 
				2, CfgString, CfgDec);

  FdIOParseAndIni(fdIO);      //--------------parse configuration and initialize

  CmMessageInstallHandler ((CmMessageHandler)FMsmsHandler, "FbGetAllSmsData");

  CfgReachState(CfgServerGolden);

  if(Root == NULL) CfgMsgAddFatal("No FFL provided");

  //-------------------------------------------------------------start thread(s)
  FFLMoni *ffl;
  for(ffl = Root; ffl != NULL; ffl = ffl->next) {
      pthread_create( &ffl->thread_id, NULL, FMThreadMain, ffl);}

  //------------------------------------------------------------------main loop
  while (!CfgFinished()) {    

    char info[4096], *p;
    int now = FrGetCurrentGPS();
    p = info;
    p += sprintf(p,"%.0f", FrGetCurrentGPS());
    for(ffl = Root; ffl != NULL; ffl = ffl->next) {
      int latency = now - ffl->lastEndTime;
      if(ffl->fflTime < 0) latency = -1;
      p += sprintf(p," %s.ffl: latency=%d ffl/gwf open times=%.0f/%.0f ms",
	ffl->shortName, latency, 1000*ffl->fflTime, 1000*ffl->gwfTime);}
    CfgMsgAddUserInfo(info);

    CfgMsgSendWithTimeout(0.01);}

  return(0);
}
