/*---------------------------------------------------------------------------*/
/*        Copyright (c) 1996 LAL Orsay, UPS-IN2P3-CNRS (France).             */
/*                                                                           */
/* Redistribution and use in source and binary forms, with or without        */
/* modification, are permitted provided that the following conditions        */
/* are met:                                                                  */
/* 1. Redistributions of source code must retain the above copyright         */
/*    notice, this list of conditions and the following disclaimer.          */
/* 2. Redistributions in binary form must reproduce the above copyright      */
/*    notice, this list of conditions and the following disclaimer in the    */
/*    documentation and/or other materials provided with the distribution.   */
/* 3. All advertising materials mentioning features or use of this software  */
/*    must display the following acknowledgement:                            */
/*      This product includes software developed by the Computer Application */
/*      Development Group at LAL Orsay (Laboratoire de l'Accelerateur        */
/*      Linaire - UPS-IN2P3-CNRS).                                           */
/* 4. Neither the name of the Institute nor of the Laboratory may be used    */
/*    to endorse or promote products derived from this software without      */
/*    specific prior written permission.                                     */
/*                                                                           */
/* THIS SOFTWARE IS PROVIDED BY THE LAL AND CONTRIBUTORS ``AS IS'' AND       */
/* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE     */
/* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR        */
/* PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE LAL OR CONTRIBUTORS BE      */
/* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR       */
/* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF      */
/* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS  */
/* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN   */
/* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)   */
/* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF    */
/* THE POSSIBILITY OF SUCH DAMAGE.                                           */
/*---------------------------------------------------------------------------*/
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <CmMessage.h>

/* OS9 tag used to compile shell_execute () on OS9 (see ChangeLog)*/
#undef CM_OS9

#ifdef OS9
#undef CM_ANY_UNIX
#define CM_OS9
#endif


/*--------------------------------------------------------------*/
CmMessageStatus NSNamesHandler (CmMessage message);
CmMessageStatus NSConnectsHandler (CmMessage message);
CmMessageStatus NSCommentsHandler (CmMessage message);
CmMessageStatus NSPortsHandler (CmMessage message);
CmMessageStatus NSPeriodHandler (CmMessage message);
CmMessageStatus NSCleanupHandler (CmMessage message);
CmMessageStatus NSStoppedHandler (CmMessage message);
CmMessageStatus NSFaultHandler (CmMessage message);
CmMessageStatus NSBlindHandler (CmMessage message);
CmMessageStatus NSVersionHandler (CmMessage message);
/*--------------------------------------------------------------*/
static char** Split (const char* text);
static char* shell_execute (const char* command);

/*--------------------------------------------------------------*/
static int Done = 0;
static int NeedWait = 0;
/*--------------------------------------------------------------*/

/*--------------------------------------------------------------*/
void Usage ()
/*--------------------------------------------------------------*/
{
  fprintf (stderr, " > cm <command> [args...]\n");
  fprintf (stderr, "     command :\n");
  fprintf (stderr, "\n");
  fprintf (stderr,
	   "      names                    : "
	   "Gets the names of active Cm applications\n");
  fprintf (stderr,
	   "      connects                 : "
	   "Gets the descriptions of active connections\n");
  fprintf (stderr,
	   "      comments                 : "
	   "Gets the comments of active servers\n");
  fprintf (stderr,
	   "      ports                    : "
	   "Get the list of allocated ports\n");
  fprintf (stderr,
	   "      period [<new period>]    : "
	   "Get or set the cleanup period of the NameServer\n");
  fprintf (stderr,
	   "      reconnect <name> <port> <host> <owner>   : "
	   "Try to reconnect NameServer to a lost application\n");
  fprintf (stderr,
	   "      restart_server           : "
	   "Restart the NameServer\n");
  fprintf (stderr,
	   "      stop_server              : "
	   "Stop the NameServer\n");
  fprintf (stderr,
	   "      domain                   : "
	   "Show domain info\n");
  fprintf (stderr,
	   "      domains                  : "
	   "Show all domains\n");
  fprintf (stderr,
	   "      version                  : "
	   "Show Cm version\n");
  fprintf (stderr,
	   "      trace <level> <name>     : "
	   "Trace messages in application <name> \n"
           "                                 "
           " 0     : off\n"
           "                                 "
           " 1/2/3 : on\n");
  fprintf (stderr,
	   "      debug [on/off] <name>    : "
	   "Set/reset debug mode in application <name>\n");
  fprintf (stderr,
	   "      check_base               : "
	   "Perform various checks on the database\n");
  fprintf (stderr,
	   "      send ....                : "
	   "Build and send a message to an application\n");
  fprintf (stderr,
	   "      receive ....             : "
	   "Receive messages\n");
  fprintf (stderr,
	   "      cleanup <name>           : "
	   "cleanup <name> in the Cm connects list (process does not exist anymore)\n");
  fprintf (stderr,
	   "      print <name>             : "
	   "print out all connections currently opened with <name>\n");
  fprintf (stderr,
	   "      kill <connect>           : "
	   "close <connect> Cm connect opened with the NS \n");
  fprintf (stderr,
	   "      kill <connect> <name>    : "
	   "close <connect> Cm connect opened with <name> \n");

}

/*---------------------------------------------------------------*/
static void SendUsage ()
/*---------------------------------------------------------------*/
{
  fprintf (stderr,
           " > cm send <commands> [-next <commands>...]\n"
           "     Send messages (separated by -next)\n"
           "\n"
           "     command : \n"
           "       -to <dest>               : set the destination\n"
           "       -type <type>             : set the message type\n"
           "       -handler <type>          : set a handler for an answer\n"
           "       \n"
           "       -c[har] <char>[,...]     : add a char item\n"
           "       -s[hort] <short>[,...]   : add a short item\n"
           "       -i[nt] <int>[,...]       : add an int item\n"
           "       -f[loat] <float>[,...]   : add a float item\n"
           "       -d[ouble] <double>[,...] : add a double item\n"
           "       -t[ext] <text>           : add a text item\n"
           "       \n" 
           "       -next                    : build another message\n"
           "       \n"
           "     eg. : \n"
           " > cm send -to abcd -type data -int 1,2,3 -double 3.156\n"
           "     \n"
           "\n");
}

/*---------------------------------------------------------------*/
static void ReceiveUsage ()
/*---------------------------------------------------------------*/
{
  fprintf (stderr,
           " > cm receive [-handle <type>...]\n"
           "     Receive messages\n"
           "\n");
}

/*---------------------------------------------------------------*/
static void fill_chars (CmMessage message, char* arg)
/*---------------------------------------------------------------*/
{
  static char*   c_array = 0;
  static int     c_number = 0;
  char value;

  c_number = 0;
      
  while (strlen (arg) > 0)
    {
      value = *arg;
      arg++;
      
      c_number++;
      if (c_array == 0)
        {
          c_array = (char*) malloc (c_number * sizeof (char));
        }
      else
        {
          c_array = (char*) realloc (c_array,
                                     c_number * sizeof (char));
        }
      
      c_array[c_number - 1] = value;
      
      if (*arg == 0) break;
      arg++;
    }
  
  if (c_number == 1)
    {
      CmMessagePutChar (message, c_array[0]);
    }
  else if (c_number > 0)
    {
      CmMessagePutArray (message, CmMessageChar, c_number, c_array);
    }
}

/*---------------------------------------------------------------*/
static void fill_shorts (CmMessage message, char* arg)
/*---------------------------------------------------------------*/
{
  static short*  s_array = 0;
  static int     s_number = 0;

  short value;
      
  s_number = 0;
      
  while (strlen (arg) > 0)
    {
      value = (short) strtol (arg, &arg, 10);
      
      s_number++;
      if (s_array == 0)
        {
          s_array = (short*) malloc (s_number * sizeof (short));
        }
      else
        {
          s_array = (short*) realloc (s_array,
                                      s_number * sizeof (short));
        }
      
      s_array[s_number - 1] = value;
      
      if (*arg == 0) break;
      arg++;
    }
  
  if (s_number == 1)
    {
      CmMessagePutShort (message, s_array[0]);
    }
  else if (s_number > 0)
    {
      CmMessagePutArray (message, CmMessageShort, s_number, s_array);
    }
}

/*---------------------------------------------------------------*/
static void fill_ints (CmMessage message, char* arg)
/*---------------------------------------------------------------*/
{
  static int*    i_array = 0;
  static int     i_number = 0;

  int value;
      
  i_number = 0;
      
  while (strlen (arg) > 0)
    {
      value = (int) strtol (arg, &arg, 10);
      
      i_number++;
      if (i_array == 0)
        {
          i_array = (int*) malloc (i_number * sizeof (int));
        }
      else
        {
          i_array = (int*) realloc (i_array,
                                    i_number * sizeof (int));
        }
      
      i_array[i_number - 1] = value;
      
      if (*arg == 0) break;
      arg++;
    }
  
  if (i_number == 1)
    {
      CmMessagePutInt (message, i_array[0]);
    }
  else if (i_number > 0)
    {
      CmMessagePutArray (message, CmMessageInt, i_number, i_array);
    }
}


/*---------------------------------------------------------------*/
static void fill_floats (CmMessage message, char* arg)
/*---------------------------------------------------------------*/
{
  static float*  f_array = 0;
  static int     f_number = 0;

  float value;
      
  f_number = 0;
      
  while (strlen (arg) > 0)
    {
      value = (float) strtod (arg, &arg);
          
      f_number++;
      if (f_array == 0)
        {
          f_array = (float*) malloc (f_number * sizeof (float));
        }
      else
        {
          f_array = (float*) realloc (f_array,
                                      f_number * sizeof (float));
        }
      
      f_array[f_number - 1] = value;

      /*      printf ("fill_floats> i=%d f=%g\n", f_number, f_array[f_number - 1]);*/
      
      if (*arg == 0) break;
      arg++;
    }
  
  if (f_number == 1)
    {
      CmMessagePutFloat (message, f_array[0]);
    }
  else if (f_number > 0)
    {
      CmMessagePutArray (message, CmMessageFloat, f_number, f_array);
    }
}

/*---------------------------------------------------------------*/
static void fill_doubles (CmMessage message, char* arg)
/*---------------------------------------------------------------*/
{
  static double* d_array = 0;
  static int     d_number = 0;

  double value;
      
  d_number = 0;
      
  while (strlen (arg) > 0)
    {
      value = strtod (arg, &arg);
      
      d_number++;
      if (d_array == 0)
        {
          d_array = (double*) malloc (d_number * sizeof (double));
        }
      else
        {
          d_array = (double*) realloc (d_array,
                                       d_number * sizeof (double));
        }
      
      d_array[d_number - 1] = value;
      
      /*      printf ("fill_doubles> i=%d f=%lg\n", d_number, d_array[d_number - 1]);*/
      
      if (*arg == 0) break;
      arg++;
    }
  
  if (d_number == 1)
    {
      CmMessagePutDouble (message, d_array[0]);
    }
  else if (d_number > 0)
    {
      CmMessagePutArray (message, CmMessageDouble, d_number, d_array);
    }
}

/*---------------------------------------------------------------*/
static void fill_text (CmMessage message, char* arg)
/*---------------------------------------------------------------*/
{
  char* value;
      
  value = arg;
      
  CmMessagePutText (message, value);
}

/*---------------------------------------------------------------*/
CmMessageStatus handler (CmMessage message, char* sender, char* server)
/*---------------------------------------------------------------*/
{
  char* type;
  CmMessageItemType item_type;

  type = CmMessageGetType (message);
  printf ("Message type %s received from %s\n", type, sender);

  do
    {
      item_type = CmMessageGetItemType (message);
      switch (item_type)
        {
          case CmMessageItemChar :
          {
            char c = CmMessageGetChar (message);
            printf ("  -char %c\n", c);
          }
          break;
          case CmMessageItemShort :
          {
            short s = CmMessageGetShort (message);
            printf ("  -short %d\n", s);
          }
          break;
          case CmMessageItemInt :
          {
            int i = CmMessageGetInt (message);
            printf ("  -int %d\n", i);
          }
          break;
          case CmMessageItemLong :
          {
            long l = CmMessageGetLong (message);
            printf ("  -long %ld\n", l);
          }
          break;
          case CmMessageItemFloat :
          {
            float f = CmMessageGetFloat (message);
            printf ("  -float %g\n", f);
          }
          break;
          case CmMessageItemDouble :
          {
            double d = CmMessageGetDouble (message);
            printf ("  -double %lg\n", d);
          }
          break;
          case CmMessageItemArray :
          {
            CmMessageArrayType array_type;
            int elements;
            char* array;
            char* carray;
            short* sarray;
            int* iarray;
            float* farray;
            double* darray;
            int i;

            array = (char*) CmMessageGetArray (message, &array_type, &elements);

            switch (array_type)
              {
                case CmMessageChar :
                  carray = (char*) array;
                  printf ("  -char ");
                  for (i = 0; i < elements; i++)
                    {
                      if (i > 0) printf (",");
                      printf ("%d", *carray);
                      carray++;
                    }
                  printf ("\n");
                  break;
                case CmMessageShort :
                  sarray = (short*) array;
                  printf ("  -short ");
                  for (i = 0; i < elements; i++)
                    {
                      if (i > 0) printf (",");
                      printf ("%d", *sarray);
                      sarray++;
                    }
                  printf ("\n");
                  break;
                case CmMessageInt :
                  iarray = (int*) array;
                  printf ("  -int ");
                  for (i = 0; i < elements; i++)
                    {
                      if (i > 0) printf (",");
                      printf ("%d", *iarray);
                      iarray++;
                    }
                  printf ("\n");
                  break;
                case CmMessageFloat :
                  farray = (float*) array;
                  printf ("  -float ");
                  for (i = 0; i < elements; i++)
                    {
                      if (i > 0) printf (",");
                      printf ("%g", *farray);
                      farray++;
                    }
                  printf ("\n");
                  break;
                case CmMessageDouble :
                  darray = (double*) array;
                  printf ("  -double ");
                  for (i = 0; i < elements; i++)
                    {
                      if (i > 0) printf (",");
                      printf ("%lg", *darray);
                      darray++;
                    }
                  printf ("\n");
                  break;
              }
          }
          break;
          case CmMessageItemText :
          {
            char* t = CmMessageGetText (message);
            printf ("  -text %s\n", t);
          }
          break;
          case CmMessageItemBytes :
          {
          }
          break;
          case CmMessageItemTail :
          break;
          case CmMessageItemUnknown :
          break;
        }
    } while (item_type != CmMessageItemTail);

  NeedWait--;


  Done = 1;
  if (NeedWait > 0) 
    return (CmMessageOk);
  else 
    return (CmMessageBreak);
}

/*--------------------------------------------------------------*/
/*    Ask the NameNerver any question.                          */
/*--------------------------------------------------------------*/
int main (int argc, char** argv)
{
  char* verb;
  CmMessage message;
  CmConnect server;
  char* destination = NULL;
  int has_answer = 0;

  char* domain;
  char answer[80];
  char* login;
	      
  if (!CmMessageOpenMultipleServer ("cm"))
    {
      CmConnectExit (CmErrorBadSocket);
    }

  if (argc == 1) 
    {
      Usage ();
      exit (1);
    }

  argc--;
  argv++;

  server = CmConnectGetNameServer ();
  destination = CmConnectGetName (server);

  message = CmMessageNew ();

  verb = argv[0];

  if (!strcmp (verb, "names"))
    {
      has_answer = 1;
      CmMessageInstallHandler ((CmMessageHandler) NSNamesHandler,
                               "NSNames");
      CmMessageSetType (message, "NSGetNames");
    }
  else if (!strcmp (verb, "connects"))
    {
      has_answer = 1;
      CmMessageInstallHandler ((CmMessageHandler) NSConnectsHandler,
                               "NSConnects");
      CmMessageSetType (message, "NSGetConnects");
    }
  else if (!strcmp (verb, "comments"))
    {
      has_answer = 1;
      CmMessageInstallHandler ((CmMessageHandler) NSCommentsHandler,
                               "NSComments");
      CmMessageSetType (message, "NSGetComments");
    }
  else if (!strcmp (verb, "ports"))
    {
      has_answer = 1;
      CmMessageInstallHandler ((CmMessageHandler) NSPortsHandler,
                               "NSPorts");
      CmMessageSetType (message, "NSGetPorts");
    }
  else if (!strcmp (verb, "period"))
    {
      has_answer = 1;
      CmMessageInstallHandler ((CmMessageHandler) NSPeriodHandler,
                               "NSPeriod");

      argc--;
      argv++;

      if (argc > 0)
        {
          int period;
          
          CmMessageSetType (message, "NSSetPeriod");
          sscanf (argv[0], "%d", &period);
          CmMessagePutInt (message, period);
        }
      else
        {
          CmMessageSetType (message, "NSGetPeriod");
        }
    }
  else if (!strcmp (verb, "reconnect"))
    {
      argc--;
      argv++;

      if (argc >= 4)
        {
          char* name = argv[0];
          int port = 0;
          char* host = argv[2];
          char* owner = argv[3];

          sscanf (argv[1], "%d", &port);

          CmMessageSetType (message, "NSReconnect");
          CmMessagePutText (message, name);
          CmMessagePutInt  (message, port);
          CmMessagePutText (message, host);
          CmMessagePutText (message, owner);
        }
    }
  else if (!strcmp (verb, "cleanup"))
    {
      has_answer = 1;
      CmMessageInstallHandler ((CmMessageHandler) NSCleanupHandler,
                               "NSCleanup");

      argc--;
      argv++;

      if (argc > 0)
        {
          CmMessageSetType (message, "NSDoCleanup");
          CmMessagePutText (message, argv[0]);
        }
      else verb = NULL;
    }
  else if (!strcmp (verb, "stop_server"))
    {
      has_answer = 1;

      /*Changes for SPR 84 */

      domain = getenv ("CMDOMAIN");
      login = shell_execute("whoami"); 

      if (!strcmp (login, "virgod") )
	{
	  if (!strcmp(domain, "Cascina"))
	    {
	      printf ("  #    #    ##    #####   #    #     #    #    #   ####  \n");
	      printf ("  #    #   #  #   #    #  ##   #     #    ##   #  #    # \n");
	      printf ("  #    #  #    #  #    #  # #  #     #    # #  #  #      \n");
	      printf ("  # ## #  ######  #####   #  # #     #    #  # #  #  ### \n");
	      printf ("  ##  ##  #    #  #   #   #   ##     #    #   ##  #    # \n");
	      printf ("  #    #  #    #  #    #  #    #     #    #    #   ####  \n\n");

	      printf ("   ####     ##     ####    ####      #    #    #    ##   \n");
	      printf ("  #    #   #  #   #       #    #     #    ##   #   #  #  \n");
	      printf ("  #       #    #   ####   #          #    # #  #  #    # \n");
	      printf ("  #       ######       #  #          #    #  # #  ###### \n");
	      printf ("  #    #  #    #  #    #  #    #     #    #   ##  #    # \n");
	      printf ("   ####   #    #   ####    ####      #    #    #  #    # \n");
	    } /*end if*/
	  
	  printf ("You are going to stop the server %s in the domain %s\n",
		  CmConnectGetName (server), domain);
	  printf ("Do you really want to do this ? y/[n] ");
	  fflush (stdout);
	  answer[0] = 0;
	 fgets(answer, sizeof answer, stdin );
	  
	  if ((answer[0] == 'y') || (answer[0] == 'Y'))
	    {
	      CmMessageInstallHandler ((CmMessageHandler) NSStoppedHandler, "NSStopped");
	      CmMessageSetType (message, "NSStop");
	      
	    }
	}/*end if*/
      else
	{
	  if (!strcmp (domain, "Cascina"))
	    {
	      printf ("User %s is not authorized to stop server %s in domain Cascina\n",
		      login, CmConnectGetName (server));
	      return (0);
	    }
	  else
	    {
	      printf ("You are going to stop the server %s of the domain %s\n", 
		      CmConnectGetName (server), domain);
	      printf ("Do you really want to do this ? y/[n] ");
	      fflush (stdout);
	      answer[0] = 0;
	      fgets(answer, sizeof answer, stdin );
	      
	      if ((answer[0] == 'y') || (answer[0] == 'Y'))
		{
		  CmMessageInstallHandler ((CmMessageHandler) NSStoppedHandler, "NSStopped");
		  CmMessageSetType (message, "NSStop");
		  
		}
	    }/*end else*/
	}/*end else*/
    }
  else if (!strcmp (verb, "restart_server"))
    {
      has_answer = 1;
      CmMessageInstallHandler ((CmMessageHandler) NSStoppedHandler,
                               "NSStopped");
      CmMessageSetType (message, "NSRestart");
    }
  else if (!strcmp (verb, "fault"))
    {
      has_answer = 1;
      CmMessageInstallHandler ((CmMessageHandler) NSFaultHandler,
                               "NSFault");
      CmMessageSetType (message, "NSForceFault");
    }
  else if (!strcmp (verb, "blind"))
    {
      argc--;
      argv++;

      if (argc > 0)
        {
          int seconds;

          sscanf (argv[0], "%d", &seconds);

          has_answer = 1;

          CmMessageInstallHandler ((CmMessageHandler) NSBlindHandler,
                                   "NSBlind");

          CmMessageSetType (message, "NSSetBlind");
          CmMessagePutInt (message, seconds);
        }
      else
        {
          destination = NULL;
        }
    }
  else if (!strcmp (verb, "trace"))
    {
      argc--;
      argv++;

      if (argc > 1)
        {
          int level = 0;
          
          CmMessageSetType (message, "CmMessageDebug");

          sscanf (argv[0], "%d", &level);
          destination = argv[1];

          if (level <= 0)
            {
              CmMessagePutText (message, "TraceOff");
            }
          else
            {
              CmMessagePutText (message, "TraceOn");
              CmMessagePutInt (message, level);
            }
        }
      else verb = NULL;
    }
  else if (!strcmp (verb, "debug"))
    {
      argc--;
      argv++;

      if (argc > 1)
        {
          CmMessageSetType (message, "CmMessageDebug");

          destination = argv[1];

          if (!strcmp (argv[0], "on"))
            {
              CmMessagePutText (message, "DebugOn");
            }
          else
            {
              CmMessagePutText (message, "DebugOff");
            }
        }
      else verb = NULL;
    }
  else if (!strcmp (verb, "check_base"))
    {
      char* env;
      char command[256];

      argc--;
      argv++;

      env = getenv ("CMROOT");
      sprintf (command, "%s/mgr/check_base.sh", env);
      system (command);

      destination = NULL;
    }
  else if (!strcmp (verb, "send"))
    {
      char dest[256];
      char type[256];

      destination = NULL;
      dest[0] = 0;

      argc--;
      argv++;

      if (argc == 0)
        {
          SendUsage ();
        }

      while (argc > 0)
        {
          char* arg;

          arg = argv[0];

          if (!strcmp (arg, "-to"))
            {
              if (argc > 1)
                {
                  argc--;
                  argv++;

                  arg = argv[0];
                  strcpy (dest, arg);
                }
              else
                {
                  SendUsage ();
                  exit (0);
                }
            }
          else if (!strcmp (arg, "-type"))
            {
              if (argc > 1)
                {
                  argc--;
                  argv++;

                  arg = argv[0];
                  strcpy (type, arg);
                }
              else
                {
                  SendUsage ();
                  exit (0);
                }
            }
          else if (!strcmp (arg, "-handler"))
            {
              if (argc > 1)
                {
                  argc--;
                  argv++;

                  arg = argv[0];
                  CmMessageInstallHandler (handler, arg);
                  NeedWait++;
                }
              else
                {
                  SendUsage ();
                  exit (0);
                }
            }
          else if (!strcmp (arg, "-next"))
            {
              if ((strlen (dest) != 0) &&
                  (strlen (type) != 0))
                {
                  CmMessageSetType (message, type);

                  if (!CmMessageSend (message, dest))
                    {
                      fprintf (stderr, "Application %s not found\n", dest);
                      NeedWait = 0;
                    }
              
                  if (NeedWait > 0)
                    {
                      CmMessageWait ();
                    }
                }
              
              CmMessageReset (message);
              
              NeedWait = 0;
            }
          else if (arg[0] == '-')
            {
              switch (arg[1])
                {
                  case 'c' :
                    if (argc > 1)
                      {
                        argc--;
                        argv++;
                        
                        arg = argv[0];
                        fill_chars (message, arg);
                      }
                    else
                      {
                        SendUsage ();
                        exit (0);
                      }
                    break;
                  case 's' :
                    if (argc > 1)
                      {
                        argc--;
                        argv++;
                        
                        arg = argv[0];
                        fill_shorts (message, arg);
                      }
                    else
                      {
                        SendUsage ();
                        exit (0);
                      }
                    break;
                  case 'i' :
                    if (argc > 1)
                      {
                        argc--;
                        argv++;
                        
                        arg = argv[0];
                        fill_ints (message, arg);
                      }
                    else
                      {
                        SendUsage ();
                        exit (0);
                      }
                    break;
                  case 'f' :
                    if (argc > 1)
                      {
                        argc--;
                        argv++;
                        
                        arg = argv[0];
                        fill_floats (message, arg);
                      }
                    else
                      {
                        SendUsage ();
                        exit (0);
                      }
                    break;
                  case 'd' :
                    if (argc > 1)
                      {
                        argc--;
                        argv++;
                        
                        arg = argv[0];
                        fill_doubles (message, arg);
                      }
                    else
                      {
                        SendUsage ();
                        exit (0);
                      }
                    break;
                  case 't' :
                    if (argc > 1)
                      {
                        argc--;
                        argv++;
                        
                        arg = argv[0];
                        fill_text (message, arg);
                      }
                    else
                      {
                        SendUsage ();
                        exit (0);
                      }
                    break;
                  default:
                    SendUsage ();
                    exit (0);
                }
            }
          else
            {
              SendUsage ();
              exit (0);
            }
          
          argc--;
          argv++;
        }
      
      if ((strlen (dest) != 0) &&
          (strlen (type) != 0))
        {
          CmMessageSetType (message, type);
          
          if (!CmMessageSend (message, dest))
            {
              fprintf (stderr, "Application %s not found\n", dest);
              NeedWait = 0;
            }
          
          if (NeedWait > 0)
            {
              CmMessageWait ();
            }
        }
      
      CmMessageReset (message);
      
      NeedWait = 0;
    }
  else if (!strcmp (verb, "receive"))
    {
      argc--;
      argv++;

      if (argc == 0)
        {
          ReceiveUsage ();
        }

      NeedWait = 0;

      while (argc > 0)
        {
          char* arg;

          arg = argv[0];

          if (!strcmp (arg, "-handler"))
            {
              if (argc > 1)
                {
                  argc--;
                  argv++;

                  arg = argv[0];
                  CmMessageInstallHandler (handler, arg);
                  NeedWait++;
                }
              else
                {
                  ReceiveUsage ();
                  exit (0);
                }
            }
          else
            {
              ReceiveUsage ();
              exit (0);
            }
          
          argc--;
          argv++;
        }
      
      if (NeedWait > 0)
        {
          CmMessageWait ();
        }
      
      NeedWait = 0;
    }
  else if (!strcmp (verb, "domains"))
    {
      char domains_file[256];
      char* cm_root;
      FILE* domains;
      char line[256];
      char all_domains[256] = "";
      int quiet = 0;
      
      argc--;
      argv++;
      
      while (argc > 0)
	{
	  char* arg = argv[0];
	  
	  if (!strcmp (arg, "-quiet"))
	    {
	      quiet = 1;
	    }
	  
	  argc--;
	  argv++;
	}
      
      cm_root = getenv ("CMMGR");
      sprintf (domains_file,"%s/CmDomains", cm_root);
      
      domains = fopen (domains_file, "r");
      if (domains != NULL)
	{
	  while (fgets (line, sizeof(line), domains) != NULL)
	    {
	      char** words;
	      char* nl;
                  
	      nl = strchr (line, '\n');
	      if (nl != NULL) *nl = 0;
	      
	      words = Split (line);
	      if (words != NULL)
		{
		  if (words[0][0] != '#')
		    {
		      strcat (all_domains, words[0]);
		      strcat (all_domains, "\n");
		    }
		  
		  free (words);
		}
	    }
	  
	  fclose (domains);
	}
      
      if (!quiet)  printf ("Possible Cm domains are\n");
      printf ("%s\n", all_domains);
    }
  else if (!strcmp (verb, "domain"))
    {
      char domain[256] = "";
      char cm_data_root[256] = "";
      char domains_file[256];
      char* cm_root;
      FILE* domains;
      char line[256];
      char all_domains[256] = "";
      int quiet = 0;

      argc--;
      argv++;

      while (argc > 0)
	{
	  char* arg = argv[0];
	  
	  if (!strcmp (arg, "-quiet"))
	    {
	      quiet = 1;
	    }
	  else
	    {
	      strcpy (domain, argv[0]);
	    }
	  argc--;
	  argv++;
	}
      
      if (strlen (domain) == 0)
	{
	  char* env;
	  
	  env = getenv ("CMDOMAIN");
	  if (env != NULL)
	    {
	      strcpy (domain, env);
	    }
	}
      
      cm_root = getenv ("CMMGR");
      sprintf (domains_file,"%s/CmDomains", cm_root);
              
      domains = fopen (domains_file, "r");
      if (domains != NULL)
	{
	  while (fgets (line, sizeof(line), domains) != NULL)
	    {
	      char** words;
	      char* nl;
	      
	      nl = strchr (line, '\n');
	      if (nl != NULL) *nl = 0;
	      
	      words = Split (line);
	      if (words != NULL)
		{
		  if (!strcmp (words[0], domain))
		    {
		      strcpy (cm_data_root, words[5]);
		    }
		  
		  if (words[0][0] != '#')
		    {
		      strcat (all_domains, words[0]);
		      strcat (all_domains, "\n");
		    }
		  
		  free (words);
		}
	    }
	  
	  fclose (domains);
	}
      
      if (strlen (domain) == 0)
	{
	  printf ("The Db domain is NOT defined.\n");
	  printf ("Please define it using the dbdomain command among\n"
		  "%s\n",
		  all_domains);
	}
      else
	{
	  char* server_name;
	  char* cm_domain;
	  
	  cm_domain = getenv ("CMDOMAIN");
	  server_name = CmConnectGetName (server);
	  
	  printf ("Definition of the %s domain\n", domain);
	  printf (" Cm domain      : %s\n", cm_domain);
	  
	  printf (" Cm Name Server : %s", server_name);

	  if (server == NULL)
	    {
	      printf (" (not running)\n");
	    }
	  else
	    {
	      printf (" (running on %s)\n", CmConnectGetHost (server));
	    }
	  
	  printf (" CM data directory : %s\n", cm_data_root);
	}
      if (!quiet)
	{ 
	  printf ("Possible Cm domains are\n");
	  printf ("%s\n", all_domains);
	}
    }
  else if (!strcmp (verb, "version"))
    {
      /*      char* version;
      
	      argc--;
	      argv++;
	      
	      version = shell_execute ("basename $CMROOT");
	      
	      printf ("%s\n", version);
      */

      has_answer = 1;
      CmMessageInstallHandler ((CmMessageHandler) NSVersionHandler,
                               "NSVersion");
      CmMessageSetType (message, "NSGetVersion");

    }
  else if (!strcmp (verb, "print"))
    {
      if (argc > 1)
	{
          destination = argv[1];

	  argc--;
	  argv++;
	  printf ("%s\n", destination);
	  CmMessageInstallHandler ((CmMessageHandler) handler,
				   "CmPrint");

	  CmMessageSetType (message, "CmRequestPrint");

	  NeedWait++;	  
	  has_answer = 1;
	}
      else
	{
	  destination = NULL;
	  Usage ();
	  exit (1); 
	}
    }
  else if (!strcmp (verb, "kill"))
    {
      if (argc == 2)
	{
	  char* name = argv[1];
	  argc--;
	  argv++;

	  CmMessageSetType (message, "CmRequestKill");

	  CmMessagePutText (message, name);
	}
      else if (argc == 3)
	{
	  char* name = argv[1];
	  destination = argv[2];

	  argc--;
	  argv++;
	  argc--;
	  argv++;

	  CmMessageSetType (message, "CmRequestKill");

	  CmMessagePutText (message, name);
	}

      else
	{
	  destination = NULL;
	  Usage ();
	  exit (1); 
	}
    }
  else
    {
      Usage ();
      exit (1);
    }

  if (destination != NULL)
    {
      CmMessageSend (message, destination);

      if (has_answer)
        {
          while (!Done)
            {
              CmMessageWait ();
            }
        }
      else
        {
          CmMessageWaitWithTimeout (1.);
        }
    }

  CmMessageDelete (message);

  CmMessageCleanup ();

  return (0);
}


/*--------------------------------------------------------------*/
CmMessageStatus NSNamesHandler (CmMessage message)
/*--------------------------------------------------------------*/
{
  char* name;

  while (CmMessageGetItemType (message) == CmMessageItemText)
    {
      name = CmMessageGetText (message);
      printf ("%s\n", name);
    }

  Done = 1;
  return (CmMessageBreak);
}

/*--------------------------------------------------------------*/
CmMessageStatus NSConnectsHandler (CmMessage message)
/*--------------------------------------------------------------*/
{
  char* name;
  char* host;
  int port;
  char* owner;
  int length;
  int i;

  while (CmMessageGetItemType (message) == CmMessageItemText)
    {
      name = CmMessageGetText (message);
      host = CmMessageGetText (message);
      port = CmMessageGetInt (message);
      owner = CmMessageGetText (message);
      printf ("%s", name);

      length = strlen (name) / 8;
      printf ("\t");
      for (i = length + 1; i <= 2; i++) printf ("\t");

      printf ("%s", host);

      length = strlen (host) / 8;
      printf ("\t");
      for (i = length + 1; i <= 2; i++) printf ("\t");

      printf ("%d\t%s\n", port, owner);
    }

  Done = 1;
  return (CmMessageBreak);
}

/*--------------------------------------------------------------*/
CmMessageStatus NSCommentsHandler (CmMessage message)
/*--------------------------------------------------------------*/
{
  char* name;
  char* comment;

  while (CmMessageGetItemType (message) == CmMessageItemText)
    {
      int i;

      name    = CmMessageGetText (message);
      comment = CmMessageGetText (message);

      printf ("%s", name);
      for (i = 0; i < 4-(strlen (name)/8); i++) printf ("\t");
      printf ("-> ");

      if ((comment == NULL) || (strlen(comment) == 0))
        {
          printf ("<none>\n");
        }
      else
        {
          printf ("%s\n", comment);
        }
    }

  Done = 1;
  return (CmMessageBreak);
}

/*--------------------------------------------------------------*/
CmMessageStatus NSPortsHandler (CmMessage message)
/*--------------------------------------------------------------*/
{
  int port;

  while (CmMessageGetItemType (message) == CmMessageItemText)
    {
      char* host = CmMessageGetText (message);
      printf ("host %s\n", host);

      while (CmMessageGetItemType (message) == CmMessageItemInt)
        {
          int status;

          port = CmMessageGetInt (message);

          printf ("  %d", port);

          if (CmMessageGetItemType (message) == CmMessageItemInt)
            {
              status = CmMessageGetInt (message);

              switch (status)
                {
                  case 0:
                    printf ("(available)\n");
                    break;
                  case 1:
                    printf ("(used)\n");
                    break;
                  case 2:
                    printf ("(busy)\n");
                    break;
                  default:
                    printf ("\n  %d\n", status);
                    break;
                }
            }
          else
            {
              printf ("\n");
            }
        }
    }

  Done = 1;
  return (CmMessageBreak);
}

/*--------------------------------------------------------------*/
CmMessageStatus NSPeriodHandler (CmMessage message)
/*--------------------------------------------------------------*/
{
  int period;

  printf ("period : ");
  period = CmMessageGetInt (message);
  printf ("%d ", period);
  printf ("\n");

  Done = 1;
  return (CmMessageBreak);
}

/*--------------------------------------------------------------*/
CmMessageStatus NSCleanupHandler (CmMessage message)
/*--------------------------------------------------------------*/
{
  char* text;

  printf ("cleanup : ");
  text = CmMessageGetText (message);
  printf ("%s ", text);
  printf ("\n");

  Done = 1;
  return (CmMessageBreak);
}

/*--------------------------------------------------------------*/
CmMessageStatus NSStoppedHandler (CmMessage message)
/*--------------------------------------------------------------*/
{
  printf ("stopped");
  printf ("\n");

  Done = 1;
  return (CmMessageBreak);
}

/*--------------------------------------------------------------*/
CmMessageStatus NSFaultHandler (CmMessage message)
/*--------------------------------------------------------------*/
{
  printf ("fault");
  printf ("\n");

  Done = 1;

  return (CmMessageBreak);
}

/*--------------------------------------------------------------*/
CmMessageStatus NSBlindHandler (CmMessage message)
/*--------------------------------------------------------------*/
{
  printf ("blind\n");

  Done = 1;

  return (CmMessageBreak);
}

/*--------------------------------------------------------------*/
CmMessageStatus NSVersionHandler (CmMessage message)
/*--------------------------------------------------------------*/
{
  char* name;

  while (CmMessageGetItemType (message) == CmMessageItemText)
    {
      name = CmMessageGetText (message);
      printf ("Version: %s\n", name);
    }

  Done = 1;
  return (CmMessageBreak);
}

/*----------------------------------------------------------*/
static char** Split (const char* text)
/*----------------------------------------------------------*/
/*  Split a line into words. Separators are spaces and tabs */
/*  Text enclosed in double quotes is one word.             */
/*----------------------------------------------------------*/
{
  char** result = NULL;
  int texts = 0;
  int finished = 0;
  char* t;

  if (text == NULL) return (NULL);

  t = (char*) malloc (strlen (text) + 1);
  strcpy (t, text);

  while (strlen (text) != 0)
    {
      char* space;

      while ((*t == ' ') ||
             (*t == '\t')) t++;

      if ((t[0] == '"') ||
          (t[0] == '\'') ||
          (t[0] == ':'))
        {
          char sep = t[0];
          char* quote;

          t++;
          quote = strchr (t, sep);
          if (quote != NULL) *quote = 0;
          else finished = 1;
        }
      else
        {
          int offset;

          offset = strcspn (t, " \t");
          if (offset < 0 || t[offset] == 0) finished = 1;
          if (!finished)
            {
              space = t + offset;
              *space = 0;
            }
        }

      if (result == NULL)
        {
          texts = 1;
          result = (char**) malloc ((texts + 1) * sizeof (char*));
        }
      else
        {
          texts++;
          result = (char**) realloc (result, 
                                     (texts + 1) * sizeof (char*));
        }

      result[texts - 1] = t;
      result[texts] = NULL;

      if (finished) break;

      t += strlen (t) + 1;
    }

  return (result);
}


/*----------------------------------------------------------*/
static char* shell_execute (const char* command)
/*----------------------------------------------------------*/
{
#ifdef CM_OS9

  printf ("cm> you can't execute this command (%s) on OS9 \n");
  return ("NULL");

#else

  static char output[256];
  FILE* file;

  output[0] = 0;

  file = popen (command, "r");
  if (file == NULL) return (output);

  if (fgets (output, sizeof (output), file) != NULL)
    {
      char* nl;
      nl = strchr (output, '\n');
      if (nl != NULL) *nl = 0;
    }

  pclose (file);

  return (output);

#endif


}
