/*---------------------------------------------------------------------------*/
/*        Copyright (c) 1996 LAL Orsay, UPS-IN2P3-CNRS (France).             */
/*                                                                           */
/* Redistribution and use in source and binary forms, with or without        */
/* modification, are permitted provided that the following conditions        */
/* are met:                                                                  */
/* 1. Redistributions of source code must retain the above copyright         */
/*    notic, this list of conditions and the following disclaimer.          */
/* 2. Redistributions in binary form must reproduce the above copyright      */
/*    notice, this list of conditions and the following disclaimer in the    */
/*    documentation and/or other materials provided with the distribution.   */
/* 3. All advertising materials mentioning features or use of this software  */
/*    must display the following acknowledgement:                            */
/*      This product includes software developed by the Computer Application */
/*      Development Group at LAL Orsay (Laboratoire de l'Accelerateur        */
/*      Linaire - UPS-IN2P3-CNRS).                                           */
/* 4. Neither the name of the Institute nor of the Laboratory may be used    */
/*    to endorse or promote products derived from this software without      */
/*    specific prior written permission.                                     */
/*                                                                           */
/* THIS SOFTWARE IS PROVIDED BY THE LAL AND CONTRIBUTORS ``AS IS'' AND       */
/* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE     */
/* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR        */
/* PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE LAL OR CONTRIBUTORS BE      */
/* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR       */
/* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF      */
/* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS  */
/* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN   */
/* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)   */
/* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF    */
/* THE POSSIBILITY OF SUCH DAMAGE.                                           */
/*---------------------------------------------------------------------------*/
#include <signal.h>
#include <errno.h>
#define __USE_BSD
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <netdb.h>
#include <poll.h>

/* All possible specific tags for Cm */
#undef CM_ANY_UNIX
#undef CM_VMS
#undef CM_LYNXOS
#undef CM_LYNXOSPOWERPC
#undef CM_LYNXOSCES
#undef CM_OS9
#undef CM_WIN32
#undef CM_AIX
#undef CM_LINUX

/* Select the appropriate tag(s) */

#define CM_ANY_UNIX

#ifdef vms
#undef CM_ANY_UNIX
#define CM_VMS
#endif

#ifdef LYNXOS
#define CM_LYNXOS
#endif

#ifdef LYNXOSPOWERPC
#define CM_LYNXOSPOWERPC
#endif

#ifdef LYNXOSCES
#define CM_LYNXOSCES
#endif

#ifdef OS9
#undef CM_ANY_UNIX
#define CM_OS9
#endif

#ifdef _WIN32
#undef CM_ANY_UNIX
#define CM_WIN32
#endif

#ifdef AIX
#undef CM_ANY_UNIX
#define CM_AIX
#endif

#ifdef linux
#undef CM_ANY_UNIX
#define CM_LINUX
#endif


/*-----  VMS  ------------------------*/
#ifdef CM_VMS

#define FD_ZERO(p)      memset((char *)(p), 0, sizeof(*(p)))
#define fd_set Fd_set
#include <time.h>
#include <fd.h>
#include <socket.h>
#include <inet.h>
#include <ucx$inetdef.h>

#ifdef __ALPHA
#ifndef htons
u_long rpc$htons();
#define htons rpc$htons
#endif /* htons */
#endif /* __ALPHA */

#endif
/*-----  VMS  ------------------------*/

/*-----  LYNXOS  ---------------------*/
#ifdef CM_LYNXOS

#define TSVME13x 1
#include <headers_30/conf.h>
#include <headers_30/param.h>
#include <headers_30/absolute.h>
#define FD_SETSIZE USR_NFDS
#define howmany(x,y) (((x)+((y)-1))/(y))

#endif
/*-----  LYNXOS  ---------------------*/

/*-----  LYNXOSPOWERPC  ---------------------*/
#ifdef CM_LYNXOSPOWERPC

#define howmany(x,y) (((x)+((y)-1))/(y))

#endif
/*-----  LYNXOSPOWERPC  ---------------------*/

/*-----  LYNXOSCES  ------------------*/
#ifdef CM_LYNXOSCES

#include <conf.h>
#include <param.h>
#include <absolute.h>
#define FD_SETSIZE USR_NFDS
#define howmany(x,y) (((x)+((y)-1))/(y))

#endif
/*-----  LYNXOSCES  ------------------*/

/*-----  OS9  ------------------------*/
#ifdef CM_OS9

#include <types.h>
#include <rpc/types.h>
#include <time.h>
#include <socket.h>
#include <in.h>
#include <select.h>
#include <sgstat.h>

#define TCP_NODELAY 0x01

typedef size_t ssize_t;

#endif   /*! OS9 */
/*-----  OS9  ------------------------*/


/*-----  LINUX -----------------------*/
#ifdef CM_LINUX

#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>

#define fd_mask __fd_mask

#define NFDBITS   (8 * sizeof(unsigned long))

#define h_addr  h_addr_list[0] /* To compile with gcc 4.3.3 h_addr is the first
				  address in h_addr_list.*/
#endif
/*-----  LINUX -----------------------*/

/*-----  ANY_UNIX --------------------*/
#ifdef CM_ANY_UNIX

#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>

#endif
/*-----  ANY_UNIX --------------------*/

/*-----  AIX -------------------------*/
#ifdef CM_AIX

#include <sys/types.h>
#include <time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>

#endif
/*-----  AIX -------------------------*/

/*-----  Windows  --------------------*/
#ifdef CM_WIN32

#include <io.h>
#include <time.h>
#include <winsock.h>
#define howmany(x,y) (((x)+((y)-1))/(y))
#define fds_bits fd_array
#define NFDBITS (sizeof (SOCKET) * 8)

#endif
/*-----  Windows  --------------------*/


#ifndef howmany
#define howmany(x,y) (((x)+((y)-1))/(y))
#endif

#include <netdb.h>

#include <CmMessage.h>
#include <CmVersion.h>
#include <CmHandle.h>

int vprintf (const char *format, va_list ap);

/*--------------------------------------------------------------*/
typedef struct _CmConnectRec
{
    /* Every Connect has a name */
  
  char*                      name;
  char*                      comment;

    /* Statistics */

  CmConnectStatisticsRec     statistics;

  CmConnectReceiveHandler    receive;
  CmConnectRearmHandler      rearm;
  
  CmConnectStatus            data_pending;
  int                        locked;
  CmConnectStatus            data_sending;
  
    /* Local for pipe-managed connects */
  
  int                        local;

    /*
      Associate connections :
      server     is the Server that handles this connection
      receiver   is the special local connection (using pipes)
                 associated with every server in order to deal
                 with self-to-self communication.
      */
  
  CmHandle                   server;
  CmHandle                   receiver;
  
    /* TCPIP parameters */
  
  char*                      host_name;
  char*                      owner;
  
  struct sockaddr_in         socket_structure;
  int                        socket_id;
  int                        port;

    /* Attributes */
  
  struct timeval*            timeout;
  int                        alive;
  int                        multiple;

    /* Pipe parameters for local connection */
  int                        local_read;
  int                        local_write;

    /* The reader object for this connection */

  CmHandle                   message;
  List                       messages;

  CmHandle                   iterator;
  List                       iterators;

    /* 
       The transaction opened for the connection request
       with the NameServer
       */

  int                        transaction;
  int                        fds_id; 
  short                      events;
} CmConnectRec;

typedef struct _CmConnectIteratorRec
{
  Entry* entry;
  int forward;
} CmConnectIteratorRec;

typedef struct _CmConnectConnectionHandlerStack
{
  int size;
  CmConnectConnectionHandler* entries;
} CmConnectConnectionHandlerStack;
/*--------------------------------------------------------------*/


/*--------------------------------------------------------------*/
/*    Global variables.                                         */
/*--------------------------------------------------------------*/

static struct
{
  char*                           version;
  List                            connects;
  CmHandle                        current_server;

  CmConnectConnectionHandlerStack connection_handlers;
    /*CmConnectConnectionHandler      do_connect;*/

  CmConnectConnectionHandlerStack disconnection_handlers;
    /*CmConnectConnectionHandler      do_disconnect;*/

  CmMessage                       message;
  CmConnectPrinter                printer;
  int                             sndBufSize;
  int                             rcvBufSize;
  int                             nfds;
  struct pollfd                   *fds;
  int                             maxFds;
} CmConnectMgr = { .sndBufSize = 0, .rcvBufSize = 0, .fds = NULL, .nfds = 0 };


static CmNameServerInfos NameServerInfos  = NULL;

static char*             CmDomain              = NULL;

static int               CmConnectInitialized    = 0;

static int               CmConnectNumber         = 0;
static int               CmConnectNameNumber     = 0;
static int               CmConnectHostNameNumber = 0;
static int               CmConnectOwnerNumber    = 0;
static int               CmConnectTimeoutNumber  = 0;

static int               CmConnectDebug          = 0;

static double            CmNameServerTimeout     = 10.0;

  /*
    These variables should become obsolete when transaction
    mechanisms are generalised
    They are installed for backward compatibility
    */

static int CurrentGetAddressTransaction = 0;
static int CurrentGetNewPortTransaction = 0;

/*--------------------------------------------------------------*/
/*    Prototypes locaux.                                        */
/*--------------------------------------------------------------*/

static void CmConnectGetVersion ();
static int IsNameServer (char* name);
static int DoPrint (char* format, ...);

static CmConnect ServerInit (char* name, int multiple);
static int ServerBind (CmConnect connect, int port);

static void CmConnectSetComment (CmConnect connect, char* comment);

static void CmConnectSetServer (CmConnect connect, CmConnect server);
static void CmConnectDelete (CmConnect connect);
static int CmConnectGetPortAndName (CmConnect connect);
static int CmConnectGetNewPort (CmConnect connect);
static int CmConnectSendServerInfos (CmConnect connect);
static void CmConnectSetSocketProperties( CmConnect c, int snd, int rcv );
static CmConnect CreateNewConnect (char* name,
                                   char* host_name,
                                   int port);
static void CmConnectClear (CmConnect connect);

static CmConnectCondition CheckOrWait (int check_only, 
                                       struct timeval* big_timeout,
                                       CmMessageIterator it, char *tag);

static CmMessageStatus CmServerInfosHandler (CmMessage message,
                                             char* sender,
                                             char* server_name);
static CmMessageStatus NSPortHandler (CmMessage message,
                                      char* sender,
                                      char* server);
static CmMessageStatus NSNewPortHandler (CmMessage message);
static CmMessageStatus NSAddressHandler (CmMessage message);
static CmMessageStatus NSWakeupHandler (CmMessage message);

static CmMessageStatus RequestComment (CmMessage message,
                                       char* sender,
                                       char* server_name);
static CmMessageStatus RequestStatistics (CmMessage message,
                                          char* sender,
                                          char* server_name);
static CmMessageStatus RequestKill (CmMessage message,
				    char* sender,
				    char* server_name);
static CmMessageStatus RequestPrint (CmMessage message,
				     char* sender,
				     char* server_name);
static CmMessageStatus SetSocketSize (CmMessage message,
				     char* sender,
				     char* server_name);

static CmConnectStatus ServerDoConnect (CmConnect connect);

static char* BuildNewNameServerName (char *server_name);
static CmConnect ConnectToNameServer (CmConnect server);

static char* InitializeForHost (char* host_name, 
                                struct sockaddr_in* struct_socket);
static int Connect (CmConnect connect, int port);
#if 0
/* static void WaitSomeSeconds (int seconds); */
#endif
static void CmConnectLock (CmConnect connect);
static void CmConnectUnlock (CmConnect connect);


/* #if defined (CM_ANY_UNIX) || defined (CM_LINUX) */
/* static void SigAction (int sig); */
/* #endif */

static int TCPSocket (void );
static int TCPListen (int id);
#ifndef CM_USE_SELECT
static int TCPPoll( int nfds, struct timeval *timeout );
static int FillMasksPoll ( int *nfds );
static int CheckMasksPoll ( int nfds, int requested);
#else
static void printMask (fd_set* set);
static int TCPSelect ( int nfds, fd_set* rSet, fd_set* wSet, fd_set* xSet,
                      struct timeval* timeout);
static int FillMasksSelect (  int *nfds, fd_set* rmask, fd_set* wmask );
static int CheckMasksSelect (fd_set* rmask, fd_set* wmask, int requested);
#endif
static ssize_t PipeRecv (int id, char* buffer, int length);
static ssize_t TCPRecv (int id, char* buffer, int length);
static int TCPTest (int id);
static ssize_t PipeSend (int id, char* buffer, int bytes);
static ssize_t TCPSend (int id, char* buffer, int bytes);
static int TCPAccept (int id, struct sockaddr_in* soc);
static int TCPClose (int id);
static int TCPShutdown (int id);
static int TCPBind (int id, struct sockaddr_in* soc);
static int TCPConnect (int id, struct sockaddr_in* soc);
static int TCPGethostname (char* name, int size);
static void DontBlock (int id);                                      
static void Block (int id);

static void ConnectionHandlerStackInit (CmConnectConnectionHandlerStack* stack);
static void ConnectionHandlerStackPush (CmConnectConnectionHandlerStack* stack,
                                        CmConnectConnectionHandler function);
static void ConnectionHandlerStackCall (CmConnectConnectionHandlerStack* stack,
                                        CmConnect connect);

/*--------------------------------------------------------------*/



/*--------------------------------------------------------------*/
/*                                                              */
/*                                                              */
/*  Public global functions.                                    */
/*                                                              */
/*                                                              */
/*--------------------------------------------------------------*/


/*--------------------------------------------------------------*/
static int startup ()
/*--------------------------------------------------------------*/
{
#ifdef CM_WIN32
  WORD wVersionRequested; 
  WSADATA wsaData; 
  int err; 
  wVersionRequested = MAKEWORD(1, 1); 
 
  err = WSAStartup(wVersionRequested, &wsaData); 
 
  if (err != 0)
    {
        /* Tell the user that we couldn't find a useable */ 
        /* winsock.dll.     */ 
      return (0);
    }
  
/* Confirm that the Windows Sockets DLL supports 1.1.*/ 
/* Note that if the DLL supports versions greater */ 
/* than 1.1 in addition to 1.1, it will still return */ 
/* 1.1 in wVersion since that is the version we */ 
/* requested. */ 
  
  if (LOBYTE (wsaData.wVersion) != 1 || 
      HIBYTE (wsaData.wVersion) != 1) 
    { 
        /* Tell the user that we couldn't find a useable */ 
        /* winsock.dll. */ 
      WSACleanup(); 
      return (0); 
    } 
    /* The Windows Sockets DLL is acceptable. Proceed. */ 
#endif
  return (1);
}

/*--------------------------------------------------------------*/
void CmConnectForceStartup ()
/*--------------------------------------------------------------*/
{
  CvtForceStartup ();
  CmConnectInitialized = 0;
  CmDomain             = NULL;
  CmConnectInitialize ();
}

/*--------------------------------------------------------------*/
void CmConnectInitialize ()
/*--------------------------------------------------------------*/
{
  if (!CmConnectInitialized)
    {
      if (!startup ()) 
        {
          return;
        }

      CmConnectInitialized = 1;

      CmConnectMgr.version        = NULL;

#ifndef CM_USE_SELECT
      CmConnectMgr.maxFds = 16;
      CmConnectMgr.fds    = (struct pollfd *)calloc( CmConnectMgr.maxFds, sizeof( struct pollfd ) );
      CmConnectMgr.nfds   = 0;
      {
	int i;
	for( i = 0 ; i < CmConnectMgr.maxFds; i++ ) {
	  CmConnectMgr.fds[i].fd     = -1;
	  CmConnectMgr.fds[i].events = 0;
	}
      }
      DoPrint ("Cm> Use poll function.\n");
#else
      DoPrint ("Cm> Use select function.\n");

#endif

      CmConnectHandleInstall (&CmConnectMgr.current_server);

      ConnectionHandlerStackInit (&(CmConnectMgr.connection_handlers));
        /*CmConnectMgr.do_connect     = NULL;*/

      ConnectionHandlerStackInit (&(CmConnectMgr.disconnection_handlers));
        /*CmConnectMgr.do_disconnect  = NULL;*/

      CmConnectMgr.printer        = (CmConnectPrinter) vprintf;
       
#ifdef CM_LINUX
      if( !CmConnectMgr.rcvBufSize ) CmConnectMgr.rcvBufSize = 16*1024;
      if( !CmConnectMgr.sndBufSize ) CmConnectMgr.sndBufSize  = 8*1024;
#else
      if( !CmConnectMgr.rcvBufSize ) CmConnectMgr.rcvBufSize =  8*1024;
      if( !CmConnectMgr.sndBufSize ) CmConnectMgr.sndBufSize  = 8*1024;
#endif
      ListInit (&CmConnectMgr.connects);

      
      CmConnectNumber             = 0;
      CmConnectNameNumber         = 0;
      CmConnectHostNameNumber     = 0;
      CmConnectOwnerNumber        = 0;
      CmConnectTimeoutNumber      = 0;

      CmConnectDebug              = 0;

      CvtStartup ();

      CmConnectGetVersion ();

      {
        char* env;

        env = getenv ("CMCONNECTDEBUG");
        if (env != NULL)
          {
            CmConnectDebug = 0;

            sscanf (env, "%d", &CmConnectDebug);
          }
      }
      
      if (CmDomain == NULL)
        {
          char* domain;
          
          domain = getenv ("CMDOMAIN");
          if (domain == NULL)
            {
              DoPrint ("Cm> Please select a Cm domain before any use of it.\n");
              exit (CmErrorBadEnvironment);
            }
          CmConnectSetDomain (domain);
        }
      
        /* Initialize */
      
      NameServerInfos = CmNameServerGetInfos ();
      if (NameServerInfos == NULL)
        {
          DoPrint ("Cm> Please select a Cm domain before any use of it.\n");
          exit (CmErrorBadEnvironment);
        }

      CmMessageInitialize ();
      
        /*
          Still missing the wakeup handler.
          */
  
      CmMessageInstallHandler ((CmMessageHandler) CmServerInfosHandler,
                               "CmServerInfos");

      CmMessageInstallHandler ((CmMessageHandler) NSPortHandler,
                               "NSPort");

      CmMessageInstallHandler ((CmMessageHandler) NSNewPortHandler,
                               "NSNewPort");

      CmMessageInstallHandler ((CmMessageHandler) NSAddressHandler,
                               "NSAddress");

      CmMessageInstallHandler ((CmMessageHandler) NSWakeupHandler,
                               "NSWakeup");

      CmMessageInstallHandler ((CmMessageHandler) RequestComment,
                               "CmRequestComment");

      CmMessageInstallHandler ((CmMessageHandler) RequestStatistics,
                               "CmRequestStatistics");

      CmMessageInstallHandler ((CmMessageHandler) RequestPrint,
                               "CmRequestPrint");

      CmMessageInstallHandler ((CmMessageHandler) RequestKill,
                               "CmRequestKill");

      CmMessageInstallHandler ((CmMessageHandler) SetSocketSize,
                               "CmSetSocketSize");

    }
}

/*--------------------------------------------------------------*/
void CmConnectCleanup ()
/*--------------------------------------------------------------*/
{
  if (CmConnectInitialized)
    {
      CmConnect c;

      c = CmConnectGetNameServer ();

      if (CmConnectDebug > 0)
        {
          DoPrint ("CmConnectCleanup>\n");
        }

      CmConnectKill (c);
      CmConnectKill (CmConnectWhoAmI ());

      if (CmConnectMgr.version != NULL)
        {
          free (CmConnectMgr.version);
          CmConnectMgr.version = NULL;
        }

      
      if (CmConnectMgr.fds != NULL)
        {
          free (CmConnectMgr.fds);
          CmConnectMgr.fds = NULL;
        }


      CmConnectHandleUninstall (&CmConnectMgr.current_server);

      {
        List* list = &CmConnectMgr.connects;
        Entry* entry;

        while ((entry = list->first) != NULL)
          {
            c = (CmConnect) entry->reference;
            CmConnectKill (c);
            CmConnectDelete (c);
          }
      }

      CvtCleanup ();

      CmConnectInitialized = 0; 
   
      CmConnectSetDomain (NULL);
    }
}

/*--------------------------------------------------------------*/
void CmConnectCheckBalance ()
/*--------------------------------------------------------------*/
{
  if (CmConnectNumber != 0)
    {
      fprintf (stderr, "Cm>>> Bad balance on CmConnects : %d\n",
               CmConnectNumber);
    }
  if (CmConnectNameNumber != 0)
    {
      fprintf (stderr, "Cm>>> Bad balance on CmConnectNames : %d\n",
               CmConnectNameNumber);
    }
  if (CmConnectHostNameNumber != 0)
    {
      fprintf (stderr,
               "Cm>>> Bad balance on CmConnectHostNames : %d\n",
               CmConnectHostNameNumber);
    }
  if (CmConnectOwnerNumber != 0)
    {
      fprintf (stderr,
               "Cm>>> Bad balance on CmConnectOwners : %d\n",
               CmConnectOwnerNumber);
    }
  if (CmConnectTimeoutNumber != 0)
    {
      fprintf (stderr,
               "Cm>>> Bad balance on CmConnectTimeouts : %d\n",
               CmConnectTimeoutNumber);
    }

  CvtCheckBalance ();
}

/*----------------------------------------------------------------*/
CmConnectPrinter CmConnectInstallPrinter (CmConnectPrinter printer)
/*----------------------------------------------------------------*/
{
  CmConnectPrinter old;

  CmConnectInitialize ();

  old = CmConnectMgr.printer;

  CmConnectMgr.printer = printer;

  return (old);
}

/*--------------------------------------------------------------*/
char* CmConnectGetDomain ()
/*--------------------------------------------------------------*/
{
  return (CmDomain);
}

/*--------------------------------------------------------------*/
void CmConnectSetDomain (char* name)
/*--------------------------------------------------------------*/
{
  if (CmDomain != NULL)
    {
      free (CmDomain);
      CmDomain = NULL;
    }
      
  if (name != NULL)
    {
      CmDomain = (char*) malloc (strlen(name) + 1);
      strcpy (CmDomain, name);
    }
}

/*--------------------------------------------------------------*/
void CmConnectExit (CmErrorCode error)
/*--------------------------------------------------------------*/
{
  CmConnectCleanup ();

  exit (error);
}

/*--------------------------------------------------------------*/
CmConnectConnectionHandler CmConnectInstallConnectionHandler (
    CmConnectConnectionHandler function)
/*--------------------------------------------------------------*/
{
  ConnectionHandlerStackPush (&(CmConnectMgr.connection_handlers), 
                              function);

    /*
  CmConnectConnectionHandler old = CmConnectMgr.do_connect;
  
  CmConnectMgr.do_connect = function;
  return (old);
    */

  return (NULL);
}

/*--------------------------------------------------------------*/
CmConnectConnectionHandler CmConnectInstallDisconnectionHandler (
    CmConnectConnectionHandler function)
/*--------------------------------------------------------------*/
{
  ConnectionHandlerStackPush (&(CmConnectMgr.disconnection_handlers), 
                              function);

    /*
      CmConnectConnectionHandler old = CmConnectMgr.do_disconnect;

      CmConnectMgr.do_disconnect = function;
      return (old);
    */

  return (NULL);
}

/*----------------------------------------------------------------*/
void CmConnectDebugOn ()
/*----------------------------------------------------------------*/
{
  CmConnectDebug = 1;
}

/*----------------------------------------------------------------*/
void CmConnectDebugOff ()
/*----------------------------------------------------------------*/
{
  CmConnectDebug = 0;
}

/*----------------------------------------------------------------*/
char* CmConnectTime ()
/*----------------------------------------------------------------*/
{
  time_t ltime;
  char* strtime;
  char* nl;

  time (&ltime);
  strtime = ctime (&ltime);
  nl = strchr (strtime, '\n');
  if (nl != NULL) *nl = 0;

  return (strtime);
}


/*  Private functions ------------------------------------------*/

/*--------------------------------------------------------------*/
static void CmConnectGetVersion ()
/*--------------------------------------------------------------*/
{
  char* version;

  version = (char*) CM_PROTOCOL_VERSION;
  if (version == NULL)
    {
      DoPrint ("Cm> Cm environment not set.\n");
      CmConnectExit (CmErrorBadEnvironment);
    }
  
  CmConnectMgr.version = (char*) malloc (strlen(version) + 1);
  if (CmConnectMgr.version == NULL)
    {
      DoPrint ("Cm> Malloc error for Version.\n");
    }
  else
    {
      strcpy (CmConnectMgr.version, version);
    }
}

/*--------------------------------------------------------------*/
static int IsNameServer (char* name)
/*--------------------------------------------------------------*/
{
  if (NameServerInfos == NULL)
    {
      NameServerInfos = CmNameServerGetInfos ();
    }

  if (name == NULL) return (0);

  if (strlen (name) < strlen (NameServerInfos->name)) return (0);
  return (!strncmp (name,
                    NameServerInfos->name,
                    strlen (NameServerInfos->name)));
}


/*----------------------------------------------------------------*/
static int DoPrint (char* format, ...)
/*----------------------------------------------------------------*/
{
  va_list args;

  if (CmConnectMgr.printer)
    {
      va_start (args, format);
  
      CmConnectMgr.printer (format, args);
      
      va_end (args);
    }
  return (0);
}






/*--------------------------------------------------------------*/
/*                                                              */
/*                                                              */
/*  Server management.                                          */
/*                                                              */
/*                                                              */
/*--------------------------------------------------------------*/




/*--------------------------------------------------------------*/
CmConnect CmConnectStartupAsServer (char* name)
/*--------------------------------------------------------------*/
{
  printf ("Cm> This function is now obsolete.\n");
  printf ("Cm> Please use CmMessageOpenServer instead.\n");
  return (NULL);
}

/*--------------------------------------------------------------*/
CmConnect CmConnectStartupAsMultipleServer (char* name)
/*--------------------------------------------------------------*/
{
  printf ("Cm> This function is now obsolete.\n");
  printf ("Cm> Please use CmMessageOpenMultipleServer instead.\n");
  return (NULL);
}

/*--------------------------------------------------------------*/
CmConnect CmConnectOpenServer (char* name)
/*--------------------------------------------------------------*/
{
  CmConnect me;

  CmConnectInitialize ();
  
  me = ServerInit (name, 0);
  if (me == NULL)
    {
      DoPrint ("Cm> Error in server initialization.\n");
      CmConnectExit (CmErrorBadSocket);
    }

  return (me);
}

/*--------------------------------------------------------------*/
CmConnect CmConnectOpenMultipleServer (char* name)
/*--------------------------------------------------------------*/
/*    Initialisation en tant que serveur multiple.              */
/*    Le nom fourni en argument ne sera pas le nom              */
/*    effectif apres initialisation. Le NameServer ajoutera     */
/*    un suffixe [_<numero>] automatiquement le premier         */
/*    commencant a 1.                                           */
/*--------------------------------------------------------------*/
{
  CmConnect me;

  CmConnectInitialize ();
  
  me = ServerInit (name, 1);
  if (me == NULL)
    {
      DoPrint ("Cm> Error in multiple server initialization.\n");
      CmConnectExit (CmErrorBadSocket);
    }
  
  return (me);
}

/*--------------------------------------------------------------*/
void CmConnectCloseServer (char* name)
/*--------------------------------------------------------------*/
{
}

/*--------------------------------------------------------------*/
CmConnect CmConnectSelectServer (char* name)
/*--------------------------------------------------------------*/
{
  CmConnect connect;
  
  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (NULL);
    }

  connect = CmConnectGetReference (name);
  if (connect != NULL)
    {
      CmConnectHandleSet (&CmConnectMgr.current_server, connect);
    }

  return (connect);
}

/*--------------------------------------------------------------*/
void CmServerSetComment (char* comment)
/*--------------------------------------------------------------*/
{
  CmConnect me;

  me = CmConnectHandleGet (&CmConnectMgr.current_server);
  if (me == NULL) return;

  CmConnectSetComment (me, comment);
}

/*--------------------------------------------------------------*/
char* CmServerGetComment ()
/*--------------------------------------------------------------*/
{
  CmConnect me;

  me = CmConnectHandleGet (&CmConnectMgr.current_server);
  if (me == NULL) return ("");

  return (CmConnectGetComment (me));
}

/*--------------------------------------------------------------*/
CmConnect CmConnectWhoAmI ()
/*--------------------------------------------------------------*/
{
  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (NULL);
    }
  return (CmConnectHandleGet (&CmConnectMgr.current_server));
}


/*  Private functions ------------------------------------------*/

/*--------------------------------------------------------------*/
static int check_name (char* name)
/*--------------------------------------------------------------*/
{
  if (name == NULL) return (0);
  if (strlen (name) == 0) return (0);

  while (*name != 0)
    {
      unsigned char c = (unsigned char) *name;

      if (strchr (" \t~!^*()-+\"'|\\<>?/", c) != NULL) return (0);

      name++;
    }

  return (1);
}

/*--------------------------------------------------------------*/
static CmConnect ServerInit (char* name, int multiple)
/*--------------------------------------------------------------*/
{
  CmConnect me;
  CmConnect receiver;
  char      host_name[256];
  int       port;
  char*     fullHost;

  if (!check_name (name))
    {
      DoPrint ("Cm> Illegal character in '%s'.\n", name);
      CmConnectExit (CmErrorBadName);
    }

  me = CmConnectGetReference (name);
  if (me != NULL) return (me);

  if (TCPGethostname (host_name, sizeof (host_name)))
    {
      DoPrint ("Cm> Error in gethostname.\n");
      CmConnectExit (CmErrorBadHost);
    }

  if (IsNameServer (name))
    {
      struct hostent* localHost;
      struct hostent* requestedHost;
  
      localHost     = gethostbyname (host_name);
      requestedHost = gethostbyname (NameServerInfos->host);

      if ((localHost == NULL) ||
          (requestedHost == NULL) ||
          (localHost->h_addr[0] != requestedHost->h_addr[0]) ||
          (localHost->h_addr[1] != requestedHost->h_addr[1]) ||
          (localHost->h_addr[2] != requestedHost->h_addr[2]) ||
          (localHost->h_addr[3] != requestedHost->h_addr[3]))
        {
          DoPrint ("Cm> NameServer should be started on %s\n",
                   NameServerInfos->host);
          CmConnectExit (CmErrorBadHost);
        }
    }
  

#ifdef CM_OS9
  
  memset (host_name, '\0', 256);
  strcpy (host_name, getenv ("MYHOST"));
  DoPrint ("Cm> CmConnectOpenServer -> host = %s\n", host_name);
  
#endif

    /*
     Then the private reception connection is now created.

     We cannot use the ConnectNewWithAddress since the address (the port)
     is not yet known.
     */

  me = CreateNewConnect (name, host_name, 0);
  if (me == NULL)
    {
      DoPrint ("Cm> Malloc error on connect.\n");
      return (NULL);
    }

    /*
      The server's private connection is the only one that does not
      gets a CmMessage protocol, since it is only used for accepting
      connection requests.
      */
  
  CmConnectInitHandlers (me, ServerDoConnect, NULL);

  if ((me->socket_id = TCPSocket ()) == -1)
    {
      DoPrint ("Cm> Socket creation impossible.\n");
      
      me->alive = 0;
      return (NULL);
    }
  CmConnectSetSocketProperties( me, -1, -1 );

  fullHost = InitializeForHost (host_name, &me->socket_structure);
  if (fullHost == NULL)
    {
      DoPrint ("Cm> Host initialization impossible.\n");
      
      me->alive = 0;
      return (NULL);
    }

  CmConnectSetHost (me, fullHost);

    /*
      This new server becomes the 'current' server.
      */

  CmConnectHandleSet (&CmConnectMgr.current_server, me);

    /*
      The receiver connect is now created which makes use of
      Unix pipes for internal (self-to-self) communications.
      */
  receiver = CreateNewConnect (name, host_name, 0);

  CmConnectHandleSet (&me->receiver, receiver);
  
  receiver->socket_id = 0;
  receiver->local = 1;
  
  CmMessagePrepareHandler (receiver);
  
#if defined (CM_ANY_UNIX) || defined (CM_LINUX)
  {
    int desc[2];

    pipe (desc);

    receiver->local_read  = desc[0];
    receiver->local_write = desc[1];
  }
#endif

  if (multiple) CmConnectSetMultiple (me);

  if (IsNameServer (name))
    {
      if (!ServerBind (me, NameServerInfos->port))
        {
          me->alive = 0;
          return (NULL);
        }
    }
  else
    {
      CmConnect server;
      
        /*
          The 'server' relationship of a server is filled in
          with the Connect to the NameServer.
          */

      me->transaction = CmOpenTransaction ("CmConnectToNameServer", NULL);

      /*
       Le kernel Linux 2.4 produit un evt de lecture qui active le select
       sur la socket 'me'. Il faut desactiver le select pendant la periode
       d'initialisation.
       */

      CmConnectLock (me);

      server = ConnectToNameServer (me);
      
      CmConnectSetServer (me, server);
      if (server == NULL)
        {
          DoPrint ("Cm> NameServer is not reachable.\n");
          CmConnectExit (CmErrorNoNameServer);
        }
      
      CmConnectSetServer (receiver, me);
      
      port = CmConnectGetPortAndName (me);

      /*
       Le protocole d'initialisation avec le NameServer est 
       termine (bien ou mal) donc on peut liberer la socket locale.
       */

      CmConnectUnlock (me);

      if (port == 0)
        {
          me->alive = 0;
          return (NULL);
        }
      else
        {
          while (!ServerBind (me, port))
            {
	      DoPrint("Cm>ConnectToNameServer %s - port %d already used ???\n",
		      CmConnectGetName(me), port);
              port = CmConnectGetNewPort (me);
              if (!port)
                {
                  me->alive = 0;
                  return (NULL);
                }
            }
        }
    }
  
  ConnectionHandlerStackCall (&(CmConnectMgr.connection_handlers),
                              me);
    /*if (CmConnectMgr.do_connect != NULL) CmConnectMgr.do_connect (me);*/
  
  return (me);
}

/*--------------------------------------------------------------*/
static int ServerBind (CmConnect connect, int port)
/*--------------------------------------------------------------*/
{
  int status;

  connect->port = port;

  connect->socket_structure.sin_port        = htons (port);
  connect->socket_structure.sin_addr.s_addr = 0;
  connect->socket_structure.sin_family      = AF_INET;

  status = TCPBind (connect->socket_id, &connect->socket_structure);
  if (status == -1)
    {
      DoPrint ("Cm> Bind error.\n");
      connect->alive = 0;
      return (0);
    }

  status = TCPListen (connect->socket_id);
  if (status == -1)
    {
      DoPrint ("Cm> Bind error.\n");
      connect->alive = 0;
      return (0);
    }

  return (1);
}






/*--------------------------------------------------------------*/
/*                                                              */
/*  Connect object management.                                  */
/*                                                              */
/*--------------------------------------------------------------*/


static void
WaitForTransaction (const char *context, const char *type, int transaction, double timeout)
{
	struct timespec start_time;
	struct timespec time;
	struct timespec elapsed_time;

	clock_gettime (CLOCK_MONOTONIC_RAW, &start_time);

	while (!CmIsTransactionTerminated (transaction)) {
		CmConnectCondition condition;

		condition = CmConnectWaitWithTimeout (timeout);
		if (condition == CmConnectTimeoutDetection) {
			DoPrint ("%s> NameServer does not answer\n", context);
			break;
		}

		clock_gettime (CLOCK_MONOTONIC_RAW, &time);
		elapsed_time.tv_sec  = time.tv_sec  - start_time.tv_sec;
		elapsed_time.tv_nsec = time.tv_nsec - start_time.tv_nsec;
		if (elapsed_time.tv_nsec < 0) {
			--elapsed_time.tv_sec;
			elapsed_time.tv_nsec += 1000000000L;
		}

		if (((double) elapsed_time.tv_sec + (double) elapsed_time.tv_nsec / 1000000000.0) > timeout) {
			DoPrint ("%s> Timeout while waiting for %s from NameServer\n", context, type);
			break;
		}

	}
}

/*--------------------------------------------------------------*/
CmConnect CmConnectNew (char* name)
/*--------------------------------------------------------------*/
{
  CmConnect connect = NULL;
  CmConnect server = NULL;
  CmConnect nameServer = NULL;
  char* nameServerName = NULL;
  
  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (NULL);
    }

  if (IsNameServer (name)) connect = CmConnectGetNameServer ();
  if (connect == NULL) connect = CmConnectGetReference (name);
  if (connect != NULL) return (connect);

  server = CmConnectHandleGet (&CmConnectMgr.current_server);
  if (server != NULL) 
    {
      nameServer = CmConnectGetServer (server);
    }

  if (nameServer == NULL)
    {
      return (NULL);
    }
  else
    {
      CmMessage message;
      int transaction;
      int status;

      nameServerName = CmConnectGetName (nameServer);

      if (nameServerName == NULL)
        {
          /* Bug somewhere ??? */

          return (NULL);
        }
  
        /*
          Interrogation du NameServer pour recuperer le port et le host :
          
          1) On envoie le nom du destinataire.
          2) On recoit le numero de port (si le destinataire existe).
          le host_name du destinataire, et sa cle de connection.
        */
      
      transaction = CmOpenTransaction ("Cm> get address", NULL);
      
        /*
          Temporary solution for maintaining backward compatibility
          with non-transaction based protocol
        */
      CurrentGetAddressTransaction = transaction;
      
      message = CmMessageNew ();
      CmMessageSetType (message, "NSGetAddress");
      CmMessagePutText (message, name);
        /* Next int for backward compatibility */
      CmMessagePutInt  (message, 0);
      CmMessagePutInt  (message, transaction);

      status = CmMessageSend (message, nameServerName);
      CmMessageDelete (message);
  
      if (!status)
        {
          CmCloseTransaction (transaction);
          DoPrint ("Cm> Bad connection with NameServer.\n");
          return (NULL);
        }

    /*
      The answer about the port and name for this connection will
      arrive straight in the NSAddress Message handler.

      In this handler, the creation and configuration of the new
      connection will performed and a Break will signal about
      the correct execution.
      */

      WaitForTransaction ("CmConnectNew", "NSAddress", transaction, CmNameServerTimeout);

      CmCloseTransaction (transaction);

      connect = CmConnectGetReference (name);
    }

  return (connect);
}

/*--------------------------------------------------------------*/
CmConnect CmConnectNewWithAddress (CmConnect server,
                                   char* name, int port, char* host)
/*--------------------------------------------------------------*/
/*  Once the connection parameters are known, one may create    */
/* the Connect object and setup the TCPIP link.                 */
/*                                                              */
/*  This function does this job and will be typically used for  */
/*                                                              */
/*   o Create a normal new connection after the dialogue with   */
/*       the NameServer                                         */
/*   o Create a connection to the NameServer itself.            */
/*   o Re-create a connection by the NameServer.                */
/*                                                              */
/*                                                              */
/* Actions performed in this function:                          */
/*                                                              */
/* .The connect is created (if needed)                          */
/* .TCPIP initialization is done (socket - host - port)         */
/* .Link to Message system is done                              */
/* .Local connection parameters are sent to the new connection  */
/* .User connection handler is called                           */
/*                                                              */
/*--------------------------------------------------------------*/
{
  CmConnect connect;
  char* full_host;

  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (NULL);
    }

  if (!port)
    {
      DoPrint ("Cm> Bad port number specification.\n");
      return (NULL);
    }    

  /* On a suffisamment d'infos pour creer la connection. */

  connect = CmConnectGetReference (name);
  if (connect == NULL)
    {
      connect = CreateNewConnect (name, host, port);
      if (connect == NULL)
        {
          DoPrint ("Cm> Malloc error on connect.\n");
          return (NULL);
        }
    }

  CmConnectSetServer (connect, server);

  if (connect->socket_id < 0)
    {
      int id;
      
      id = TCPSocket ();
      if (id < 0)
        {
          DoPrint ("Cm> Socket creation error.\n");
          connect->alive = 0;
          return (NULL);
        }
      connect->socket_id = id;
      CmConnectSetSocketProperties( connect, -1, -1 );
    }

  full_host = InitializeForHost (host, &connect->socket_structure);
  if (full_host == NULL)
    {
      connect->alive = 0;
      return (NULL);
    }

  CmConnectSetHost (connect, full_host);

#ifdef CM_OS9
  if (Connect (connect, connect->port) == -1)
#else
  if (Connect (connect, connect->port))
#endif
    {
      if (CmConnectDebug > 0)
        {
          DoPrint ("Cm> Connect error.\n");
        }
      connect->alive = 0;
      return (NULL);
    }

  CmMessagePrepareHandler (connect);
  
  if (!CmConnectSendServerInfos (connect))
    {
      DoPrint ("Cm> Cannot send server infos.\n");
      connect->alive = 0;
      return (NULL);
    }

  ConnectionHandlerStackCall (&(CmConnectMgr.connection_handlers),
                              connect);
    /*if (CmConnectMgr.do_connect != NULL) CmConnectMgr.do_connect (connect);*/
  
  return (connect);
}

/*--------------------------------------------------------------*/
CmConnect CmConnectNewWithSocket (char* name, int socket_id)
/*--------------------------------------------------------------*/
{
  CmConnect connect;

  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (NULL);
    }

  connect = CreateNewConnect (name, NULL, 0);
  if (connect == NULL)
    {
      DoPrint ("Cm> Malloc error on connect.\n");

      /*---------------------------- Modified by F. BELLACHIA */
      /*------------> connect->alive = 0; <-------------------*/
      return (NULL);
    }
  
  connect->socket_id = socket_id;
  return (connect);
}

/*--------------------------------------------------------------*/
int CmConnectGetSocketId (CmConnect connect)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return (-1);

  return (connect->socket_id);
}

/*--------------------------------------------------------------*/
CmMessage CmConnectGetMessage (CmConnect connect)
/*--------------------------------------------------------------*/
{
  CmConnect receiver;

  if (connect == NULL) return (NULL);

  receiver = CmConnectHandleGet (&connect->receiver);

  if (receiver != NULL)
    {
      return (CmConnectGetMessage (receiver));
    }
  else
    {
      CmMessage message = CmMessageHandleGet (&connect->message);

      if (message == NULL)
        {
          CmMessageHandleSet (&connect->message, CmMessageNew ());
          message = CmMessageHandleGet (&connect->message);
        }

      return (message);
    }
}

/*--------------------------------------------------------------*/
void CmConnectPushMessage (CmConnect connect)
/*--------------------------------------------------------------*/
{
  CmConnect receiver;

  if (connect == NULL) return;

  receiver = CmConnectHandleGet (&connect->receiver);

  if (receiver != NULL)
    {
      CmConnectPushMessage (receiver);
    }
  else
    {
      CmMessage message = CmMessageHandleGet (&connect->message);

      if (message != NULL)
        {
          if (ListCount (&connect->messages) > 0)
            {
              DoPrint ("Cm> MessageQueue size %d.\n", 
                       ListCount (&connect->messages));
            }

          ListAdd (&connect->messages, message);
          CmMessageHandleClear (&connect->message);
          connect->statistics.received++;
        }
    }
}

/*--------------------------------------------------------------*/
CmMessage CmConnectTopMessage (CmConnect connect)
/*--------------------------------------------------------------*/
{
  CmConnect receiver;
  CmMessage message;

  if (connect == NULL) return (NULL);

  receiver = CmConnectHandleGet (&connect->receiver);

  if (receiver != NULL)
    {
      return (CmConnectTopMessage (receiver));
    }

  if (ListCount (&connect->messages) == 0) return (NULL);
  
  message = (CmMessage) ListFirst (&connect->messages);
  
  return (message);
}

/*--------------------------------------------------------------*/
CmMessage CmConnectPopMessage (CmConnect connect)
/*--------------------------------------------------------------*/
{
  CmConnect receiver;
  CmMessage message;

  if (connect == NULL) return (NULL);

  receiver = CmConnectHandleGet (&connect->receiver);

  if (receiver != NULL)
    {
      return (CmConnectPopMessage (receiver));
    }

  if (ListCount (&connect->messages) == 0) return (NULL);
  
  message = (CmMessage) ListPopFirst (&connect->messages);
  
  return (message);
}

/*--------------------------------------------------------------*/
int CmConnectIsNameServer (CmConnect connect)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return (0);

  return (IsNameServer (CmConnectGetName (connect)));
}

/*--------------------------------------------------------------*/
int CmConnectReceive (CmConnect connect, char* message, int length)
/*--------------------------------------------------------------*/
/*  When we get to this point, we know the connection has       */
/* something to say. Therefore we use directly the data receive */
/* operations.                                                  */
/*                                                              */
/*--------------------------------------------------------------*/
{
  int received;

  if (connect == NULL) return (0);
  if (!connect->alive) return (0);
  if (message == NULL) return (0);
  if (length == 0) return (0);

    /*
      Since we create a CmConnect object dedicated to receive
      local transmissions (through pipes) the "CheckOrWait" mechanism
      is able to branch directly to this Connect. Thus the receive
      handler knows about this reveiver connect instead of the main one.

      (which also means that for the Server itself there are two
      receive handlers one for the main Connect - used for connection
      requests - and one for the pipe - used for local transmissions)
      
      */
  
  if (connect->local)
    {
      received = PipeRecv (connect->local_read, message, length);
    }
  else
    {
        /*
          We change the mechanisms so that getting there occurs
          only when a detection on data available was triggered.
          */
      
      received = TCPRecv (connect->socket_id, message, length);
    }
  
  if (received >= 0)
    {
      connect->statistics.bytes_received += received;
    }

  if  (!received)
    {
      char* name = connect->name;

      if (name == NULL) name = (char*) "<unknown>";
      
      if (CmConnectDebug > 0)
        {
          DoPrint ("CmConnectReceive> Connection with %s lost.\n", name);
        }

      CmConnectKill (connect);
      
      return (0);
    }
  
  return (received);
}

/*--------------------------------------------------------------*/
int CmConnectSend (CmConnect connect, char* message, int length)
/*--------------------------------------------------------------*/
{
  CmConnect receiver;
  int written;
  
  if ((connect == NULL) || !connect->alive) return (0);

    /*
      If the message is to be sent to a connect owned by a local server,
      then it must be sent through the pipe.

      The detection of this case is done by the fact there exist a
      receiver Connect.
      
      */

  receiver = CmConnectHandleGet (&connect->receiver);
  if (receiver != NULL)
    {
      written = PipeSend (receiver->local_write, message, length);
      if (written < 0) written = 0;
    }
  else
    {
      written = TCPSend (connect->socket_id, message, length);
    }

  if (written >= 0)
    {
      connect->statistics.bytes_sent += written;
    } 

  return (written);
}

/*--------------------------------------------------------------*/
CmConnect CmConnectGetReference (char* name)
/*--------------------------------------------------------------*/
{
  CmConnect result = NULL;
  CmConnect connect;
  List* list;
  Entry* entry;

  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (NULL);
    }

  if (name == NULL) return (NULL);

  list = &CmConnectMgr.connects;
  entry = list->first;
  while (entry != NULL)
    {
      connect = (CmConnect) entry->reference;
      if ((connect != NULL) &&
          (connect->alive) &&
          (!connect->local) &&
          (connect->name != NULL) &&
          !strcmp(connect->name, name))
        {
          result = connect;
          break;
        }
      entry = entry->next;

    }

  return (result);
}

/*--------------------------------------------------------------*/
CmConnect CmConnectGetNthReference (char* name, int number)
/*--------------------------------------------------------------*/
{
  CmConnect result = NULL;
  CmConnect connect;
  List* list;
  Entry* entry;

  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (NULL);
    }

  if (name == NULL) return (NULL);
  if (number <= 0) number = 1;

  list = &CmConnectMgr.connects;
  entry = list->first;
  while (entry != NULL)
    {
      connect = (CmConnect) entry->reference;
      if ((connect != NULL) &&
          (connect->alive) &&
          (!connect->local) &&
          (connect->name != NULL) &&
          !strcmp(connect->name, name))
        {
          number--;
          if (number == 0)
            {
              result = connect;
              break;
            }
        }
      entry = entry->next;
    }

  return (result);
}

/*--------------------------------------------------------------*/
char* CmConnectGetName (CmConnect connect)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return (NULL);
  return (connect->name);
}

/*--------------------------------------------------------------*/
CmConnect CmConnectGetServer (CmConnect connect)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return (NULL);
  return (CmConnectHandleGet (&connect->server));
}

/*--------------------------------------------------------------*/
CmConnect CmConnectGetNameServer ()
/*--------------------------------------------------------------*/
{
  CmConnect server;
  CmConnect result = NULL;

  server = CmConnectHandleGet (&CmConnectMgr.current_server);
  
  while (server != NULL)
    {
      if (IsNameServer (CmConnectGetName (server)))
        {
          result = server;
          break;
        }
      server = CmConnectGetServer (server);
    }

  return (result);
}

/*--------------------------------------------------------------*/
void CmConnectSetName (CmConnect connect, char* newName)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;

  if (connect->name != NULL)
    {
      free (connect->name);
      CmConnectNameNumber--;
      connect->name = NULL;
    }

  if (newName != NULL)
    {
      connect->name = (char*) malloc (strlen (newName)+1);
      if (connect->name == NULL)
        {
          DoPrint ("Cm> Malloc error on connect name.\n");
          return;
        }
      else
        {
          CmConnectNameNumber++;
          strcpy (connect->name, newName);
        }
    }
}

/*--------------------------------------------------------------*/
int CmConnectIsLocal (CmConnect connect)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return (0);

  return (connect->local);
}

/*--------------------------------------------------------------*/
int CmConnectIsMultiple (CmConnect connect)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return (0);

  return (connect->multiple);
}

/*--------------------------------------------------------------*/
void CmConnectSetMultiple (CmConnect connect)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;

  connect->multiple = 1;
}

/*--------------------------------------------------------------*/
void CmConnectSetPort (CmConnect connect, int port)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;

  connect->port = port;
}


/*---------------------------------------------------------------------------*/
static void CmConnectSetSocketProperties (CmConnect connect, int snd, int rcv)
/*---------------------------------------------------------------------------*/
{
  int id;
  
  if (connect == NULL) return;
  if( (id = connect->socket_id) == -1) return;
  if( snd == -1 ) snd = CmConnectMgr.sndBufSize;
  if( rcv == -1 ) rcv = CmConnectMgr.rcvBufSize;
#ifdef CM_LINUX
  setsockopt( id, SOL_SOCKET, SO_SNDBUF, &snd, sizeof(int));
  setsockopt( id, SOL_SOCKET, SO_RCVBUF, &rcv, sizeof(int));
  /*------------------------------------------------------------*/
#endif
  return;
}


/*----------------------------------------------------------------------------*/
int CmConnectSnprint( CmConnect c, char *txt, int size )
/*----------------------------------------------------------------------------*/
{
  int id;
  int sBuf;
  int rBuf;
  unsigned int len;


  if( !c || !txt ) return(-1);
  /*--------------------------------------------------------------------------*/
  if( (id = c->socket_id) > 0 ) { 
    len = sizeof (int);
    getsockopt (id, SOL_SOCKET, SO_SNDBUF, &sBuf, &len);
    getsockopt (id, SOL_SOCKET, SO_RCVBUF, &rBuf, &len);
  } else sBuf = rBuf = -1; 
  id = snprintf (txt, size, "CmName %s, port %d,  Host %s, owner %s, "
		 "socket_id %d (%d,%d),  local %d, alive %d",
		(c->name ? c->name : "NULL"), 
		c->port, 
		(c->host_name ? c->host_name : "NULL"), 
		(c->owner ? c->owner : "NULL"),
		id, sBuf,  rBuf,  
		c->local, c->alive );
  /*--------------------------------------------------------------------------*/
  return( id ); 
}

/*----------------------------------------------------------------------------*/
void CmConnectSetSocketSize ( int snd, int rcv, char *tName )
/*----------------------------------------------------------------------------*/
{
  List *list;
  Entry *entry;
  CmConnect c;

  snd /= 2;
  rcv /= 2;
  /*-------------------------------------------------- Update the socket list */
  if( !tName || tName[0] == '*' ) {
    /*------------------------------------------------------------------------*/
    if( snd > 0 ) CmConnectMgr.sndBufSize = snd;
    if( rcv > 0 ) CmConnectMgr.rcvBufSize = rcv; 
    /*------------------------------------------------------------------------*/
    for( list = &CmConnectMgr.connects, entry = list->first;
	 entry != NULL; entry = entry->next ) {
      
      c = (CmConnect) entry->reference;
      /*----------------------------------------------------------------------*/
      CmConnectSetSocketProperties( c, -1, -1 );
    }
  } else {
    for( list = &CmConnectMgr.connects, entry = list->first;
	 entry != NULL; entry = entry->next ) {
      
      c = (CmConnect) entry->reference;
      if( tName && !strstr(c->name,tName) ) continue;
      /*----------------------------------------------------------------------*/
      CmConnectSetSocketProperties( c, abs(snd), abs(rcv) );	  
    }
  }
  /*--------------------------------------------------------------------------*/
  return;
}


/*--------------------------------------------------------------*/
int CmConnectGetPort (CmConnect connect)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return (0);

  return (connect->port);
}

/*--------------------------------------------------------------*/
char* CmConnectGetHost (CmConnect connect)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return (NULL);

  return (connect->host_name);
}

/*--------------------------------------------------------------*/
void CmConnectSetHost (CmConnect connect, char* host_name)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;
  if (host_name == NULL) return;

  if (connect->host_name != NULL)
    {
      free (connect->host_name);
      CmConnectHostNameNumber--;
      connect->host_name = 0;
    }

  if (host_name != NULL)
    {
      connect->host_name = (char*) malloc (strlen (host_name)+1);
      if (connect->host_name == NULL)
        {
          DoPrint ("Cm> Malloc error on host name.\n");
          return;
        }
      else
        {
          CmConnectHostNameNumber++;
          strcpy (connect->host_name, host_name);
        }
    }
}

/*--------------------------------------------------------------*/
char* CmConnectGetOwner (CmConnect connect)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return (NULL);

  return (connect->owner);
}

/*--------------------------------------------------------------*/
void CmConnectSetOwner (CmConnect connect, char* owner)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;

  if (connect->owner != NULL)
    {
      free (connect->owner);
      CmConnectOwnerNumber--;
      connect->owner = 0;
    }
  
  if (owner != NULL)
    {
      connect->owner = (char*) malloc (strlen (owner)+1);
      if (connect->owner == NULL)
        {
          DoPrint ("Cm> Malloc error on owner name.\n");
          return;
        }
      else
        {
          CmConnectOwnerNumber++;
          strcpy (connect->owner, owner);
        }
    }
}

/*--------------------------------------------------------------*/
static Entry* GetFirst ()
/*--------------------------------------------------------------*/
{
  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (NULL);
    }

  return (CmConnectMgr.connects.first);
}


/*--------------------------------------------------------------*/
static CmConnect GetNextAlive (Entry** p_entry)
/*--------------------------------------------------------------*/
/*    Parcourt la base de donnees pour obtenir les connexions   */
/*    actives.                                                  */
/*--------------------------------------------------------------*/
{
  Entry* entry;

  CmConnect result = NULL;

  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (NULL);
    }

  if (p_entry == NULL) return (NULL);

  entry = *p_entry;
  if (entry == NULL) return (NULL);

  while (entry != NULL)
    {
      CmConnect connect;
      
      connect = (CmConnect) entry->reference;

      entry = entry->next;

      if ((connect != NULL) && (connect->alive)) 
        {
          result = connect;
          break;
        }
    }

  *p_entry = entry;

  return (result);
}

/*--------------------------------------------------------------*/
static CmConnect GetNext (Entry** p_entry)
/*--------------------------------------------------------------*/
/*    Parcourt la base de donnees pour obtenir les connexions.  */
/*--------------------------------------------------------------*/
{
  Entry* entry;

  CmConnect result = NULL;

  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (NULL);
    }

  if (p_entry == NULL) return (NULL);

  entry = *p_entry;
  if (entry == NULL) return (NULL);

  while (entry != NULL)
    {
      CmConnect connect;
      
      connect = (CmConnect) entry->reference;

      entry = entry->next;

      if (connect != NULL) 
        {
          result = connect;
          break;
        }
    }

  *p_entry = entry;

  return (result);
}

/*--------------------------------------------------------------*/
CmConnectIterator CmConnectGetIterator ()
/*--------------------------------------------------------------*/
{
  CmConnectIterator iterator;

  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (NULL);
    }

  iterator = (CmConnectIterator) malloc (sizeof (CmConnectIteratorRec));
  iterator->entry = CmConnectMgr.connects.first;
  iterator->forward = 1;

  return (iterator);
}

/*--------------------------------------------------------------*/
void CmConnectIteratorDelete (CmConnectIterator it)
/*--------------------------------------------------------------*/
{
  if (it == NULL) return;

  free (it);
}

/*--------------------------------------------------------------*/
CmConnect CmConnectGetNextAlive (CmConnectIterator it)
/*--------------------------------------------------------------*/
/*    Parcourt la base de donnees pour obtenir les connexions   */
/*    actives.                                                  */
/*--------------------------------------------------------------*/
{
  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (NULL);
    }

  if (it == NULL) return (NULL);

  return (GetNextAlive (&it->entry));
}

/*--------------------------------------------------------------*/
CmConnect CmConnectGetNext (CmConnectIterator it)
/*--------------------------------------------------------------*/
/*    Parcourt la base de donnees pour obtenir les connexions.  */
/*--------------------------------------------------------------*/
{
  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (NULL);
    }

  if (it == NULL) return (NULL);

  return (GetNext (&it->entry));
}

/*--------------------------------------------------------------*/
CmConnect CmConnectScan (CmConnectIterator it)
/*--------------------------------------------------------------*/
/*    Retourne la connection n'ayant pas de handler mais        */
/*    des donnees en attente.                                   */
/*--------------------------------------------------------------*/
{
  CmConnect result = NULL;
  Entry* entry;
  
  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (NULL);
    }

  if (it == NULL) return (NULL);

  entry = it->entry;
  if (entry == NULL) return (NULL);

  while (entry != NULL)
    {
      CmConnect connect;
      
      connect = (CmConnect) entry->reference;

      entry = entry->next;

      if ((connect != NULL) && 
          (connect->alive) &&
          (connect->data_pending) &&
          (connect->receive == NULL))
        {
          result = connect;
          break;
        }
    }

  it->entry = entry;

  return (result);
}

/*--------------------------------------------------------------*/
void CmConnectInstallReceiveHandler (CmConnect connect,
                                     CmConnectReceiveHandler receive)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;
  
  connect->receive = receive;
}

/*--------------------------------------------------------------*/
void CmConnectInstallRearmHandler (CmConnect connect,
                                   CmConnectRearmHandler rearm)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;
  
  connect->rearm = rearm;
}

/*--------------------------------------------------------------*/
void CmConnectInitData (CmConnect connect,
                        CmConnectReceiveHandler receive,
                        CmConnectRearmHandler rearm)
/*--------------------------------------------------------------*/
{
  printf ("Cm> This function is now obsolete.\n");
  printf ("Cm> Please use CmConnectInitHandlers instead.\n");
}

/*--------------------------------------------------------------*/
void CmConnectInitHandlers (CmConnect connect,
                            CmConnectReceiveHandler receive,
                            CmConnectRearmHandler rearm)
/*--------------------------------------------------------------*/
{
  CmConnect receiver;

  if (connect == NULL) return;

  receiver = CmConnectHandleGet (&connect->receiver);

  if (receiver != NULL)
    {
      CmConnectInitHandlers (receiver, receive, rearm);
    }
  else
    {
      connect->receive = receive;
      connect->rearm = rearm;
      connect->data_pending = CmConnectStatusNoData;
      connect->data_sending = CmConnectStatusNoData;
    }
}

/*--------------------------------------------------------------*/
void CmConnectInstallMessage (CmConnect connect, CmMessage message)
/*--------------------------------------------------------------*/
{
  CmConnect receiver;

  if (connect == NULL) return;

  receiver = CmConnectHandleGet (&connect->receiver);

  if (receiver != NULL)
    {
      CmConnectInstallMessage (receiver, message);
    }
  else
    {
      CmMessageHandleSet (&connect->message, message);
    }
}
  
/*--------------------------------------------------------------*/
void CmConnectInstallMessageIterator (CmConnect connect,
                                      CmMessageIterator it)
/*--------------------------------------------------------------*/
{
  CmConnect receiver;

  if (connect == NULL) return;
  if (it == NULL) return;

  receiver = CmConnectHandleGet (&connect->receiver);

  if (receiver != NULL)
    {
      CmConnectInstallMessageIterator (receiver, it);
    }
  else
    {
      CmMessageIterator current = CmIteratorHandleGet (&connect->iterator);

      if (current == NULL)
        {
          CmIteratorHandleSet (&connect->iterator, it);
          current = it;
        }
      else
        {
          ListAdd (&connect->iterators, it);
        }

      if (CmConnectDebug > 0)
        {
          DoPrint ("CmConnectInstallMessageIterator> it=%lx stack=%d\n",
                   current, ListCount (&connect->iterators));
        }
    }
}
  
/*--------------------------------------------------------------*/
void CmConnectInstallNoTimeOut (CmConnect connect)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;

  if (connect->timeout != NULL)
    {
      free (connect->timeout);
      CmConnectTimeoutNumber--;
      connect->timeout = NULL;
    }
}

/*--------------------------------------------------------------*/
void CmConnectInstallTimeOut (CmConnect connect, int Seconds, int MicroSeconds)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;

  if (connect->timeout == NULL)
    {
      connect->timeout = (struct timeval*) malloc (sizeof (struct timeval));
      if (connect->timeout == NULL)
        {
          DoPrint ("Cm> Malloc error on timeout.\n");
          return;
        }
      CmConnectTimeoutNumber++;
    }
  
  connect->timeout->tv_sec = Seconds;
  connect->timeout->tv_usec = MicroSeconds;
}

/*--------------------------------------------------------------*/
void CmConnectKill (CmConnect connect)
/*--------------------------------------------------------------*/
{
  CmConnect receiver;
  CmMessageIterator iterator;

  if (connect == NULL) return;
  /*--------------------------------- Modified by F. BELLACHIA */
  /* ----------> if (!connect->alive) return; <----------------*/
  if (connect->socket_id < 0) return;
  

  connect->alive = 0;

#ifndef CM_USE_SELECT
  if( connect->fds_id != -1 && CmConnectMgr.fds ) {
    struct pollfd *lFds;

    lFds = CmConnectMgr.fds + connect->fds_id;
    lFds->fd        = -1;
    lFds->events    =  0;
    connect->fds_id = -1;
    connect->events =  0; 
  }
#endif

  if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> Killing connect ");
      if (connect->name != NULL)
        {
          DoPrint ("%s", connect->name);
        }
      DoPrint ("\n");
    }
  
  connect->data_pending = CmConnectStatusNoData;
  connect->data_sending = CmConnectStatusNoData;

  iterator = CmIteratorHandleGet (&connect->iterator);
  if (iterator != NULL)
    {
      CmMessageIteratorStop (iterator);
      CmIteratorHandleClear (&connect->iterator);
    }

  if (ListCount (&connect->iterators) != 0)
    {
      CmMessageIterator it;

      while ((it = (CmMessageIterator) ListPopFirst (&connect->iterators)))
        {
          CmMessageIteratorStop (it);
        }
      ListClear (&connect->iterators);
    }
  
  if (connect->socket_id > 2)
    {
      TCPShutdown (connect->socket_id);
      TCPClose (connect->socket_id);

      connect->socket_id = -1;

      ConnectionHandlerStackCall (&(CmConnectMgr.disconnection_handlers),
                                  connect);

        /*
          if (CmConnectMgr.do_disconnect != NULL) 
          {
          CmConnectMgr.do_disconnect (connect);
          }
        */
    }

  receiver = CmConnectHandleGet (&connect->receiver);

  if (receiver != NULL)
    {
      CmConnectKill (receiver);
      CmConnectHandleClear (&connect->receiver);
    }

  if (connect->local)
    {
#if defined (CM_ANY_UNIX) || defined (CM_LINUX)
      if( connect->local_read > 1 ) close (connect->local_read);
      if( connect->local_read > 1 ) close (connect->local_write);
#endif
      connect->local_read  = -1;
      connect->local_write = -1;
    }

  if (CmConnectGetServer (connect) != NULL) CmConnectSetServer (connect, NULL);

  /* An incomplete message may be stored in connect->message. Delete it. */
  CmMessageDelete (CmMessageHandleGet (&connect->message));
  CmMessageHandleClear (&connect->message);

  if (ListCount (&connect->messages) != 0)
    {
      CmMessage m;
      Entry* entry;

      entry = connect->messages.first;

      while (entry != NULL)
        {
          m = (CmMessage) entry->reference;
          if (m != NULL) CmMessageDelete (m);

          entry = entry->next;
        }
      ListClear (&connect->messages);
    }
}

/*--------------------------------------------------------------*/
static int CmConnectReset (CmConnect connect)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return (0);
  if (!connect->alive) return (0);
  
  connect->alive = 0;

#ifndef CM_USE_SELECT
  if( connect->fds_id != -1 && CmConnectMgr.fds ) {
    struct pollfd *lFds;

    lFds = CmConnectMgr.fds + connect->fds_id;
    lFds->fd        = -1;
    lFds->events    =  0;
    connect->fds_id = -1;
    connect->events =  0;
  }
#endif

  if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> Resetting connect ");
      if (connect->name != NULL)
        {
          DoPrint ("%s", connect->name);
        }
      DoPrint ("\n");
    }
  
  connect->data_pending = CmConnectStatusNoData;
  connect->data_sending = CmConnectStatusNoData;

  if (connect->socket_id >= 2 )
    {
      TCPShutdown (connect->socket_id);
      TCPClose (connect->socket_id);

      connect->socket_id = -1;
    }

  connect->socket_id = TCPSocket ();
  if (connect->socket_id < 0)
    {
      DoPrint ("Cm> Socket re-creation error.\n");
    }
  else
    {
      CmConnectSetSocketProperties( connect, -1, -1 );
      /* connect->alive = 1; */

#ifdef CM_OS9
      if (Connect (connect, connect->port) == -1)
#else
      if (Connect (connect, connect->port))
#endif
        {
          if (CmConnectDebug > 0)
            {
              DoPrint ("Cm> Re-connect error.\n");
            }
        }
      else
        {
          connect->alive = 1;
          
          ConnectionHandlerStackCall (&(CmConnectMgr.connection_handlers),
                                      connect);
            /*
          if (CmConnectMgr.do_connect != NULL) 
            {
              CmConnectMgr.do_connect (connect);
            }
            */

          return (1);
        }
    }
  
  return (0);
}

/*  Private functions ------------------------------------------*/

/*--------------------------------------------------------------*/
static void CmConnectSetServer (CmConnect connect, CmConnect server)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;

  CmConnectHandleSet (&connect->server, server);
}

/*--------------------------------------------------------------*/
static void CmConnectSetComment (CmConnect connect, char* comment)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;

  if (connect->comment != NULL)
    {
      free (connect->comment);
      connect->comment = NULL;
    }

  if ((comment != NULL) &&
      (strlen (comment) > 0))
    {
      connect->comment = (char*) malloc (strlen (comment) + 1);
      strcpy (connect->comment, comment);
    }
}

/*--------------------------------------------------------------*/
char* CmConnectGetComment (CmConnect connect)
/*--------------------------------------------------------------*/
{
  char* comment;

  if (connect == NULL) return (NULL);

  comment = connect->comment;
  if (comment == NULL) comment = (char*) "";

  return (comment);
}

/*--------------------------------------------------------------*/
void CmConnectGetStatistics (CmConnect connect, 
                             CmConnectStatisticsRec* buffer)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;
  if (buffer == NULL) return;

  buffer->sent           = connect->statistics.sent;
  buffer->bytes_sent     = connect->statistics.bytes_sent;
  buffer->received       = connect->statistics.received;
  buffer->bytes_received = connect->statistics.bytes_received;
}

/*--------------------------------------------------------------*/
static int CmConnectGetPortAndName (CmConnect connect)
/*--------------------------------------------------------------*/
/*                                                              */
/*  This function is used by a new server to get its actual     */
/* configuration from the NameServer.                           */
/*                                                              */
/*--------------------------------------------------------------*/
{
  int port = 0;

    /*
      The answer about the port and name for this connection will
      arrive straight in the NSPort Message handler.

      In this handler, the port (and possibly a new name) will be setup.

      a Break will signal about the correct execution.
      */

  WaitForTransaction ("CmConnectGetPortAndName", "NSPort", connect->transaction, CmNameServerTimeout);

  CmCloseTransaction (connect->transaction);

  port = CmConnectGetPort (connect);

  return (port);
}

/*--------------------------------------------------------------*/
static int CmConnectGetNewPort (CmConnect connect)
/*--------------------------------------------------------------*/
/*                                                              */
/*  This function is used by a new server to get its actual     */
/* configuration from the NameServer and when the first try     */
/* did not succeed. (this port number is already in use and     */
/* the NameServer did not know it !)                            */
/*                                                              */
/*--------------------------------------------------------------*/
{
  int port = 0;
  char* name_server_name;
  CmMessage message;
  int status;
  int transaction;

  transaction = CmOpenTransaction ("Cm> get new port", NULL);

    /*
      Temporary solution for maintaining backward compatibility
      with non-transaction based protocol
      */
  CurrentGetNewPortTransaction = transaction;

  message = CmMessageNew ();
  CmMessageSetType (message, "NSGetNewPort");
  CmMessagePutText (message, CmConnectGetName (connect));
  CmMessagePutInt  (message, transaction);

  name_server_name = CmConnectGetName (CmConnectGetServer (connect));
  
  status = CmMessageSend (message, name_server_name);
  CmMessageDelete (message);
  
  if (!status)
    {
      CmCloseTransaction (transaction);
      DoPrint ("Cm> Bad connection with NameServer.\n");
      CmConnectExit (CmErrorNoNameServer);
      return (0);
    }

    /*
      The answer about the port and name for this connection will
      arrive straight in the NSNewPort Message handler.

      In this handler, the port (and possibly a new name) will be setup.

      a Break will signal about the correct execution.
      */

  WaitForTransaction ("CmConnectGetNewPort", "NSNewPort", transaction, CmNameServerTimeout);

  CmCloseTransaction (transaction);

  port = CmConnectGetPort (connect);

  return (port);
}

/*--------------------------------------------------------------*/
static int CmConnectSendServerInfos (CmConnect connect)
/*--------------------------------------------------------------*/
{
  char* version;
  char* name;
  char* host;
  char* owner;
  CmConnect server;
  CmMessage message;
  int status;

  if (connect == NULL) return (0);
  
  server = CmConnectGetServer (connect);
  
  if (server == NULL) return (0);
  
  version  = CmConnectMgr.version;
  name     = CmConnectGetName (server);
  host     = CmConnectGetHost (server);
  owner    = CmConnectGetOwner (server);
  
  if (version  == NULL) version  = (char*) "<unknown>";
  if (name     == NULL) name     = (char*) "<unknown>";
  if (host     == NULL) host     = (char*) "<unknown>";
  if (owner    == NULL) owner    = (char*) "<unknown>";

  message = CmMessageNew ();
  CmMessageSetType (message, "CmServerInfos");
  CmMessagePutText (message, version);
  CmMessagePutText (message, name);
  CmMessagePutText (message, host);
  CmMessagePutInt  (message, server->port);
  CmMessagePutInt  (message, server->multiple);
  CmMessagePutText (message, owner);

  status = CmMessageSend (message, CmConnectGetName (connect));
  CmMessageDelete (message);

  return (status);
}

/*--------------------------------------------------------------*/
static CmConnect CreateNewConnect (char* name,
                                   char* host_name,
                                   int port)
/*--------------------------------------------------------------*/
/*  This function initializes the Connect object so that it     */
/* is in a correct state for private attributes.                */
/*                                                              */
/*  Nothing is done about a TCPIP configuration (no call to     */
/* TCPxxx is done there)                                        */
/*                                                              */
/*--------------------------------------------------------------*/
{
  CmConnect connect;

  connect = (CmConnect) calloc (1, sizeof(CmConnectRec));
  CmConnectNumber++;
  
  if (connect == NULL)
    {
      DoPrint ("Cm> Malloc error on connect.\n");
      return (NULL);
    }

  connect->name          = NULL;
  connect->comment       = NULL;

  connect->statistics.sent           = 0;
  connect->statistics.received       = 0;
  connect->statistics.bytes_sent     = 0;
  connect->statistics.bytes_received = 0;

  connect->receive       = NULL;
  connect->rearm         = NULL;
  connect->data_pending  = CmConnectStatusNoData;
  connect->locked        = 0;
  connect->data_sending  = CmConnectStatusNoData;
  connect->host_name     = NULL;
  connect->owner         = NULL;
  CmConnectHandleInstall (&connect->server);
  CmConnectHandleInstall (&connect->receiver);
  connect->local_read    = -1;
  connect->local_write   = -1;
  connect->local         = 0;
  connect->alive         = 1;
  connect->multiple      = 0;
  connect->timeout       = NULL;
  connect->socket_id     = -1;
  connect->port          = port;
  connect->fds_id        = -1;
  connect->events        = 0;
  CmMessageHandleInstall (&connect->message);
  ListInit (&connect->messages);
  CmIteratorHandleInstall (&connect->iterator);
  ListInit (&connect->iterators);
  
  ListAdd (&CmConnectMgr.connects, connect);

  CmConnectSetName (connect, name);
  CmConnectSetHost (connect, host_name);
  CmConnectSetOwner (connect, getenv("USER"));
  CmConnectInstallTimeOut (connect, 20, 0);
/*
  CmConnectInstallNoTimeOut (connect);
*/
  
  return (connect);
}  

/*--------------------------------------------------------------*/
static void CmConnectClear (CmConnect connect)
/*--------------------------------------------------------------*/
/*    Nettoyage final d'une connexion.                          */
/*    o Fermeture de la socket.                                 */
/*    o Destruction des composants.                             */
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;

  if (connect->timeout != NULL)
    {
      free (connect->timeout);
      CmConnectTimeoutNumber--;
      connect->timeout = NULL;
    }
  if (connect->name != NULL)
    {
      free (connect->name);
      CmConnectNameNumber--;
      connect->name = NULL;
    }
  if (connect->comment != NULL)
    {
      free (connect->comment);
      connect->comment = NULL;
    }
  if (connect->host_name != NULL)
    {
      free (connect->host_name);
      CmConnectHostNameNumber--;
      connect->host_name = NULL;
    }
  if (connect->owner != NULL)
    {
      free (connect->owner);
      CmConnectOwnerNumber--;
      connect->owner = NULL;
    }

  CmIteratorHandleClear (&connect->iterator);
  ListClear (&connect->iterators);
}

/*--------------------------------------------------------------*/
static void CmConnectDelete (CmConnect connect)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;

  CmConnectClear (connect);
  CmConnectHandleUninstall (&connect->server);
  CmConnectHandleUninstall (&connect->receiver);
  CmIteratorHandleUninstall (&connect->iterator);
  CmMessageHandleUninstall (&connect->message);

  CmConnectHandleClearAll (connect);
  ListRemove (&CmConnectMgr.connects, connect);
  
  connect->fds_id = -1;
  connect->events = 0;
 
  free (connect);
  CmConnectNumber--;
}





/*--------------------------------------------------------------*/
/*                                                              */
/*  Event wait operations.                                      */
/*                                                              */
/*--------------------------------------------------------------*/





/*--------------------------------------------------------------*/
CmConnectCondition CmConnectCheck ()
/*--------------------------------------------------------------*/
{
  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (CmConnectErrorCondition);
    }

  return (CheckOrWait (1, NULL, NULL, "CmConnectCheck"));
}

/*--------------------------------------------------------------*/
CmConnectCondition CmConnectWait ()
/*--------------------------------------------------------------*/
{
  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (CmConnectErrorCondition);
    }

  return (CheckOrWait (0, NULL, NULL,  "CmConnectWait"));
}

/*--------------------------------------------------------------*/
CmConnectCondition CmConnectWaitOnIterator (CmMessageIterator it)

/*--------------------------------------------------------------*/
{
  struct timeval timeout;
  CmConnectCondition result;

#ifdef CM_OS9
  static int seconds = 1;
#endif
  
  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (CmConnectErrorCondition);
    }

#ifdef CM_OS9
  timeout.tv_sec = seconds;
  timeout.tv_usec = 0;

  result = CheckOrWait (0, &timeout, it, "CmConnectWaitOnItertor");
#else
  timeout.tv_sec = 20;
  timeout.tv_usec = 0;

  result = CheckOrWait (0, &timeout, it, "CmConnectWaitOnItertor");
#endif

  return (result);
}

/*--------------------------------------------------------------*/
CmConnectCondition CmConnectWaitWithTimeout (double seconds)
/*--------------------------------------------------------------*/
{
  struct timeval timeout;
  
  if (!CmConnectInitialized)
    {
      DoPrint ("Cm> Cm not initialized.\n");
      return (CmConnectErrorCondition);
    }

  timeout.tv_sec = (int) seconds;
  timeout.tv_usec = (int) ((seconds - (int) seconds) * 1000000);

  return (CheckOrWait (0, &timeout, NULL, "CmConnectWaitWithTimeout"));
}

/*--------------------------------------------------------------*/
CmConnectStatus CmConnectCheckInputData (CmConnect connect)
/*--------------------------------------------------------------*/
{
  int bytesRead;
  CmConnectStatus result = CmConnectStatusNoData;
  CmConnect receiver;
  
  connect->data_pending = CmConnectStatusNoData;

  if (connect->local)
    {
      result = CmConnectStatusData;
    }
  else
    {
      receiver = CmConnectHandleGet (&connect->receiver);

      bytesRead = TCPTest (connect->socket_id);
      if ((bytesRead) > 0 || (receiver != NULL))
        {
          result = CmConnectStatusData;
        }
      else if (bytesRead < 0)
        {
          char* name = CmConnectGetName (connect);
          char* host = CmConnectGetHost (connect);


          if (name == NULL) name = "<null>";
          if (host == NULL) host = "<null>";

          DoPrint ("CmConnectCheckInputData> Status %d errno=%d "
		   "- Connect %s host %s", bytesRead, errno,  name, host);
	  if( errno == ECONNRESET ) {
	    DoPrint( " - performs a CmConnectKill\n" );
	    CmConnectKill (connect);
	  } else { 
	    DoPrint( " - performs a CmConnecReset\n" );
	    CmConnectReset (connect);
	  }
        }
      else
        {
            /* connect connection is closing... */
          
          if (CmConnectDebug > 0)
            {
              char* name = CmConnectGetName (connect);

              if (name == NULL) name = "<null>";

              DoPrint ("CmConnectCheckInputData> Connect %s closing\n", name);
            }

          CmConnectKill (connect);
        }
    }

  return (result);
}

/*  Private functions ------------------------------------------*/

/*--------------------------------------------------------------*/
static void CmConnectPopMessageIterator (CmConnect connect)
/*--------------------------------------------------------------*/
{
  CmMessageIterator iterator;

  if (connect == NULL) return;

  iterator = CmIteratorHandleGet (&connect->iterator);

  if (iterator != NULL)
    {
      CmIteratorHandleClear (&connect->iterator);
    }
  
  if (ListCount (&connect->iterators) > 0)
    {
      CmMessageIterator it;
      
      it = (CmMessageIterator) ListFirst (&connect->iterators);
      if (it != NULL)
        {
          CmIteratorHandleSet (&connect->iterator, it);
          ListPopFirst (&connect->iterators);
        }
     }

  if (CmConnectDebug > 0)
    {
      iterator = CmIteratorHandleGet (&connect->iterator);
      DoPrint ("CmConnectPopMessageIterator> it=%lx stack=%d\n",
               iterator, ListCount (&connect->iterators));
    }
}
  
/*--------------------------------------------------------------*/
static void CmConnectLock (CmConnect connect)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;

  connect->locked++;
}

/*--------------------------------------------------------------*/
static void CmConnectUnlock (CmConnect connect)
/*--------------------------------------------------------------*/
{
  if (connect == NULL) return;

  connect->locked--;
}

/*--------------------------------------------------------------*/
static CmConnectCondition CallHandlers ()
/*--------------------------------------------------------------*/
{
  CmConnect connect;
  CmConnectCondition condition = CmConnectNormalCompletion;
  CmMessageIterator message_it;

  Entry* entry = GetFirst ();

  while( (connect = GetNextAlive (&entry)) )
    {
      if ((connect->locked) && 
          (connect->data_pending == CmConnectStatusData))
        {
          if (CmConnectDebug > 0)
            {
              DoPrint ("Cm CallHandlers> connect %s locked(%d)\n",
                       CmConnectGetName (connect),
                       connect->locked);
            }
        }

      if ((connect->locked < 1) && 
          (connect->data_pending == CmConnectStatusData))
        {
          connect->data_pending = CmConnectStatusNoData;
          if (connect->receive != NULL)
            {
              CmConnectLock (connect);
              connect->data_pending = connect->receive (connect);
              CmConnectUnlock (connect);

              if (connect->data_pending == CmConnectStatusAbort)
                {
                  /* Break in handler */
                  condition = CmConnectBreakDetection;
                  break;
                }
              
              message_it = CmIteratorHandleGet (&connect->iterator);
              if ((connect->alive == 0) &&
                  (message_it != NULL))
                {
                  CmMessageIteratorStop (message_it);
                }
            }
          else
            {
              /* No handler for this connection */
              condition = CmConnectNoHandler;
              break;
            }
        }

      if (connect->alive == 1)
        {
          message_it = CmIteratorHandleGet (&connect->iterator);
          if ((message_it != NULL) &&
              CmMessageIteratorFinished (message_it))
            {
              CmMessageStatus status;

              connect->statistics.sent++;

              status = CmMessageIteratorCallHandler (message_it);
          
              CmConnectPopMessageIterator (connect);

              if (status == CmMessageBreak)
                {
                  condition = CmConnectBreakDetection;
                  break;
                }
            }
        }
    }

  return (condition);
}

/*--------------------------------------------------------------*/
static void PurgeConnects ( )
/*--------------------------------------------------------------*/
{
  CmConnect connect;
  Entry* entry = GetFirst ();
 
  while ((connect = GetNext (&entry))) {
/*     if (connect->data_pending == CmConnectStatusDelayed) { */
/*       connect->data_pending = CmConnectStatusData; */
/*     } */

    if(connect->alive) continue;
    /*--------------------------------------------------------*/

        
    if (CmConnectDebug > 0) {
	DoPrint ("Cm PurgeConnects> Killing dead connect %s\n",
		 CmConnectGetName (connect));
    }


    CmConnectKill (connect);
    CmConnectDelete (connect);
 }
#ifndef CM_USE_SELECT
  /*-----------------------------------------------------------*/
  {
    int i;
   
   if( (CmConnectMgr.nfds * 4/3 ) > CmConnectMgr.maxFds ) {
      DoPrint("Cm> PurgeConnect>  nfds %d - max %d \n",
	      CmConnectMgr.nfds, CmConnectMgr.maxFds );
      CmConnectMgr.fds = (struct pollfd *) realloc( CmConnectMgr.fds,
						    2*CmConnectMgr.maxFds*sizeof(struct pollfd ) );
      for(  i = CmConnectMgr.maxFds; i < 2*CmConnectMgr.maxFds; i++ ) {
	CmConnectMgr.fds[i].fd      = -1;
	CmConnectMgr.fds[i].events  = 0;
	CmConnectMgr.fds[i].revents = 0;
      }
      if( !CmConnectMgr.fds ) { 
	DoPrint("Cm> PurgeConnect>  malloc failed - %d -%d\n", 
		2*CmConnectMgr.maxFds,
		      2*CmConnectMgr.maxFds*sizeof(struct pollfd ) );
	exit(-1);
      }
      DoPrint("Cm> PurgeConnectt>  maxFds increase from %d to %d\n",
	      CmConnectMgr.maxFds, CmConnectMgr.maxFds*2);
      CmConnectMgr.maxFds *= 2;
    }
  }
#
#endif
}

#ifndef CM_USE_SELECT
/*--------------------------------------------------------------*/
static int FillMasksPoll ( int *nfds )
/*--------------------------------------------------------------*/
{
  int result = 0, lFd, i, type;
  CmConnect connect;
  Entry* entry;
  struct pollfd *fds;
  
  if( !(fds = CmConnectMgr.fds) ) return(result);
  /*------------------------------------------------------------*/
  for(  lFd = -1,  type = i = 0, entry = GetFirst ();
	(connect = GetNextAlive (&entry));
	lFd = -1, type = 0 ) {
    CmConnectStatus s;
    
    /*---------------------------------------- read access */
    if ((connect->data_pending == CmConnectStatusNoData) &&
	(connect->rearm != NULL))
      {
	connect->data_pending = connect->rearm (connect);
      }
    
    s = connect->data_pending;
    if (s == CmConnectStatusData)
      {
	result++;
      }
    else if ((s == CmConnectStatusNoData) ||
	     (s == CmConnectStatusAbort))
      {
#if defined (CM_ANY_UNIX) || defined (CM_LINUX)
	if (connect->local)
	  {
	    lFd = connect->local_read ;
	    type = POLLIN;
	   /*------------------------------------------------------*/
	    connect->fds_id = i;
	    fds[i].fd       = lFd;
	    fds[i].events   = type;
	    i++;

	    continue;
	  }
	else
#endif
	  {
	    if (connect->locked < 1)
	      {
		lFd   = connect->socket_id;
		type |= POLLIN;
	      }
	  }
      }
    
    /*---------------------------------------- write access */
    if (CmIteratorHandleGet (&connect->iterator) != NULL)
      {
	lFd   = connect->socket_id;
	type |= POLLOUT;
      }
    /*------------------------------------------------------*/   
    if( lFd != -1 ) { 
      connect->fds_id = i;
      connect->events = type;
      fds[i].fd       = lFd;
      fds[i].events   = type;
      i++;
    } else if( connect->fds_id != -1 ) {
      if( connect->locked < 1 && (type = connect->events) ) {
	DoPrint("Cm> FillMasksPoll> WARNING name %s - port %d - locked %d "
		"- fds_id %d/%d - fd %d - events %d\n",
		connect->name, connect->port, connect->locked,
		connect->fds_id, i,
		connect->socket_id, type );
	connect->fds_id = i;
/* 	connect->events = type; */
	fds[i].fd       = lFd;
	fds[i].events   = type;
	i++;
      } else {
	connect->fds_id = -1;
	connect->events =  0;
      }
    }
  }
	
  /*-------------------------------------- Update the field */
  CmConnectMgr.nfds =  nfds[0] = i;
  /* DoPrint("Cm> FillMasksPoll> End - result %d - nfds %d/%d\n", result, nfds[0], */
  /* 	  CmConnectMgr.maxFds); */
  return (result);
}


/*--------------------------------------------------------------*/
static int CheckMasksPoll ( int nfds, int requested )
/*--------------------------------------------------------------*/
{
  CmConnect connect;
  int read, read_pollhup, write, done, write_ready, write_pollerr, id;
  Entry* entry;
  struct pollfd *lFds;
  struct pollfd *fds;

  if( !(fds = CmConnectMgr.fds) ) return(0);
  /*-------------------------------------------------------------*/
  /*
    First look only for all pending send operations.
  */
  read = read_pollhup = write = done = write_ready = write_pollerr =  0;

  for( entry = GetFirst (); (connect = GetNextAlive (&entry));   ) {
    
    if( (id = connect->fds_id) == -1 ) continue;
    /*------------------------------------------------------------------*/

    lFds = fds + id;
    /*---------------------------------- Performs the writing if needed */
    if( ( lFds->revents & POLLERR) ) {
      DoPrint ( "Cm> CheckMasksPoll> %s - POLLERR\n", connect->name );
      CmConnectKill (connect);
      done++; write_pollerr++;
    } else if( (lFds->revents & POLLOUT) ) {
      /*--------------------------------------------- Perform the action */
      CmMessageIterator it = CmIteratorHandleGet (&connect->iterator);
      if (it != NULL) {
	CmMessageIteratorForth (it);
	write_ready++;
      }
      write++, done++;
    }
    
  }
  
  if( write != write_ready  ) 
    DoPrint ( "Cm> CheckMasksPoll> WARNING Write requested %d/%d "
	      "- write %d/%d - pollerr %d\n",
              done, requested, write_ready, write, write_pollerr );
    
  if (write_ready > 0)  return (0);

 /*
    look for the all receive  operations.
    */
  
  for( entry = GetFirst (); (connect = GetNextAlive (&entry)) ;  ) {
    
    if( (id = connect->fds_id) == -1 ) continue;
    lFds = fds + id;
    /*------------------------------ Tag that data are ready to be read */
    if( (lFds->revents & POLLIN) ) {
      connect->data_pending = CmConnectStatusData;
      /*---------------------------------- Perform the action */
      done++; read++;
    } else if( (lFds->revents & POLLHUP) ) {
      DoPrint ( "Cm> CheckMasksPoll> %s - POLLHUP\n", connect->name );
      /*---------------------------------- Perform the action */
      CmConnectKill (connect);
      done++; read_pollhup++;
    } 
  }
  if (requested != done )  {
    DoPrint ( "Cm> CheckMasksPoll> WARNING requested %d/%d -"
	      " read %d, write %d - read_pollhup %d, write_pollerr %d\n",
              done, requested, read, write, read_pollhup, write_pollerr );
  }
  return (0);
}


#else 

/*--------------------------------------------------------------*/
static int FillMasksSelect (  int *nfds, fd_set* rmask, fd_set* wmask )
/*--------------------------------------------------------------*/
{
  int result = 0, maxFd, lFd ;
  CmConnect connect;
  Entry* entry = GetFirst ();
  
  FD_ZERO (rmask);
  FD_ZERO (wmask);

  maxFd = lFd = 0;
  while ((connect = GetNextAlive (&entry)))
    {
      CmConnectStatus s;
        
      /*---------------------------------------- read access */
      if ((connect->data_pending == CmConnectStatusNoData) &&
	  (connect->rearm != NULL))
	{
	  connect->data_pending = connect->rearm (connect);
	}
      
      s = connect->data_pending;
      if (s == CmConnectStatusData)
	{
	  result++;
	}
      else if ((s == CmConnectStatusNoData) ||
	       (s == CmConnectStatusAbort))
	{
#if defined (CM_ANY_UNIX) || defined (CM_LINUX)
	  if (connect->local)
	    {
	      FD_SET (connect->local_read, rmask);
	      lFd = connect->local_read;
	    }
	  else
#endif
	    {
	      if (connect->locked < 1)
		{
		  FD_SET (connect->socket_id, rmask);
		  lFd = connect->socket_id;
		}
	    }
	}
      /*------------------------------------------------------*/
      if( lFd > maxFd ) maxFd = lFd;
 
      /*---------------------------------------- write access */
      if (CmIteratorHandleGet (&connect->iterator) != NULL)
	{
	  FD_SET (connect->socket_id, wmask);
	  lFd = connect->socket_id;
	}
      
      /*------------------------------------------------------*/
      if( lFd > maxFd ) maxFd = lFd;
      
    }

  /* BMX */
  FD_CLR (0,rmask);
  if( nfds ) nfds[0] = maxFd + 1;
  return (result);
}

/*--------------------------------------------------------------*/
static int CheckMasksSelect (fd_set* rmask, fd_set* wmask, int requested)
/*--------------------------------------------------------------*/
{
  CmConnect connect;
  int read, write, done, write_ready, lFd;
  Entry* entry;

  /*
    First look only for all pending send operations.
    */
  read = write = done = write_ready = 0;
  
  for( entry = GetFirst (); (connect = GetNextAlive (&entry));   ) {

#if defined (CM_ANY_UNIX) || defined (CM_LINUX)
    lFd = ( connect->local ? connect->local : connect->socket_id );
# else
    lFd = connect->socket_id;
#endif
         
    /*---------------------------------- Performs the writing if needed */
    if (FD_ISSET (lFd, wmask)) {
      CmMessageIterator it = CmIteratorHandleGet (&connect->iterator);
      if (it != NULL) {
	CmMessageIteratorForth (it);
	write_ready++;
      }
      write++, done++;
    }
  }

  if( write != write_ready ) 
    DoPrint ( "Cm> CheckMasksSelect Write > requested %d/%d - write %d/%d\n",
              done, requested, write_ready, write );
    
  if (write_ready > 0)  { return (0); }

 /*
    look for the all receive  operations.
    */

  for( entry = GetFirst (); (connect = GetNextAlive (&entry)) ;  ) {
#if defined (CM_ANY_UNIX) || defined (CM_LINUX)
    lFd = ( connect->local ? connect->local : connect->socket_id );
# else
    lFd = connect->socket_id;
#endif
    /*------------------------------ Tag that data are ready to be read */
    if (FD_ISSET (connect->socket_id, rmask) )  {
      connect->data_pending = CmConnectStatusData;
      done++; read++; 
    }
  }
  if (requested != done ) 
    DoPrint ( "Cm> CheckMasksSelect> requested %d/%d - read %d, write %d\n",
              done, requested, read, write );
  
  return (0);
}

#endif 

/*--------------------------------------------------------------*/
 static void PrepareForWait ()
/*--------------------------------------------------------------*/
{
  CmConnect connect, me;
  Entry* entry = GetFirst ();
  char txt[256];

  me = CmConnectWhoAmI();
  if (CmConnectIsNameServer (me)) { /*----------- CmNameServer */
    while ( (connect = GetNextAlive (&entry)) ) {
      if( connect == me || connect->name ) continue;
      if( !connect->port ) continue; /*--- connection in progress */      
      /*----------------------------------- NULL CmConnect name */
      CmConnectSnprint( connect, txt, sizeof txt );
      DoPrint ("Cm>PrepareForWait> NULL CmConnect found: Exit - %s\n", txt);
      /*-------------- Cleanup and Exit */
      CmConnectExit( CmErrorRecovered );
    }
  } else { /*-------------------------------- Others Cm servers */
    while ((connect = GetNextAlive (&entry))) {
      if( connect == me || connect->name || connect->host_name  ) continue;
      /*----------------------------------- NULL CmConnect name */
      CmConnectSnprint( connect, txt, sizeof txt );
      DoPrint ("Cm>PrepareForWait> NULL CmConnect found: Kill - %s\n", txt);
      CmConnectKill( connect );
      CmConnectDelete( connect );
    }    
  }
  
  /*-----------------------------------------------------------*/
  return;
}


/*--------------------------------------------------------------*/
static CmConnectCondition CheckOrWait (int check_only, 
                                       struct timeval* big_timeout,
                                       CmMessageIterator it, char *tag)
/*--------------------------------------------------------------*/
{
#ifdef CM_USE_SELECT 
  fd_set                read_mask;
  fd_set                write_mask;
#endif 
  int                   pending_connects;
  int                   new_connects;
  struct timeval*       timeout_ref;
  struct timeval        lTimeout;
  struct timeval        null_timeout;
  int                   waiting = 0;
  CmConnectCondition    condition;
  
  int                   done = 0;
  int                   count = 0;
  int                   nfds;

  static int level = 0;
  static int transaction = 0;

  level++;
  if (transaction == 0)
    {
      transaction = CmOpenTransaction ("Cm", NULL);
    }

  null_timeout.tv_sec = 0;
  null_timeout.tv_usec = 0;

  /* Needed for Linux due to the fact that on LINUX the wkTimeout is modified */
  /* Refer to the select manual on LINUX platform ----------------------------*/
  if( !big_timeout ) {
    /*timeout_ref = big_timeout; ----------- i.e : set to struct timeval NULL */
    timeout_ref = NULL;
  } else {
    lTimeout.tv_sec  = big_timeout->tv_sec;
    lTimeout.tv_usec = big_timeout->tv_usec;
    timeout_ref      = &lTimeout;
  }

  if( level == 1 )  {
    PrepareForWait (); 
    PurgeConnects ();
  }

/*     DoPrint ("Cm>CheckOrWait> level %d - check_only %d - timeout %p," */
/* 	     " CmIterator %p , tag %s\n", */
/* 	     level, check_only, big_timeout, it, tag ); */

  CmRestartTransaction (transaction);

  count = CmTerminatedTransactionCount ();
  
  condition = CmConnectErrorCondition;
  for( nfds = 0;  (!done); nfds = 0 ) 
    {
      condition = CallHandlers ();


      if (condition == CmConnectBreakDetection)
        {
          CmTerminateTransaction (transaction);
        }


      if (CmConnectDebug > 1)
        {
          CmDumpTransactions ();
          DoPrint ("Cm> count = %d terminated = %d\n", count, 
                   CmTerminatedTransactionCount ());
        }

      if (CmTerminatedTransactionCount () > count)
        {
          condition = CmConnectBreakDetection;
        }
      
      if ((it != NULL) ) { /*-------------------- Iterator from CmMessageSend */
	if( CmMessageIteratorFinished (it))
	  {
	    condition = CmConnectNormalCompletion;
	    /* Go out of the loop with CmConnetNormalCompletion as return code */
	    break;
	  }
      } else if ((condition == CmConnectBreakDetection) || 
		 (condition == CmConnectNoHandler)) break;

      /* Nested call of CheckOrWait - As consequence the PurgeConnect should */
      /* be done only at the first level ------------------------------------*/
      if (level == 1) PurgeConnects();

#ifdef CM_USE_SELECT 
      pending_connects = FillMasksSelect (&nfds, &read_mask, &write_mask );
# else 
      pending_connects = FillMasksPoll( &nfds );
#endif
       


      if (pending_connects || check_only) /*------------------- For polling */
	{
          timeout_ref = &null_timeout;
          waiting = 0;
        }
      else
        {
          if( timeout_ref )
	    {
	      timeout_ref = &lTimeout;
	      waiting = 1;
	    }
        }
#ifdef CM_USE_SELECT 
     new_connects = TCPSelect ( nfds, &read_mask, &write_mask, NULL,
				timeout_ref);
#else
     new_connects = TCPPoll( nfds, timeout_ref );

#endif

      if (new_connects == -1)
        {
#ifdef CM_USE_SELECT 
          DoPrint ("Cm> Select error: errno %d\n", errno );
#else
          DoPrint ("Cm> Poll error: errno %d\n", errno );
#endif
          condition = CmConnectErrorCondition;
          done = 1;
        }
      else if (new_connects == 0)
        {
          /* Timeout */
          if (waiting)
            {
              /* Timeout condition */
              condition = CmConnectTimeoutDetection;
              done = 1;
            }
	  else if( check_only )
	    {
	      /* Normal completion for CmConnectCheck */
	      condition = CmConnectNormalCompletion;
	      done = 1;
	    }
        }
      else 
	{
	  /*---------------------------------- CheckMasks always returns 0 */
#ifdef CM_USE_SELECT 
         CheckMasksSelect (&read_mask, &write_mask, new_connects);
#else
	 CheckMasksPoll( nfds, new_connects );
#endif
        }


    }
  
/*   if( level != 1  )  */
/*     DoPrint("Cm> CheckOrWait> level %d - tag %s - End\n", level, tag ); */

  level--;
 
   return (condition);
}






/*--------------------------------------------------------------*/
/*                                                              */
/*  Handlers                                                    */
/*                                                              */
/*--------------------------------------------------------------*/


/*  Private functions ------------------------------------------*/

/*--------------------------------------------------------------*/
static CmMessageStatus NSAddressHandler (CmMessage message)
/*--------------------------------------------------------------*/
/*  Handler triggered by a 'NSAddress' message.                 */
/*                                                              */
/*  This message is an answer to the 'NSGetAddress' message     */
/* sent to the name server (from the CmConnectNew function)     */
/*                                                              */
/*  The connection to which this message is sent is the         */
/* connection to the NS owned by the server that made this      */
/* request.                                                     */
/*  Therefore the connection that will be created there         */
/* will be created in the context of that same server.          */
/*                                                              */
/*  Expected received syntax :                                  */
/*                                                              */
/*    char* name              Connection name                   */
/*                                                              */
/*    char* host              host name of the Connect          */
/*    int port                port number of the Connect        */
/*    bool is_multiple                                          */
/*    char* owner             owner of the Connect              */
/*                                                              */
/*  [ int transaction         transaction id for the request ]  */
/*                                                              */
/*--------------------------------------------------------------*/
{
  char* name;
  int port;
  char* host;
  int multiple;
  char* owner;

  name = CmMessageGetText (message);

  if ( name && name[0] && 
       CmMessageGetItemType (message) == CmMessageItemText)
    {
      CmConnect connect;
      CmConnect server;

      host     = CmMessageGetText (message);
      port     = CmMessageGetInt  (message);
      multiple = CmMessageGetInt  (message);
      owner    = CmMessageGetText (message);
  
      connect = CmMessageGetConnect (message);
      server = CmConnectGetServer (connect);
     
        /* On a suffisamment d'infos pour cree la connection. */

      connect = CmConnectGetReference (name);
      if (connect == NULL)
        {
          connect = CmConnectNewWithAddress (server, name, port, host);
          CmConnectSetOwner (connect, owner);
          if (multiple) CmConnectSetMultiple (connect);
        }
    }
/*   else */
/*     { */
/*       DoPrint ("Cm> Name %s unknown by NameServer.\n", (name ? name : "NULL")); */
/*     }     */
  
  if (CmMessageGetItemType (message) == CmMessageItemInt)
    {
      int transaction;

      transaction = CmMessageGetInt (message);
      CmTerminateTransaction (transaction);

      return (CmMessageOk);
    }
  else
    {
        /*
          This should become obsolete when transaction
          mechanisms are generalised
          This is maintained for backward compatibility
          */
      CmTerminateTransaction (CurrentGetAddressTransaction);

      return (CmMessageBreak);
    }
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSWakeupHandler (CmMessage message)
/*--------------------------------------------------------------*/
/*  Handler triggered by a 'NSWakeup' message.                  */
/*                                                              */
/*--------------------------------------------------------------*/
{
  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSPortHandler (CmMessage message,
                                      char* sender,
                                      char* server)
/*--------------------------------------------------------------*/
/*                                                              */
/*  This message is an answer to the NewConnect                 */
/* sent to the name server                                      */
/*                                                              */
/*  A transaction had been opened and its id is stored within   */
/* the Connect object.                                          */
/*                                                              */
/*  Expected received syntax :                                  */
/*                                                              */
/*    char* name              Connection name                   */
/*                                                              */
/*    int port                port number of the Connect        */
/*    char* new_name          Overridden connection name        */
/*                                                              */
/*--------------------------------------------------------------*/
{
  char* name;
  int port;
  char* new_name;
  CmConnect connect;

  name     = CmMessageGetText (message);
  port     = CmMessageGetInt (message);
  new_name = CmMessageGetText (message);
  
  connect = CmConnectGetReference (name);

  if (!port)
    {
      DoPrint ("Cm> Name already used.\n");
      connect = CmConnectGetReference (server);
    }
  else if (connect == NULL)
    {
      DoPrint ("Cm> Name %s unknown by NameServer.\n", name);
    }
  else
    {
      CmConnectSetPort (connect, port);
      if (strcmp (new_name, name))
        {
          CmConnectSetName (connect, new_name);
        }
    }

  if (connect != NULL)
    {
      CmTerminateTransaction (connect->transaction);
    }

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSNewPortHandler (CmMessage message)
/*--------------------------------------------------------------*/
{
  char* name;
  int port;
  CmConnect connect;

  name     = CmMessageGetText (message);
  port     = CmMessageGetInt (message);
  
  connect = CmConnectGetReference (name);
  if (connect == NULL)
    {
      DoPrint ("Cm> Name %s unknown by NameServer.\n", name);
    }
  else if (!port)
    {
      DoPrint ("Cm> Name %s is already used.\n", name);
    }
  else
    {
      CmConnectSetPort (connect, port);
    }

  if (CmMessageGetItemType (message) == CmMessageItemInt)
    {
      int transaction;

      transaction = CmMessageGetInt (message);
      CmTerminateTransaction (transaction);

      return (CmMessageOk);
    }
  else
    {
        /*
          This should become obsolete when transaction
          mechanisms are generalised
          This is maintained for backward compatibility
          */
      CmTerminateTransaction (CurrentGetNewPortTransaction);

      return (CmMessageBreak);
    }
}

/*----------------------------------------------------------------*/
static CmMessageStatus RequestComment (CmMessage message,
                                       char* sender,
                                       char* server_name)
/*----------------------------------------------------------------*/
{
  CmConnect connect;
  char* comment;
  CmMessage answer;

  connect = CmConnectWhoAmI ();

  answer = CmMessageNew ();
  CmMessageSetType (answer, "CmComment");

  comment = CmConnectGetComment (connect);
  if (comment != NULL)
    CmMessagePutText (answer, comment);
  else
    CmMessagePutText (answer, "");

  CmMessageSend (answer, sender);

  CmMessageDelete (answer);

  return (CmMessageOk);
}

/*----------------------------------------------------------------*/
static CmMessageStatus RequestStatistics (CmMessage message,
                                          char* sender,
                                          char* server_name)
/*----------------------------------------------------------------*/
{
  CmConnect connect;
  CmMessage answer;

  connect = CmConnectWhoAmI ();

  answer = CmMessageNew ();
  CmMessageSetType (answer, "CmStatistics");

  CmMessagePutInt (answer, connect->statistics.sent);
  CmMessagePutInt (answer, connect->statistics.bytes_sent);
  CmMessagePutInt (answer, connect->statistics.received);
  CmMessagePutInt (answer, connect->statistics.bytes_received);

  CmMessageSend (answer, sender);

  CmMessageDelete (answer);

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
int CmConnectRequestKill( char *name, int flag ) {
  CmConnect c;
  Entry* entry;
  int cnt = 0;
  char txt[256];

  if (!strcmp (name, "<null>")) {
    for (entry = GetFirst(); entry != NULL; entry = entry->next)  {
      c = (CmConnect) entry->reference;
      if (c->name == NULL ) {
	cnt++;
	CmConnectSnprint( c, txt, sizeof txt );
	DoPrint ("Cm>CmConnectRequestKill> NULL CmConnect - %s\n", txt);
	if( flag ) CmConnectKill (c);
      }
    }      
  } else {
    c = CmConnectGetReference( name );
    CmConnectSnprint( c, txt, sizeof txt);
    DoPrint("Cm>CmConnectRequestKill> %s CmConnect - %s\n", name, txt);
    cnt++;
    if( flag ) CmConnectKill (c);
  }
  return(cnt);
}


/*--------------------------------------------------------------*/
static CmMessageStatus RequestKill( CmMessage message, 
				    char* sender,
				    char* server_name ) 
/*--------------------------------------------------------------*/
{

  char *name;

  if( CmMessageGetItemType(message) !=  CmMessageItemText ) 
    return (CmMessageOk);

  name = CmMessageGetText(message);

  if( !name || !name[0] || !strcmp(name, server_name) ) 
    return(CmMessageOk);
  
  DoPrint("Cm>ResquestKill \"%s\" from %s\n", name, 
	  (sender ? sender : "unknown") );
  CmConnectRequestKill( name, 1 ) ;

  return (CmMessageOk);
}

  
/*--------------------------------------------------------------*/
static CmMessageStatus RequestPrint( CmMessage message, 
				     char* sender,
				     char* server_name ) 
/*--------------------------------------------------------------*/
{
  CmConnect c;
  Entry* entry;
  CmMessage answer;
  char txt[1024];
  
  answer = CmMessageNew();

  CmMessageSetType (answer, "CmPrint");

  for (entry = GetFirst(); entry != NULL; entry = entry->next) 
  {
    c = (CmConnect) entry->reference;
#ifdef CM_OS9
    sprintf (txt, "CmName %s, Host %s, port %d, socket_id %d, owner %s""",
	     (c->name ? c->name : "NULL"), 
	     (c->host_name ? c->host_name : "NULL"), 
	     c->port, 
	     c->socket_id, 
	     (c->owner ? c->owner : "NULL"));
#else
    CmConnectSnprint( c, txt, sizeof txt );
#endif
    CmMessagePutText (answer, txt);
  }
  CmMessageSend (answer, sender);

  CmMessageDelete (answer);

  return (CmMessageOk);
}


/*----------------------------------------------------------------------------*/
static CmMessageStatus SetSocketSize( CmMessage message, 
				     char* sender,
				     char* server_name ) 
/*----------------------------------------------------------------------------*/
{
  int sndBuf, rcvBuf;
  char *tag;

  tag = NULL; sndBuf = rcvBuf = -1;
  if( CmMessageGetItemType( message ) == CmMessageItemInt ) {
    sndBuf = CmMessageGetInt( message );
  }
  if( CmMessageGetItemType( message ) == CmMessageItemInt ) {
    rcvBuf = CmMessageGetInt( message );
  }
  if( CmMessageGetItemType( message ) == CmMessageItemText ) {
    tag = CmMessageGetText( message );
  }
  DoPrint("Cm> SetSocketSize> snd %d - rcv %d - tag %s\n",
	  sndBuf, rcvBuf, (tag ? tag : "*") );
  /*--------------------------------------------------------------------------*/
  CmConnectSetSocketSize( sndBuf, rcvBuf, tag );
  /*--------------------------------------------------------------------------*/
  return (CmMessageOk);
}


/*--------------------------------------------------------------*/
static CmMessageStatus CmServerInfosHandler (CmMessage message,
                                             char* sender,
                                             char* server_name)
/*--------------------------------------------------------------*/
{
  char* version;
  char* name;
  char* host;
  int port;
  int multiple;
  char* owner;
  CmConnect connect;

  version  = CmMessageGetText (message);
  
  if (CmConnectMgr.version == NULL)
    {
      DoPrint ("Cm> Bad local version.\n");
    }
  else if (strcmp(version, CmConnectMgr.version) != 0)
    {
      DoPrint ("Cm> Version mismatch from client.\n");
    }
  else
    {
      name    = CmMessageGetText (message);
      connect = CmMessageGetConnect (message);
      /* Beware here, the connect field are not filled */
  
      if (connect == NULL)
        {
          DoPrint ("Cm> Connection %s unknown.\n", name);
        }
      else
        {
          int illegal = 0;

          host     = CmMessageGetText (message);
          port     = CmMessageGetInt (message);
          multiple = CmMessageGetInt (message);
          owner    = CmMessageGetText (message);

          connect->multiple = multiple;

          /*
            Lorsque deux applications entre-croisent leurs
            demandes de connections (avec des demandes paralleles
            au NameServer sur leurs adresses mutuelles) elles creent 
            chacune un socket . Si ces operations sont reellement 
            effectuees 'en meme temps' on cree une double connection
            (deux sockets et deux Connects).

            Ceci n'est pas correctement gere actuellement !!!

            Une approche etait de considerer une des deux connections
            comme "illegale". Ceci semble poser des problemes dans les
            applications. On va essayer de vivre avec le doublon operationnel

            */

          if (CmConnectIsNameServer (CmConnectWhoAmI ()))
	    {
	      if (!multiple)
		{
		  CmConnect c = NULL;
		  
		  c = CmConnectGetReference (name);
		  if (c != NULL)
		    {
		      illegal = 1;
		    }
		}
            }
	  else
	    {
	      /* a new instance of an existing connection when the
		 server had not sent a disconnection signal */

	      CmConnect c = NULL;
	      /* If NameServer build the CmConnect NameServer name */
	      if (IsNameServer(name)) 
		name = BuildNewNameServerName (server_name);
 
	      c = CmConnectGetReference (name);
	      if (c != NULL)
		{
		  /* old must be discarded */
		  CmConnectKill (c);
		}
	    }

	  /* name as <illegal> is used by the NameServer to set   */
	  /* to 0 and to reply to the requester with a NSPort message */ 
	  /* that the name is already used */
          if (!illegal) CmConnectSetName (connect, name);
          else CmConnectSetName (connect, "<illegal>");

          CmConnectSetHost (connect, host);
          CmConnectSetOwner (connect, owner);

	  CmConnectSetSocketProperties( connect, -1, -1 );
          /*--------------------------------- Added by F. BELLACHIA */
          CmConnectSetPort (connect, port);

          if (IsNameServer (name))
            {
              CmConnect server;

              server = CmConnectGetReference (server_name);
              CmConnectSetServer (server, connect);
            }
         /*---------------------------------------------------------*/
          
          ConnectionHandlerStackCall (&(CmConnectMgr.connection_handlers),
                                      connect);

        }
    }

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmConnectStatus ServerDoConnect (CmConnect connect)
/*--------------------------------------------------------------*/
{
  CmConnectStatus status;
  CmConnect new_connect;
  CmConnect receiver;

  struct sockaddr_in socket_structure;
  int                socket_id;
  struct hostent *host;
  char txt[128];

  status = CmConnectCheckInputData (connect);
  if (status != CmConnectStatusData) return (status);

  socket_id = TCPAccept (connect->socket_id, &socket_structure);

#ifdef CM_USE_SELECT
  /*-----------------------------select: fd_set size lomited too FD_SETSIZE */
  if( socket_id > (FD_SETSIZE-1) )  {
    DoPrint ("Cm> Accept error - fd_set FULL %d\n", FD_SETSIZE-1);
    TCPShutdown( socket_id );
    TCPClose( socket_id );
    socket_id = -1; 
    return (CmConnectStatusNoData);
  }
#endif

  if (socket_id == -1)
    {
      DoPrint ("Cm> Accept error.\n");
      return (CmConnectStatusNoData);
    }

  /*
    Now we can really create a CmConnect object
    */
  new_connect = CreateNewConnect (NULL, NULL, 0);
  if (new_connect == NULL)
    {
      DoPrint ("Cm> Connect creation error.\n");
      return (CmConnectStatusNoData);
    }

  new_connect->socket_id        = socket_id;

  new_connect->socket_structure = socket_structure;
  /*
  new_connect->socket_structure.sin_family = socket_structure.sin_family;
  new_connect->socket_structure.sin_port   = socket_structure.sin_port;
  new_connect->socket_structure.sin_addr.s_addr = socket_structure.sin_addr.s_addr;
  */

  CmConnectSetServer (new_connect, connect);
  host = gethostbyaddr( (const void*)&(new_connect->socket_structure.sin_addr),
			(int)sizeof(struct in_addr),
			(int)AF_INET );
  if( !host ) { 
    DoPrint("Cm>ServerDoConnect> gethostbyadd error %s", strerror(errno));
    CmConnectKill(new_connect);
    return (CmConnectStatusNoData);
  }
  /*-----------------------------------------------------------*/
  CmConnectSetHost( new_connect, host->h_name );
  CmConnectSnprint( new_connect, txt, sizeof txt );
  DoPrint("Cm>ServerDoConnect> Connection in progress - %s\n", txt );

  receiver = CmConnectHandleGet (&connect->receiver);
  CmConnectSetServer (receiver, connect);

  CmMessagePrepareHandler (new_connect);
    
  return (CmConnectStatusNoData);
}





/*--------------------------------------------------------------*/
/*                                                              */
/*                                                              */
/*  Name server management.                                     */
/*                                                              */
/*                                                              */
/*--------------------------------------------------------------*/


/*--------------------------------------------------------------*/
CmNameServerInfos CmNameServerGetInfos ()
/*--------------------------------------------------------------*/
{
    /* These two variables should remain static */
  static char text[1024];
  static CmNameServerInfosRec infos = 
  {
    NULL,  /* name */
    NULL,  /* host */
    0,     /* port */
    0,     /* firstPort */
    0,     /* ports */
    NULL   /* repository */
  };

  FILE* file;
  char file_name[512];
  char* env;
  char* string;
  char* space;
  char* nl;

  CmConnectInitialize ();

  if (CmDomain == NULL)
    {
      DoPrint ("Cm> CMDOMAIN is not defined.\n");
      CmConnectExit (CmErrorBadEnvironment);
      return (NULL);
    }

  if ((infos.port != 0) &&
      (infos.host != NULL) &&
      (infos.name != NULL)) return (&infos);

  infos.name = (char*) "CmNameServer";

#ifdef CM_WIN32
  env = "";
#else
  if (((env = getenv ("CMDOMAIN_DATABASE")) == NULL) &&
      ((env = getenv ("CMMGR")) == NULL))
    {
      if ((env = getenv ("CMROOT")) == NULL)
        {
          DoPrint ("Cm> CMROOT is not defined.\n");
          CmConnectExit (CmErrorBadEnvironment);
          return (NULL);
        }
      strcpy (file_name, env);
      strcat (file_name, "/mgr");
    }
  else
    {
      strcpy (file_name, env);
    }
#endif

    /* Builds the CmDomains file name. */
  
  strcat (file_name, "/CmDomains");
  
  file = fopen (file_name, "r");
  if (file == NULL)
    {
      DoPrint ("Cm> CmDomains file cannot be opened.\n");
      CmConnectExit (CmErrorBadEnvironment);
      return (NULL);
    }

    /* Detects the definition line corresponding to the selected domain */
  
  string = NULL;
  while (!feof(file))
    {
      if (fgets (text, sizeof(text), file) == NULL) break;
      if ((nl = strchr (text, '\n')) != NULL) *nl = 0;
      space = strchr (text, ' ');
      if (space == NULL) break;
      *space = 0;
      if (!strcmp (text, CmDomain))
        {
          string = space + 1;
          break;
        }
    }

  fclose (file);

  if (string == NULL)
    {
      DoPrint ("Cm> The domain %s is not defined.\n",
               CmDomain);
      CmConnectExit (CmErrorBadEnvironment);
      return (NULL);
    }

    /*

      Parse the domain definition line :

       domain-name
       nameserver host
       nameserver port
       range first port
       range size
       root database repository.
       
       */


    /* NameServer host */
  string += strspn (string, " ");
  space = strchr (string, ' ');
  if (space != NULL) *space = 0;
  
  if (strlen (string) == 0)
    {
      DoPrint ("Cm> NameServer Host is not defined.\n");
      CmConnectExit (CmErrorBadEnvironment);
      return (NULL);
    }
  
  infos.host = string;
  
  if (space != NULL) string = space + 1;

    /* NameServer port */
  string += strspn (string, " ");
  space = strchr (string, ' ');
  if (space != NULL) *space = 0;
  
  if (strlen (string) == 0)
    {
      DoPrint ("Cm> NameServer Port is not defined.\n");
      CmConnectExit (CmErrorBadEnvironment);
      return (NULL);
    }
  
  sscanf (string, "%d", &infos.port);
  
  if (space != NULL) string = space + 1;

    /* range's first port */
  string += strspn (string, " ");
  space = strchr (string, ' ');
  if (space != NULL) *space = 0;
  
  if (strlen (string) != 0)
    {
      sscanf (string, "%d", &infos.firstPort);
    }
  else return (&infos);

  if (space != NULL) string = space + 1;
  
    /* range's size */
  string += strspn (string, " ");
  space = strchr (string, ' ');
  if (space != NULL) *space = 0;
  
  if (strlen (string) != 0)
    {
      sscanf (string, "%d", &infos.ports);
    }
  else return (&infos);
  
  if (space != NULL) string = space + 1;
  
    /* info repository */
  string += strspn (string, " ");
  space = strchr (string, ' ');
  if (space != NULL) *space = 0;
  
  infos.repository = string;
  
  return (&infos);
}



/*  Private functions ------------------------------------------*/



/*--------------------------------------------------------------*/
static char* BuildNewNameServerName (char *server_name)
/*--------------------------------------------------------------*/
{
  static char name[256], *tmp;
  CmConnect c;
  
  if( (c = CmConnectGetReference( server_name )) &&
      c->multiple ) {
    tmp = strrchr( server_name, '_' );
    if( tmp ) tmp[0] = '\0';
  } else tmp = NULL;

  sprintf (name, "%s_%s", NameServerInfos->name, server_name);
  if( tmp ) tmp[0] = '_'; 
  return ((char*) name);

}

/*--------------------------------------------------------------*/
static CmConnect ConnectToNameServer (CmConnect server)
/*--------------------------------------------------------------*/
{
  CmConnect connect;
  char* name;
  
  name = BuildNewNameServerName ( CmConnectGetName(server) );
  
  connect = CmConnectNewWithAddress (server,
                                     name,
                                     NameServerInfos->port,
                                     NameServerInfos->host);

  if (connect == NULL)
    {
      DoPrint ("Cm> Impossible to reach the NameServer.\n");
      return (NULL);
    }


  return (connect);
}




/*--------------------------------------------------------------*/
/*                                                              */
/*  TCPIP operations.                                           */
/*                                                              */
/*--------------------------------------------------------------*/



/*--------------------------------------------------------------*/
int CmConnectTestPort (int port)
/*--------------------------------------------------------------*/
/*   This function creates a temporary socket, tries a "bind"   */
/* operation upon it, and releases the socket afterward.        */
/*   It returns whether the bind was possible.                  */
/* The socket used for this operation is (should be) reusable.  */
/*--------------------------------------------------------------*/
{
  int id;
  struct sockaddr_in addr;
  int status;

  id = TCPSocket ();
  if (id == 0)
    {
      DoPrint ("Cm> Impossible to create a test socket\n");
      return (0);
    }

  addr.sin_port        = htons (port);
  addr.sin_addr.s_addr = 0;
  addr.sin_family      = AF_INET;
  status = TCPBind (id, &addr);
  TCPShutdown (id);
  TCPClose (id);

  return ((status != -1));
}

/*  Private functions ------------------------------------------*/

/*--------------------------------------------------------------*/
static char* InitializeForHost (char* host_name, 
                                struct sockaddr_in* struct_socket)
/*--------------------------------------------------------------*/
{
  struct hostent* hostentPtr;
  char* full_host;
  union {
    unsigned long l;
    struct in_addr in;
    unsigned char a[sizeof(unsigned long)];
  } addr;
  unsigned int a0, a1, a2, a3;
  int items;

  full_host = host_name;

  items = sscanf (full_host, "%u.%u.%u.%u", &a0, &a1, &a2, &a3);

  if (items == 4)
    {
      addr.a[0] = a0;
      addr.a[1] = a1;
      addr.a[2] = a2;
      addr.a[3] = a3;
      struct_socket->sin_family = AF_INET;
      struct_socket->sin_addr = addr.in;
    }
  else
    {
      if (!(hostentPtr = gethostbyname (full_host)))
        {
          DoPrint ("Cm> Host %s unknown.\n", full_host);
          return (NULL);
        }
      struct_socket->sin_family = hostentPtr->h_addrtype;
      struct_socket->sin_addr = *((struct in_addr*) hostentPtr->h_addr);

      /*
         returned host name should be the full host name understandable
         by anybody.

         o The h_name field in the hostentptr is a first attempt to
         get a valid expansion.

         o If this expansion does not contain dots in it, we must build
         the complete internet address.

         */

      full_host = (char*) hostentPtr->h_name;

      if (!strchr (full_host, '.'))
        {
          static char internet[20];

          sprintf (internet, "%d.%d.%d.%d",
                   (unsigned char) hostentPtr->h_addr[0],
                   (unsigned char) hostentPtr->h_addr[1],
                   (unsigned char) hostentPtr->h_addr[2],
                   (unsigned char) hostentPtr->h_addr[3]);
          full_host = internet;
        }
    }
  return (full_host);
}

/*--------------------------------------------------------------*/
static int Connect (CmConnect connect, int port)
/*--------------------------------------------------------------*/
{
  connect->socket_structure.sin_port = htons (port);

  return (TCPConnect (connect->socket_id, &connect->socket_structure));
}

#if 0
/*----------------------------------------------------------------*/
/* static void WaitSomeSeconds (int seconds) */
/*----------------------------------------------------------------*/
/* { */
/*   static struct timeval timeout; */

/*   timeout.tv_sec = seconds; */
/*   timeout.tv_usec = 0; */
/*   TCPSelect (NULL, NULL, NULL, &timeout); */
/* } */
#endif

/* static int SignalReceived = 0; */

/* #if defined (CM_ANY_UNIX) || defined (CM_LINUX) */

/*----------------------------------------------------------------*/
/* static void SigAction (int sig) */
/*----------------------------------------------------------------*/
/* { */
/*   SignalReceived = 1; */
/*   DoPrint ("Cm> Signal %d received\n", sig); */
/* } */

/* #endif */

/*----------------------------------------------------------------*/
static int TCPSocket ( )
/*----------------------------------------------------------------*/
{
  int id;
  int status;
  int option;

  if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> socket (%d, %d, 0) -> ", AF_INET, SOCK_STREAM);
    }

  id = socket (AF_INET, SOCK_STREAM, 0);
  if (id == 0)
    {
      return (TCPSocket ());
    }


  if (CmConnectDebug > 0)
    {
      DoPrint ("id = %d\n", id);
    }

#ifdef CM_USE_SELECT
 /*-----------------------------select: fd_set size lomited too FD_SETSIZE */
  if( id > (FD_SETSIZE-1) )  {
    DoPrint ("Cm> TCPSocket - fd_set FULL %d\n", FD_SETSIZE-1);
    TCPClose( id );
    id = -1; 
  }
#endif

  if (id < 0) return (id);

  if (CmConnectDebug > 1)
    {
      DoPrint ("Cm> setsockopt (%d, %d, %d, %x, %d) -> ", id, SOL_SOCKET,
               SO_REUSEADDR, (void*) &option, sizeof(option));
    }

  option = 1;
  status = setsockopt (id, SOL_SOCKET, SO_REUSEADDR, (void*) &option,
                       sizeof(option));

  if (CmConnectDebug > 1)
    {
      DoPrint ("%d\n", status);
    }

  {
    struct protoent* ent;

    option = 1;

    ent = getprotobyname ("TCP");

    if (ent != NULL)
      {
        status = setsockopt (id, ent->p_proto, TCP_NODELAY, (void*) &option,
                             sizeof(option));

        if (CmConnectDebug > 1)
          {
            DoPrint ("%d\n", status);
          }
      }
  }

  return (id);
}

/*----------------------------------------------------------------*/
static int TCPListen (int id)
/*----------------------------------------------------------------*/
{
  int status;

  if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> listen (%d, %d) -> ", id,
               howmany (FD_SETSIZE, NFDBITS));
    }

  status = listen (id, howmany (FD_SETSIZE, NFDBITS));

  if (CmConnectDebug > 0)
    {
      DoPrint ("status = %d\n", status);
    }

  return (status);
}

/*----------------------------------------------------------------*/
static void printBytes (char* buffer, int bytes)
/*----------------------------------------------------------------*/
{
  DoPrint ("%d bytes \"", bytes);
  if (bytes)
    {
      int toDump;
      int i;
      unsigned char c;

      toDump = (bytes < 100) ? bytes : 100;
      toDump = bytes;
          
      for (i = 0; i < bytes; i++)
        {
          if (i > toDump) break;

          if ((i % 16) == 0) DoPrint ("\n%4.4d: ", i);

          c = buffer[i];
          DoPrint ("%2.2x", c);
              
          if ((c < ' ') || (c > 127)) c = '.';
          DoPrint ("%c ", c);
        }
    }
  DoPrint ("\"");
}



#ifndef CM_USE_SELECT
/*----------------------------------------------------------------*/
static int TCPPoll ( int nfds,  struct timeval* timeout)
/*----------------------------------------------------------------*/
{
  int status, tmo_ms;
  struct timeval s_poll, e_poll;
  struct pollfd *fds;

  if( !(fds = CmConnectMgr.fds) ) return(0);

  if( !timeout ) tmo_ms = -1;
  else if( !timeout->tv_sec && !timeout->tv_usec  ) tmo_ms = 0;
  else {
    tmo_ms  = timeout->tv_usec / 1000;
    tmo_ms += timeout->tv_sec  * 1000;
  }

  if( CmConnectDebug > 0 ) {
      DoPrint ("Cm> poll (%d, ", nfds );
      DoPrint (", tmo(%dms)) -> ",  tmo_ms);
      DoPrint ("\n");
  }

  gettimeofday( &s_poll, NULL );
  errno = 0; 
  status = poll( fds, nfds, tmo_ms );
  gettimeofday( &e_poll, NULL );

  if( timeout && status > 0 ) {
    timeout->tv_sec  -= ( e_poll.tv_sec  - s_poll.tv_sec );
    timeout->tv_usec -= ( e_poll.tv_usec - s_poll.tv_usec );
    if( timeout->tv_usec <  0 ) {
      timeout->tv_usec += 1000000;
      timeout->tv_sec  -= 1;
    }
    if( timeout->tv_sec <  0 ) {
      timeout->tv_usec = 0;
      timeout->tv_sec  = 0;
    }
  }

  if (CmConnectDebug > 0)  {
    DoPrint ("Cm>  -> (" );
    DoPrint (") %d [errno=%d]\n", status, errno);
  }

  return (status);
}

#else 


/*----------------------------------------------------------------*/
static void printMask (fd_set* set)
/*----------------------------------------------------------------*/
{
  int last;
  int word;

  if (set == NULL)
    {
      DoPrint ("0");
      return;
    }

  for (last = (sizeof (fd_set) / sizeof (fd_mask)) - 1; last >= 0; last--)
    {
      if (set->fds_bits[last] != 0) break;
    }

  if (last < 0)
    {
      DoPrint ("0");
      return;
    }

  for (word = 0; word <= last; word++)
    {
      int i;
      fd_mask mask;
      int max = sizeof(fd_mask)*8;
      int bit[sizeof(fd_mask)*8];
      
      mask = set->fds_bits[word];

      for (i = 0; i < sizeof(fd_mask)*8; i++)
        {
          if (mask == 0)
            {
              max = i;
              break;
            }
          bit[i] = mask % 2;
          mask >>= 1;
        }
      
      for (i = max - 1; i >= 0; i--)
        {
          if (bit[i]) DoPrint ("1");
          else        DoPrint ("o");
        }
    }
}


/*----------------------------------------------------------------*/
static int TCPSelect ( int nfds, fd_set* rSet, fd_set* wSet,
		       fd_set* xSet, struct timeval* timeout)
/*----------------------------------------------------------------*/
{
  int status;
  int i;

  fd_set r_save;
  fd_set w_save;
  fd_set x_save;
  static int special_debug = -1;

  if (special_debug == -1)
    {
      if (getenv ("CMCONNECTSPECIALDEBUG") != NULL)
        {
          special_debug = 1;
        }
      else
        {
          special_debug = 0;
        }
    }

  if (special_debug == 1)
    {

      FD_ZERO (&r_save);
      FD_ZERO (&w_save);
      FD_ZERO (&x_save);

      if (rSet != NULL)
        {
          for (i = 0; i < sizeof (fd_set) / sizeof (fd_mask); i++)
            {
              r_save.fds_bits[i] = rSet->fds_bits[i];
            }
        }

      if (wSet != NULL)
        {
          for (i = 0; i < sizeof (fd_set) / sizeof (fd_mask); i++)
            {
              w_save.fds_bits[i] = wSet->fds_bits[i];
            }
        }

      if (xSet != NULL)
        {
          for (i = 0; i < sizeof (fd_set) / sizeof (fd_mask); i++)
            {
              x_save.fds_bits[i] = xSet->fds_bits[i];
            }
        }
    }

  if (CmConnectDebug > 0)
    {
      /* DoPrint ("Cm> select (%d, ", FD_SETSIZE); */
      DoPrint ("Cm> select (%d, ", nfds );
      printMask (rSet);
      DoPrint (", ");
      printMask (wSet);
      DoPrint (", ");
      printMask (xSet);
      DoPrint (", tmo) -> ");
      DoPrint ("\n");
    }

  errno = 0;


  /* status = select (FD_SETSIZE,  */
  status = select ( nfds,
		   (fd_set*) rSet, 
		   (fd_set*) wSet,
		   (fd_set*) xSet,
		   timeout );
  /* if (status < 0) */
  /*   { */
  /*     if (CmConnectDebug > 0) */
  /* 	{ */
  /*             DoPrint ("Cm>  -> (", status); */
	      
  /*             printMask (rSet); */
  /*             DoPrint (", "); */
  /*             printMask (wSet); */
  /*             DoPrint (", "); */
  /*             printMask (xSet); */
	      
  /*             DoPrint (") %d [errno=%d]\n",  */
  /*                      status, errno); */
  /* 	} */
  /*   } */

  if (CmConnectDebug > 0)
    {
      DoPrint ("Cm>  -> (" );

      printMask (rSet);
      DoPrint (", ");
      printMask (wSet);
      DoPrint (", ");
      printMask (xSet);
      
      DoPrint (") %d [errno=%d]\n", status, errno);
    }

  if ((special_debug == 1) &&
      (status == 0) &&
      (timeout != NULL))
    {

      DoPrint ("Cm> select special (%d, ", FD_SETSIZE);
      printMask (&r_save);
      DoPrint (", ");
      printMask (&w_save);
      DoPrint (", ");
      printMask (&x_save);
      DoPrint (", tmo) -> ");

      DoPrint (" %d (", status);

      printMask (rSet);
      DoPrint (", ");
      printMask (wSet);
      DoPrint (", ");
      printMask (xSet);
      
      DoPrint (")\n");

      {
        CmConnect connect;
        Entry* entry = GetFirst ();
  
        DoPrint ("Cm> select special> connects :");
        while ((connect = GetNextAlive (&entry)))
          {
            char* name;

            name = CmConnectGetName (connect);
            if (name == NULL)
              DoPrint ("[%d %d] ", connect->socket_id, connect->data_pending);
            else
              DoPrint ("[%s-%d %d] ", name, connect->socket_id, 
                       connect->data_pending);
          }
        DoPrint ("\n");
      }
    }

  return (status);
}

#endif 


/*----------------------------------------------------------------*/
static ssize_t PipeRecv (int id, char* buffer, int length)
/*----------------------------------------------------------------*/
{
  ssize_t bytes;

  if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> pipe recv (%d, %x, %d, 0) -> ", id, buffer, length);
    }

#ifdef CM_OS9
  bytes = 0;
#else
  errno = 0;
  bytes = read (id, buffer, length);
#endif

  if (CmConnectDebug > 1)
    {
      printBytes (buffer, bytes);
      DoPrint ("\n");
    }
  else if (CmConnectDebug > 0)
    {
      DoPrint ("bytes=%d errno=%d\n", bytes, errno);
    }
/* MD */
  
#ifdef CM_OS9
  
  if (bytes == -1) return (0);
  
#endif
  
  return (bytes);
}

/*----------------------------------------------------------------*/
static ssize_t TCPRecv (int id, char* buffer, int length)
/*----------------------------------------------------------------*/
{
 ssize_t bytes;

  if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> recv (%d, %x, %d, 0) -> ", id, buffer, length);
    }

  DontBlock( id );
  errno = 0;
  bytes = recv (id, buffer, length, 0);
  Block( id );

  if (CmConnectDebug > 1)
    {
      printBytes (buffer, bytes);
      DoPrint ("\n");
    }
  else if (CmConnectDebug > 0)
    {
      DoPrint ("bytes=%d errno=%d\n", bytes, errno);
    }
/* MD */
  
#ifdef CM_OS9
  
  if (bytes == -1) return (0);
  
#endif
  if( bytes < 0 ) bytes = 0;
  
  return (bytes);
}

/*----------------------------------------------------------------*/
static int TCPTest (int id)
/*----------------------------------------------------------------*/
{
  int status;
  char test[256];

  if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> test (%d, %x, %d, %d) ",
               id,
               test[0], sizeof(test), MSG_PEEK);
    }

#ifdef CM_OS9
  
  status = _gs_rdy(id);
  if ((status == -1) && (errno == 211))
    {
      status = 0;
      errno = 0;
    }
  
#else
  
  errno = 0;
  status = recv (id, test, sizeof(test), MSG_PEEK);
  
#endif

  if (CmConnectDebug > 0)
    {
      DoPrint ("%d bytes errno=%d ", status, errno);
      DoPrint ("\n");
    }

  return (status);
}

/*----------------------------------------------------------------*/
static ssize_t PipeSend (int id, char* buffer, int bytes)
/*----------------------------------------------------------------*/
{
  ssize_t status;

  if (CmConnectDebug > 1)
    {
      DoPrint ("Cm> pipe send (%d, ", id);
      printBytes (buffer, bytes);
      DoPrint (", %d, 0) -> ", bytes);
    }
  else if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> pipe send (%d, ", id);
      DoPrint (", %d, 0) -> ", bytes);
    }

#ifdef CM_OS9
  status = 0;
#else
  DontBlock (id);
  errno = 0;
  status = write (id, buffer, bytes);
  Block (id);
#endif

  if (CmConnectDebug > 0)
    {
      DoPrint ("%d bytes errno=%d\n", status, errno);
    }

  return (status);
}

#ifdef CM_OS9
/*----------------------------------------------------------------*/
static size_t TCPSend (int id, char* buffer, int bytes)
/*----------------------------------------------------------------*/
{
  size_t status;
  int retry = 0;

  if (CmConnectDebug > 1)
    {
      DoPrint ("Cm> send (%d, ", id);
      printBytes (buffer, bytes);
      DoPrint (", %d, 0) -> \n", bytes);
    }
  else if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> send (%d, ", id);
      DoPrint (", %d, 0) -> ", bytes);
    }
  
  DontBlock (id);
      
#if defined (CM_ANY_UNIX) || defined (CM_LINUX)
  SignalReceived = 0;
  signal (SIGPIPE, SigAction);
#endif

  errno = 0;
  for (retry = 0; retry < 100; retry++)
    {
      status = write (id, buffer, bytes);
      
      if (CmConnectDebug > 0)
        {
          if ((status <= 0) || SignalReceived) 
            {
              DoPrint (" (%d/%d/errno=%d) ", retry, status, errno);
            }
        }

      if (status > 0) break;
     /*  As status is an unsigned int(size_t), even if write returns -1*/
     /* the following code is newer called and status is returned with */
     /* a value of 0xffffffff(-1) of 4294967295 */
     /* When a SIGPIPE is received setting the return code (status) to 0 */
     /* raise on infinite loop of retries with SIGPIPE dectection */  
#ifdef CM_ANY_UNIX
      if (SignalReceived)
        {
          status = 0;
          continue;
        }
#endif
      
      bytes /= 2;

      if (bytes == 0) 
        {
          status = 0;
          break;
        }
    }
  
  Block (id);
      
  if (CmConnectDebug > 0)
    {
      DoPrint ("%d bytes errno=%d\n", status, errno);
    }
  
  return (status);
}

#else

/*----------------------------------------------------------------*/
static ssize_t TCPSend (int id, char* buffer, int bytes)
/*----------------------------------------------------------------*/
{
/* As size_t in a unsigned int, change the prototype to ssize_t(int) */
/* to be compatible with the write prototype -----------------------*/  
  ssize_t status;
/*   void (*sigHdl)(int);  */

  if (CmConnectDebug > 1)
    {
      DoPrint ("Cm> send (%d, ", id);
      printBytes (buffer, bytes);
      DoPrint (", %d, 0) -> \n", bytes);
    }
  else if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> send (%d, ", id);
      DoPrint (", %d, 0) -> ", bytes);
    }
  
  DontBlock (id);
      
#if defined (CM_ANY_UNIX) || defined (CM_LINUX)
  /*------------------------------------- Trap the signal but do nothing !!! */
/*   SignalReceived = 0; */
/*   sigHdl = signal (SIGPIPE, SigAction); */
#endif

  errno = 0;
/*   status = write (id, buffer, bytes); */
  status =  send( id, buffer, bytes, MSG_NOSIGNAL );
  /*---------------------------- Ressource temporaly unavailable */
  if( (status < 0) && (errno == EAGAIN)  ) status = 0; 
  /*-------------------------------------------- Broken socket */
  if( (status < 0) && (errno == EPIPE) ) status = -2;
     
#if defined (CM_ANY_UNIX) || defined (CM_LINUX)
/*   signal (SIGPIPE, sigHdl );  */
/*   if( SignalReceived ) { */
/*     /\* Never return a status equal to 0 otherwise infinite loop of retry *\/ */
/*     status = -2; */
/*   } */
#endif
 
  Block (id);

  if (CmConnectDebug > 0)
    {
      DoPrint ("%d bytes errno=%d\n", status, errno);
    }

  return (status);
}

#endif

/*----------------------------------------------------------------*/
static int TCPAccept (int id, struct sockaddr_in* soc)
/*----------------------------------------------------------------*/
{
  int status;
  unsigned int length;

  length = sizeof (struct sockaddr_in);

  if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> accept (%d, %x, %x) -> ", id, soc, &length);
    }

  errno = 0;
  status = accept (id, (struct sockaddr*) soc, &length);


  if (CmConnectDebug > 0 )
    {
      DoPrint ("%d <f=%d p=%d a=%x> errno=%d\n",
               status,
               soc->sin_family,
               ntohs(soc->sin_port),
               soc->sin_addr.s_addr, errno);
    }
		       
  return (status);
}

/*----------------------------------------------------------------*/
static int TCPClose (int id)
/*----------------------------------------------------------------*/
{
  int status;

  if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> close (%d) -> ", id);
    }

  errno = 0;
  status = close (id);

  if (CmConnectDebug > 0)
    {
      DoPrint ("status = %d errno=%d\n", status, errno);
    }

  return (status);
}

/*----------------------------------------------------------------*/
static int TCPShutdown (int id)
/*----------------------------------------------------------------*/
{
  int status;
  int mode;

  if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> shutdown (%d) -> ", id);
    }

  for (mode = 2; mode >= 0; mode--)
    {
      errno = 0;
      status = shutdown (id, mode);

      if (CmConnectDebug > 0)
        {
          DoPrint ("status = %d mode = %d errno = %d\n", status, mode, errno);
        }

      if (status == 0) break;
    }

  return (status);
}

/*----------------------------------------------------------------*/
static int TCPBind (int id, struct sockaddr_in* soc)
/*----------------------------------------------------------------*/
{
  int status;
  int length;

  length = sizeof (struct sockaddr_in);

  if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> bind (%d, <f=%d p=%d a=%x>, %d) -> ",
               id,
               soc->sin_family,
               ntohs(soc->sin_port),
               soc->sin_addr.s_addr,
               length);
    }

  errno = 0;
  status = bind (id, (struct sockaddr*) soc, length);

  if (CmConnectDebug > 0)
    {
      DoPrint ("%d errno=%d\n", status, errno);
    }
      
  return (status);
}

/*----------------------------------------------------------------*/
static int TCPConnect (int id, struct sockaddr_in* soc)
/*----------------------------------------------------------------*/
{
  int status;
  int length;

  length = sizeof (struct sockaddr_in);

  if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> connect (%d, <f=%d p=%d a=%x>, %d) -> ",
               id,
               soc->sin_family,
               ntohs(soc->sin_port),
               soc->sin_addr.s_addr,
               length);
    }

  errno = 0;
  status = connect (id, (struct sockaddr*) soc, length);

    /* if (status < 0) status = 0; */

  if (CmConnectDebug > 0)
    {
      DoPrint ("%d errno=%d\n", status, errno);
    }

  return (status);
}

/*----------------------------------------------------------------*/
static int TCPGethostname (char* name, int size)
/*----------------------------------------------------------------*/
{
  int status = -1;
  char* cmName;

  cmName = getenv ("CMHOSTNAME");
  if (cmName != NULL)
    {
      if (size > strlen(cmName))
        {
          strcpy (name, cmName);
          status = 0;
        }
    }
  else
    {
      errno = 0;
      status = gethostname (name, size);
    }

  if (CmConnectDebug > 0)
    {
      DoPrint ("Cm> gethostname -> ");
      if (!status) DoPrint ("%s\n", name);
      else DoPrint ("status = %d errno=%d\n", status, errno);
    }

  return (status);
}

/*----------------------------------------------------------------*/
/*----------------------------------------------------------------*/

#if defined (CM_ANY_UNIX) || defined (CM_LINUX)
#include <fcntl.h>
#endif

/*----------------------------------------------------------------*/
static void DontBlock (int id)
/*----------------------------------------------------------------*/
{
#if defined (CM_ANY_UNIX) || defined (CM_LINUX)

  fcntl (id, F_SETFL, O_NONBLOCK);

#endif

#ifdef CM_OS9

  struct sgbuf buf;

  _gs_opt (id, &buf);
  buf.sg_noblock = 1;
  _ss_opt (id, &buf);

#endif
}                                                                            

/*----------------------------------------------------------------*/
static void Block (int id)
/*----------------------------------------------------------------*/
{
#if defined (CM_ANY_UNIX) || defined (CM_LINUX)

  fcntl (id, F_SETFL, 0);

#endif

#ifdef CM_OS9

  struct sgbuf buf;

  _gs_opt (id, &buf);
  buf.sg_noblock = 0;
  _ss_opt (id, &buf);

#endif
}

static void ConnectionHandlerStackInit (CmConnectConnectionHandlerStack* stack)
{
  if (stack == NULL) return;

  stack->size = 0;
  stack->entries = NULL;
}

static void ConnectionHandlerStackPush (CmConnectConnectionHandlerStack* stack,
                                        CmConnectConnectionHandler function)
{
  int had_old_entry = 0;

  if (stack == NULL) return;

  if (stack->size > 0)
    {
      int i;

        /*
          Start purging the stack of an old entry for this
          function.
         */

      for (i = 0; i < stack->size; i++)
        {
          if (stack->entries[i] == function)
            {
              had_old_entry = 1;

              for (; i < (stack->size - 1); i++)
                {
                  stack->entries[i] = stack->entries[i+1];
                }

              stack->entries[stack->size - 1] = NULL;
            }
        }
    }

  if (had_old_entry)
    {
      stack->entries[stack->size - 1] = function;
    }
  else
    {
      CmConnectConnectionHandler* new_entries = NULL;
      int i;

      new_entries = (CmConnectConnectionHandler*) 
          malloc ((stack->size + 1) * sizeof (CmConnectConnectionHandler));

      for (i = 0; i < stack->size; i++)
        {
          new_entries[i] = stack->entries[i];
        }

      stack->size++;
      new_entries[stack->size - 1] = function;

      if (stack->entries != NULL)
        {
          free (stack->entries);
        }

      stack->entries = new_entries;
    }
}

static void ConnectionHandlerStackCall (CmConnectConnectionHandlerStack* stack,
                                        CmConnect connect)
{
  int i;

  for (i = (stack->size - 1); i >= 0; i--)
    {
      CmConnectConnectionHandler handler = stack->entries[i];

      if (handler != NULL)
        {
          handler (connect);
        }
    }
}
