#include "FdIO.h"

/*-------------------------------------------------------private functions---*/

typedef struct FdNewFrame FdNewFrame;

struct FdNewFrame   
{
  char*     name;      /* name of the new frame FrameH structure       */
  double    length;    /* duration of surogate frame: 0= not produced  */
  int       startT;    /* GPS time of the first produced frame         */
  int       totalT;    /* total time produce for new frame; 0=for ever */
  int       DQ;        /* dataQuality flag sed for the newFrame        */
  int       GTimeS;    /* current GTimeS value for the new frame       */
  int       GTimeN;    /* current GTimeS value for the new frame       */
  FdAction* action;    /* owner of this action in the main linked list */
};

void FdNewFrameProcess(FdNewFrame *nf, FrameH* frame);

/*---------------------------------------------------------------------------*/
int FdNewFrameNew(void *arg, CfgDataType_t *data)
/*---------------------------------------------------------------------------*/
{
  FdNewFrame *nf;
  long  currentGPS, startGPS, nFrames;

  /*------------create the object and add it at the end of the linked list---*/
  nf = (FdNewFrame*) calloc(1,sizeof(FdNewFrame));
  if(nf == NULL) CfgMsgAddFatal("malloc FdNewFrame failed");

  nf->action = FdActionNew((FdAction**) arg, (void*) nf, 
			   FdNewFrameProcess, "FDIN_NEW_FRAME");

  if(nf->action != *((FdAction**) arg)) 
    CfgMsgAddFatal("FDIN_NEW_FRAME  could not be use with the %s key",
                    (*((FdAction**) arg))->type);

  /*------------------------------------------------------------ fill info---*/
  FrStrCpy(&nf->name, CfgParseGetNextString(data));
  nf->length =        CfgParseGetNextReal(data);
  nf->startT =        CfgParseGetNextDec(data);
  nf->totalT =        CfgParseGetNextDec(data);
  nf->DQ     =        CfgParseGetNextHex(data);

  /*-------------------first call: set the starting GPS time if requested---*/
  if(nf->startT <= 1.e6) {
    currentGPS = FrGetCurrentGPS() - nf->startT;
    /*--- the start time should be a multiple of the frame length---*/
    startGPS = currentGPS/nf->length;
    startGPS = startGPS*nf->length; 
    nf->startT = startGPS;}
  else {
    nFrames = nf->startT/nf->length;
    if(nFrames*nf->length != nf->startT) CfgMsgAddFatal("FdNewFrameNew: "
	"requested GPS start is not a multiple of the frame duration");}

  nf->GTimeS = nf->startT;
  CfgMsgAddInfo("New frame %s of length %g s. start at %d for %d s. DQ=0x%x", 
		nf->name,   nf->length,	nf->startT, nf->totalT, nf->DQ);
    
  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
FrameH* FdNewFrameProcessOne(FdNewFrame* nf)
/*---------------------------------------------------------------------------*/
{
  FrameH *frame;
  int  dtS, dtN;

  /*------------------------check if the requested time has been produced----*/
  if(nf->totalT > 0) {
    if(nf->GTimeS >= nf->startT + nf->totalT) {
      CfgMsgAddInfo("Stop new frame production at %d", nf->GTimeS);
      CfgTerminate();
      return(NULL);}}
 
  /*------------------------------------------------first create the frame---*/
  frame = FrameNew(nf->name);
  if(frame == NULL) CfgMsgAddFatal("malloc frame failed");
  frame->dt = nf->length;
  frame->GTimeS = nf->GTimeS;
  frame->GTimeN = nf->GTimeN;
  frame->dataQuality = nf->DQ;

  /*---------------------------------------------------------- update time---*/
  dtS = nf->length;
  dtN = (nf->length - dtS) * 1.e9;
  nf->GTimeN += dtN;
  nf->GTimeS += dtS;
  if(nf->GTimeN >= 1.e9) {
    nf->GTimeN -= 1.e9;
    nf->GTimeS += 1;}

  return(frame);
}
/*---------------------------------------------------------------------------*/
void FdNewFrameProcess(FdNewFrame* newFrame,
		       FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  frame = FdNewFrameProcessOne(newFrame);
  if(frame == NULL) return;

  /*------------------------------------------------------call next action---*/
  next = newFrame->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*---------------------------------------------------------------------------*/
void FdNewFrameParserAdd(FdIO* fdIO)
/*---------------------------------------------------------------------------*/
{
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_NEW_FRAME",
                         FdNewFrameNew, (void *) &(fdIO->actionsIn), 5,
                         CfgString, CfgReal, CfgDec, CfgDec, CfgHex);
  
  return;
}
