#include "FdIO.h"

/*-------------------------------------------------------private functions---*/
typedef struct FdStat FdStat;

struct FdStat{
  char*      suffix;         /* channel name suffix */
  char*      serName;        /* name of the serData*/
  int        nAdcMin;        // minimum number of ADC  channels over one second
  int        nAdcMax;        // maximum number of ADC  channels over one second
  int        nSmsMin;        // minimum number of SMS  channels over one second
  int        nSmsMax;        // maximum number of SMS  channels over one second
  int        nProcMin;       // minimum number of Proc channels over one second
  int        nProcMax;       // maximum number of Proc channels over one second
  FdAction*  action;         /* action object which hold this recast object */
};

/*---------------------------------------------------------------------------*/
void FdStatBuild(FdIO *fdIO, FrameH* frame, FdStat *s)
/*---------------------------------------------------------------------------*/
{
  FrAdcData *adc;
  FrSerData *sms;
  FdAction *action;
  FrProcData *proc;
  FrEvent *evt;
  int nAdc, nSms, nProc, len, nImages, adcSize, nEvt;
  double latency;
  char *p, *serData, *userInfo;

  latency = FrameLatency(frame);

  /*--------------------------------------- compute the number of channels---*/
  nAdc    = 0;
  nImages = 0;
  nSms    = 0;
  nProc   = 0;
  adcSize = 0;
  nEvt    = 0;

  if(frame->rawData != NULL)  {
    for(adc = frame->rawData->firstAdc; adc != NULL; adc = adc->next) {
       if(adc->data == NULL) continue;
       if(adc->data->nDim == 1) nAdc++;
       else                     nImages++;}
    for(sms = frame->rawData->firstSer; sms != NULL; sms = sms->next) nSms++;
    adcSize = FrameGetAdcSize(frame);}

  for(proc = frame->procData; proc != NULL; proc = proc->next) nProc++;
   
  for(evt = frame->event; evt != NULL; evt = evt->next) nEvt++;

  /*----------------------------------------------allocate FrSerData space---*/
  len = 1024;

  for(action = fdIO->actionsIn; action != NULL; action = action->next) {
     if(action->serData != NULL) len += strlen(action->serData) + 2;}
  for(action = fdIO->actionsOut; action != NULL; action = action->next) {
     if(action->serData != NULL) len += strlen(action->serData) + 2;}

  serData = malloc(len);
  if(serData == NULL) CfgMsgAddFatal("Malloc serData failed");

  /*---------------------------------------------fill the FrSerData buffer---*/
  p = serData;
  p += sprintf(p,"latency %.1f nAdc %d nImages %d nSms %d nProc %d adcSize %d",
	       latency, nAdc, nImages, nSms, nProc, adcSize);

 for(action = fdIO->actionsIn; action != NULL; action = action->next) {
   if(action->serData != NULL) p += sprintf(p," %s",action->serData);}
 for(action = fdIO->actionsOut; action != NULL; action = action->next) {
   if(action->serData != NULL) p += sprintf(p," %s",action->serData);}

  if(fdIO->actionsIn != NULL) {
    p += sprintf(p," inTime %.4f", fdIO->inputTime);}

  if(fdIO->actionsOut != NULL) {
    p += sprintf(p," outTime %.4f", fdIO->outputTime);}

  if(fdIO->serData != NULL) free(fdIO->serData);
  fdIO->serData = serData;

  /*----------------------for frames shorter than 1s, accumulate statistic---*/
  if(frame->dt < 1) {
    if(nAdc  < s->nAdcMin)  s->nAdcMin  = nAdc; 
    if(nAdc  > s->nAdcMax)  s->nAdcMax  = nAdc; 
    if(nProc < s->nProcMin) s->nProcMin = nProc; 
    if(nProc > s->nProcMax) s->nProcMax = nProc; 
    if(nSms  < s->nSmsMin)  s->nSmsMin  = nSms; 
    if(nSms  > s->nSmsMax)  s->nSmsMax  = nSms; 
   
    //--- skip the userInfo part if this is not the last frame of the second. 
    int last = 1.e9 * (1./frame->dt - 1) * frame->dt;
    if(frame->GTimeN != last) return;}
    
  /*------------------------------------------------allocate userInfo space---*/
  len = 1024;

  for(action = fdIO->actionsIn; action != NULL; action = action->next) {
     if(action->userInfo != NULL) len += strlen(action->userInfo) + 2;}
  for(action = fdIO->actionsOut; action != NULL; action = action->next) {
     if(action->userInfo != NULL) len += strlen(action->userInfo) + 2;}

  if(fdIO->moreUserInfo != NULL) len += sizeof(fdIO->moreUserInfo);

  userInfo = malloc(len);
  if(userInfo == NULL) CfgMsgAddFatal("Malloc userInfo failed");

  /*----------------------------------------------fill the userInfo buffer---*/
  p = userInfo;

  if(frame->dt >= 1) {
    p += sprintf(p,"%d-%d",frame->GTimeS,((int)frame->dt));}
  else if(frame->dt == 0.1 || frame->dt == 0.2 || frame->dt == 0.5) {
    p += sprintf(p,"%.1f-%g",frame->GTimeS+1.e-9*frame->GTimeN, frame->dt);}
  else {
    p += sprintf(p,"%.2f-%g",frame->GTimeS+1.e-9*frame->GTimeN, frame->dt);}

  p += sprintf(p," latency %.1f", latency);

  if(fdIO->moreUserInfo != NULL) p += sprintf(p," %s", fdIO->moreUserInfo);

  if(frame->dt < 1) {
    if(s->nAdcMax  > 0) p += sprintf(p," nAdc=%d/%d (min/max)",  s->nAdcMin, s->nAdcMax);
    if(s->nSmsMax  > 0) p += sprintf(p," nSms=%d/%d (min/max)",  s->nSmsMin, s->nSmsMax);
    if(s->nProcMax > 0) p += sprintf(p," nProc=%d/%d (min/max)", s->nProcMin,s->nProcMax);
    s->nAdcMin  = 1e9;
    s->nAdcMax  = 0;
    s->nSmsMin  = 1e9;
    s->nSmsMax  = 0;
    s->nProcMin = 1e9;
    s->nProcMax = 0;}
  else {
    if(nAdc  > 0) p += sprintf(p," nAdc=%d",  nAdc);
    if(nSms  > 0) p += sprintf(p," nSms=%d",  nSms);
    if(nProc > 0) p += sprintf(p," nProc=%d", nProc);}

  if(nEvt  >= 10000) p += sprintf(p," nEvt=%d",  nEvt);
  else if(nEvt > 0)  p += sprintf(p," nEvt=%4d", nEvt);
  p += sprintf(p,";");

 for(action = fdIO->actionsIn; action != NULL; action = action->next) {
   if(action->userInfo == NULL)     continue;
   if(strlen(action->userInfo) == 0) continue; 
   p += sprintf(p," %s",action->userInfo);}
 for(action = fdIO->actionsOut; action != NULL; action = action->next) {
   if(action->userInfo == NULL)     continue;
   if(strlen(action->userInfo) == 0) continue; 
   p += sprintf(p," %s",action->userInfo);}

  if(fdIO->actionsIn != NULL) {
    p += sprintf(p," inTime %.4f", fdIO->inputTime);}

  if(fdIO->actionsOut != NULL) {
    p += sprintf(p," outTime %.4f", fdIO->outputTime);}

  if(fdIO->userInfo != NULL) free(fdIO->userInfo);
  fdIO->userInfo = userInfo;

  CfgMsgAddUserInfo(fdIO->userInfo);

  return;
}
/*---------------------------------------------------------------------------*/
void FdStatProcessOne(FdStat *s, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  double sampleRate;
  FdIO* fdIO;
  char *p;
  int sLen;

  /*--------------------------------------------build the statistic string---*/
  fdIO = FdIOGetRoot();
  FdStatBuild(fdIO, frame, s);
  if(fdIO->userInfo == NULL) return;

  /*--------------------------------build FrSerData name for the first call---*/
  if(s->serName == NULL) {
    sLen = strlen(CfgGetCmName())+2;
    if(fdIO->chPrefix != NULL) sLen += strlen(fdIO->chPrefix);
    if(s->suffix != NULL)      sLen += strlen(s->suffix);
    s->serName = malloc(sLen);

    p = s->serName;
    if(fdIO->chPrefix != NULL && strcmp(fdIO->chPrefix,"NONE") != 0) 
      p += sprintf(p, "%s", fdIO->chPrefix);
    p   += sprintf(p, "%s", CfgGetCmName());
    if(s->suffix != NULL)
      p += sprintf(p, "%s", s->suffix);

    CfgMsgAddInfo("Will compute statistic (latency, nAdc...)"
		" SerData name:%s ",s->serName);}

  /*----------------------------------------------------build the FrSerData---*/
  sampleRate = 1./frame->dt;
  FrSerDataNew (frame, s->serName, frame->GTimeS + 1.e-9*frame->GTimeN, 
		fdIO->serData, sampleRate);

  return;
}
/*---------------------------------------------------------------------------*/
void FdStatProcess(FdStat *stat, FrameH* frame)
/*---------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdStatProcessOne(stat, frame);

  next = stat->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}
/*--------------------------------------------------------------------------*/
int FdStatNew(void *arg,   CfgDataType_t *data)
/*--------------------------------------------------------------------------*/
{
  FdStat *s, *sCheck;
  FRBOOL duplicated;
  FdAction *a;
  FdIO* fdIO;

  s = (FdStat*) calloc(1,sizeof(FdStat));
  if(s == NULL) CfgMsgAddFatal("malloc FdStat failed");

  FrStrCpy(&(s->suffix), CfgParseGetNextString(data));

  /*------------------is there another FDIN/OUT_STAT with the same suffix?---*/
  duplicated = FR_NO;
  fdIO = FdIOGetRoot();

  for(a = fdIO->actionsIn; a != NULL; a = a->next) {
    if(strcmp(a->type, "Stat") != 0) continue;
    sCheck = a->data;
    if(FrStrcmp(s->suffix, sCheck->suffix) == 0) duplicated = FR_YES;}

  for(a = fdIO->actionsOut; a != NULL; a = a->next) {
    if(strcmp(a->type, "Stat") != 0) continue;
    sCheck = a->data;
    if(FrStrcmp(s->suffix, sCheck->suffix) == 0) duplicated = FR_YES;}

  if(duplicated == FR_YES) 
    CfgMsgAddFatal("Two FDIN/OUT_STAT keys with the same suffix");

  /*------------------add the action to the list after the duplicated test---*/
  s->action = FdActionNew((FdAction**) arg, 
			  (void*) s, FdStatProcess, "Stat");

  return(CFG_OK);
}
/*---------------------------------------------------------------------------*/
void FdStatParserAdd(FdIO* fdIO)
/*---------------------------------------------------------------------------*/
{
 
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_STAT",
			 FdStatNew, (void *) &(fdIO->actionsIn), 1, 
			 CfgString);
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_STAT", 
			 FdStatNew, (void *) &(fdIO->actionsOut), 1,
			 CfgString);

 
  return;
}
