/*****************************************************************************/
/*  FdFrMrgr.c    Contains the frame merger code                             */
/*****************************************************************************/

#include <FdFrMrgr.h>
#include <stdio.h>

#define FDFRMRGR_BAD_LATENCY -1.e-20 /*a latency could not have this value*/

/*----------------------------------------- Declare private functions -------*/
FrameH*   FdFrMrgrOut(FdFrMrgr *mrg, int iOut);

/*---------------------------------------------------------------------------*/
int FdFrMrgrAddSource(FdFrMrgr* mrg, char* newSource)
/*---------------------------------------------------------------------------*/
{
  int i, iSource;

  CfgMsgAddInfo("FrameMerger: add %s as new source", newSource);
  for(i = 0; i< mrg->nSources; i++) {
    if(strcmp(mrg->sNames[i], newSource) == 0) break;}

  if(i != mrg->nSources) {
    CfgMsgAddWarning("%s is already a source for the frame merger", newSource);
    return(i+1);}

  mrg->nSources++;
  /*------------------------------------------------------allocate buffers---*/
  mrg->latencies=(double*)realloc(mrg->latencies,
				  mrg->nSources *mrg->nFrames *sizeof(double));
  mrg->sNames   =(char**) realloc(mrg->sNames,   mrg->nSources*sizeof(char*));
  mrg->missingSt=(FRBOOL*)realloc(mrg->missingSt,mrg->nSources*sizeof(FRBOOL));
  mrg->missingPr=(FRBOOL*)realloc(mrg->missingPr,mrg->nSources*sizeof(FRBOOL));

  mrg->serDataSize = 100*mrg->nSources;
  mrg->serData     = (char*) realloc(mrg->serData,    mrg->serDataSize);
  mrg->missingSrc  = (char*) realloc(mrg->missingSrc, mrg->serDataSize);
  mrg->status      = (char*) realloc(mrg->status,     mrg->serDataSize+60);
  if(mrg->latencies  == NULL ||
     mrg->sNames     == NULL ||
     mrg->missingSt  == NULL ||
     mrg->missingPr  == NULL ||
     mrg->serData    == NULL ||
     mrg->missingSrc == NULL ||
     mrg->status     == NULL)
    CfgMsgAddFatal("FdFrMrgrNew failed nFrames=%d nSource=%d", 
		   mrg->nFrames, mrg->nSources);

  /*--------------------------------initialize the newlly allocated  space---*/
  mrg->sNames   [mrg->nSources-1] = NULL;
  mrg->missingSt[mrg->nSources-1] = FR_NO;
  mrg->missingPr[mrg->nSources-1] = FR_NO;

  for(i = 0; i<mrg->nFrames; i++) {
    iSource = mrg->nSources-1;
    mrg->latencies[i + mrg->nFrames*iSource] = FDFRMRGR_BAD_LATENCY;}

  /*------------------------------------------------fill the sources names---*/
  FrStrCpy(&(mrg->sNames[mrg->nSources-1]), newSource);
  if(mrg->sNames[mrg->nSources-1] == NULL) 
    CfgMsgAddFatal("FdFrMrgrNew: malloc");

  for(i = 0; i < mrg->nSources; i++){
    CfgMsgAddInfo("  FdFrMrgrNew: source %3d is %s\n",i,mrg->sNames[i]);}

  return(0);
}
/*---------------------------------------------------------------------------*/
int FdFrMrgrRemoveSource(FdFrMrgr* mrg, char* newSource)
/*---------------------------------------------------------------------------*/
{
  int i, j;

  CfgMsgAddInfo(" FrameMerger: remove %s from the source list", newSource);
  for(i = 0; i< mrg->nSources; i++) {
    if(strcmp(mrg->sNames[i], newSource) == 0) break;}

  if(i == mrg->nSources) {
    CfgMsgAddWarning("%s is not a source for the frame merger",newSource);
    return(1);}

  if(mrg->nSources <= 1) {
    CfgMsgAddWarning("Do not remove this source to keep at least one source");
    return(2);}

  /*------move this source at the end of the list and reduce the list by 1---*/
  mrg->nSources--;
  for(; i<mrg->nSources; i++) {
    mrg->sNames[i]    = mrg->sNames[i+1];
    mrg->missingSt[i] = mrg->missingSt[i+1];
    mrg->missingPr[i] = mrg->missingPr[i+1];
    for(j = 0; j<mrg->nFrames; j++) {
      mrg->latencies[j + mrg->nFrames*i] = mrg->latencies[j + mrg->nFrames*(i+1)];}}

  /*-----------------------------------------dump the new list of sources---*/
  for(i = 0; i < mrg->nSources; i++){
    CfgMsgAddInfo("   FdFrMrgrRemove: source %3d is %s\n",i,mrg->sNames[i]);}

  return(0);
}
/*---------------------------------------------------------------------------*/
FdFrMrgr* FdFrMrgrNew(int nFrames, char* sList)
/*---------------------------------------------------------------------------*/
{
  FdFrMrgr *mrg;
  int nSources, i, iSource;
  FrTag *tList, *t;

  mrg = (FdFrMrgr*) calloc(1, sizeof(FdFrMrgr));
  if(mrg == NULL) CfgMsgAddFatal("FdFrMrgrNew failed");

  mrg->nFrames = nFrames;
  if(sList != NULL) FrStrCpy(&(mrg->sList),sList);
  mrg->debug = 0;

  nFrames  = mrg->nFrames;
  mrg->lastGpsOut = -1;
  mrg->nOuts = 0;

  /*--------- break the list of input names to single names and count them---*/
  tList = FrTagNew(mrg->sList); 
  nSources = 0;
  for(t = tList; t != NULL; t = t->next) {nSources++;}
  mrg->nSources = nSources; 
  mrg->nMissing = nSources;

  /*------------------------------------------------------allocate buffers---*/
  mrg->frames    = (FrameH**) calloc(nFrames, sizeof(FrameH*));
  mrg->gps       = (double*)  calloc(nFrames, sizeof(double));
  mrg->latencies = (double*)  calloc(nFrames*nSources,sizeof(double));
  mrg->sNames    = (char**)   calloc(nSources, sizeof(char*));
  mrg->missingSt = (FRBOOL*)  calloc(nSources, sizeof(FRBOOL));
  mrg->missingPr = (FRBOOL*)  calloc(nSources, sizeof(FRBOOL));

  mrg->serDataSize = 80*nSources;
  mrg->serData     = (char*) calloc(mrg->serDataSize, sizeof(char));
  mrg->missingSrc  = (char*) calloc(mrg->serDataSize, sizeof(char));
  mrg->status      = (char*) calloc(mrg->serDataSize+60, sizeof(char));
  if(mrg->frames     == NULL ||
     mrg->gps        == NULL || 
     mrg->latencies  == NULL ||
     mrg->sNames     == NULL ||
     mrg->missingSt  == NULL ||
     mrg->missingPr  == NULL ||
     mrg->serData    == NULL ||
     mrg->missingSrc == NULL ||
     mrg->status     == NULL)
    CfgMsgAddFatal("FdFrMrgrNew failed nFrames=%d nSource=%d", 
		   nFrames, nSources);

  for(i = 0; i<mrg->nFrames; i++) {
    for(iSource = 0; iSource<mrg->nSources; iSource++) {
      mrg->latencies[i + mrg->nFrames*iSource] = FDFRMRGR_BAD_LATENCY;}}

  sprintf(mrg->status,"FrMerger: starting"); 

  /*------------------------------------------------fill the sources names---*/
  i = 0;
  for(t = tList; t != NULL; t = t->next) {
    FrStrCpy(&(mrg->sNames[i]), t->start);
    if(mrg->sNames[i] == NULL) CfgMsgAddFatal("FdFrMrgrNew %d",t->length);
    CfgMsgAddInfo("FdFrMrgrNew: source %d is %s\n",i,mrg->sNames[i]);
    i++;}

  return(mrg);
}
/*---------------------------------------------------------------------------*/
int FdFrMrgrGetNParts(FdFrMrgr* mrg, int iFrame)
/*---------------------------------------------------------------------------*/
/* This function returns the number of frame parts for this frame index      */
/*---------------------------------------------------------------------------*/
{
  int nParts, iSource;

  nParts = 0;
  for(iSource=0; iSource < mrg->nSources; iSource++) {
    if(mrg->latencies[iFrame + mrg->nFrames*iSource] 
       != FDFRMRGR_BAD_LATENCY) nParts++;}

  return(nParts);
}
/*---------------------------------------------------------------------------*/
FrameH* FdFrMrgrOut(FdFrMrgr *mrg, int iOut)
/*---------------------------------------------------------------------------*/
/* This function prepare a frame to be returned and do some monitoring       */
/*---------------------------------------------------------------------------*/
{
  FrameH* returnedFrame;
  double latency, gps;
  int iSource;
  char* s, *sEnd, *m, *mEnd;

  returnedFrame = mrg->frames[iOut];
  gps = returnedFrame->GTimeS + 1.e-9*returnedFrame->GTimeN;
  mrg->nOuts++;

  /*-------------------------------------------print missing/back sources---*/
  for(iSource=0; iSource < mrg->nSources; iSource++) {
    if(mrg->latencies[iOut + mrg->nFrames*iSource] == FDFRMRGR_BAD_LATENCY) {
      if(mrg->missingPr[iSource] == FR_NO) {
	CfgMsgAddWarning("FdFrMrgr:%.1f frames from %s start to be missing (out)",
			 gps, mrg->sNames[iSource]);
	mrg->missingPr[iSource] = FR_YES;}}
    else if(mrg->missingPr[iSource] == FR_YES) {
      CfgMsgAddInfo("FdFrMrgr:%.1f frames from %s are back",
			 gps, mrg->sNames[iSource]);
      mrg->missingPr[iSource] = FR_NO;}}

  /*------------------------------------first build monitoring information---*/
  s    = mrg->serData;
  m    = mrg->missingSrc;
  sEnd = s + mrg->serDataSize;
  mEnd = m + mrg->serDataSize;

  mrg->nMissing = 0;
  for(iSource=0; iSource < mrg->nSources; iSource++) {
    latency = mrg->latencies[iOut + mrg->nFrames*iSource];
    s += snprintf(s,(int)(sEnd-s),"%s_latency %.3f ",
		  mrg->sNames[iSource],latency);
    if(latency == FDFRMRGR_BAD_LATENCY) {
      m += snprintf(m,(int)(mEnd-m),"%s ",mrg->sNames[iSource]);
      mrg->nMissing++;}}

  /*---------------------------------------------- fill status information---*/
  if(m != mrg->missingSrc) 
    sprintf(mrg->status," Missing %d sources: %s",
            mrg->nSources - FdFrMrgrGetNParts(mrg, iOut), mrg->missingSrc);
  else
    sprintf(mrg->status," All frame parts (%d) available", mrg->nSources);

  /*--------------------reset the slot used by the frame we are returning---*/
  for(iSource=0; iSource < mrg->nSources; iSource++) {
    mrg->latencies[iOut + mrg->nFrames*iSource] = FDFRMRGR_BAD_LATENCY;}

  mrg->gps[iOut] = 0;
  mrg->frames[iOut] = NULL;
  mrg->lastGpsOut   = gps;

  if(mrg->debug > 0) 
    CfgMsgAddInfo("FdFrMrgrOut: output frame from slot %d gps=%.1f %s",
		  iOut, gps, mrg->status);
  return(returnedFrame);
}
/*---------------------------------------------------------------------------*/
FRBOOL FdFrMrgrFeed(FdFrMrgr* mrg, FrameH* frame, char* sender)
/*---------------------------------------------------------------------------*/
/* This function feed the frame merger object                                */
/*---------------------------------------------------------------------------*/
{
  int iSource, iFrame, mytime;
  double gps, latency;

  latency = FrameLatency(frame);
  gps = frame->GTimeS + 1.e-9*frame->GTimeN;

  /*-------------------------------check if the frame is part of the list ---*/
  for(iSource = 0; iSource<mrg->nSources; iSource++) {
    if(strcmp(sender, mrg->sNames[iSource]) == 0) break;}
  if(iSource >= mrg->nSources) {
    CfgMsgAddError("FdFrMrgrFeed: reject unwanted frame from %s at %.1f",
		   sender, gps);
    FrameFree(frame);
    return(FR_NO);}

  /*------------------------------find if a source is back sending frames---*/
  if(mrg->missingSt[iSource] == FR_YES) {
    CfgMsgAddInfo("FdFrMrgrFeed: %.1f receiving again frames from %s ",
                   gps, mrg->sNames[iSource]);
    mrg->missingSt[iSource] = FR_NO;}

  /*----------------------------------------------------reject late frames---*/
  if(gps <= mrg->lastGpsOut && mrg->nOuts > 0) {
    CfgMsgAddWarning("FdFrMrgrFeed: reject too old frame (%.1f) from %s;"
		     " last produced was %.1f", gps, sender, mrg->lastGpsOut);
    FrameFree(frame);
    return(FR_NO);}

  /*--------------- search if the frame part already exist and then add it---*/
  for(iFrame = 0; iFrame<mrg->nFrames; iFrame++) {
    if(mrg->gps[iFrame] == gps) break;}

  if(iFrame < mrg->nFrames) {
    if(mrg->latencies[iFrame + mrg->nFrames*iSource] > FDFRMRGR_BAD_LATENCY) {
      CfgMsgAddError("FdFrMrgrFeed: %s has already sent a frame part for %.1f",
		     sender, gps);
      FrameFree(frame);
      return(FR_NO);}

    if(frame->dt != mrg->frames[iFrame]->dt) {
      CfgMsgAddError("FdFrMrgrFeed: Reject frame: length missmatch: "
                     "dt=%.1f and %.1f (%s)",
                     frame->dt, mrg->frames[iFrame]->dt, sender);
      FrameFree(frame);
      return(FR_NO);} 

    if(mrg->debug > 0) 
      CfgMsgAddInfo("FdFrMrgrFeed:  merge %.1f from %s (%d)" 
		    "with iFrame=%d nParts=%d", gps, sender, 
		    iSource, iFrame, FdFrMrgrGetNParts(mrg,iFrame));
    FrHistoryFree(frame->history);
    frame->history = NULL;
    FrameMerge(mrg->frames[iFrame], frame);}

  /*--------this is a new frame part; store it in a free slot of the table---*/
  else{
    for(iFrame=0;  iFrame<mrg->nFrames; iFrame++) {
      if(mrg->gps[iFrame] == 0) break;}
    if(iFrame >= mrg->nFrames) {
      CfgMsgAddError("FdFrMrgrFeed: no free slot at %.1f",gps);
      FrameFree(frame);
      return(FR_NO);}

    if(mrg->debug > 0) 
      CfgMsgAddInfo("FdFrMrgrFeed: store %.1f from %s (%d) with iFrame=%d",
		    gps, sender, iSource, iFrame);
    mrg->gps[iFrame]    = gps;
    FrHistoryFree(frame->history);
    mytime = FrGetCurrentGPS();
    frame->history = FrHistoryNew("FdIO_merger", mytime, mrg->sList);
    mrg->frames[iFrame] = frame;}

  mrg->latencies[iFrame + mrg->nFrames*iSource] = latency;

  return(FR_YES);
}
/*---------------------------------------------------------------------------*/
FrameH* FdFrMrgrIsReady(FdFrMrgr* mrg)
/*---------------------------------------------------------------------------*/
/* This function check if a merged frame is ready for output and returns it  */
/*---------------------------------------------------------------------------*/
{
  int iSource, iOldest, iFrame, iFirst;
  double latency, minLatency;

  /*------------------------------------find the oldest frame in the list---*/
  iOldest = -1;
  for(iFrame=0;  iFrame<mrg->nFrames; iFrame++) {
    if(mrg->gps[iFrame] == 0) continue;
    if(iOldest < 0) iOldest = iFrame;
    if(mrg->gps[iFrame] < mrg->gps[iOldest]) iOldest = iFrame;}
 
  if(iOldest < 0) return(NULL);  /*--------------------no frames avalable---*/

  /*-------------------------------output the oldest merged frame 
    ----------------------------if all active sources have been received ---*/
  for(iSource=0; iSource < mrg->nSources; iSource++) {
    latency = mrg->latencies[iOldest + mrg->nFrames*iSource];
    if((latency == FDFRMRGR_BAD_LATENCY) &&      /* frame part not received */
      (mrg->missingSt[iSource] == FR_NO))break;} /* and the source is active*/
  if(iSource >= mrg->nSources) {
    if(mrg->debug > 0)  CfgMsgAddInfo("FdFrMrgr: output oldest active frame: "
				"%.1f", mrg->gps[iOldest]);
    return(FdFrMrgrOut(mrg, iOldest));}

  /*------------------if a youngest frame is fully ready, send out the oldest 
    --------------------------------------since no extra part is expected---*/
  for(iFrame=0;  iFrame<mrg->nFrames; iFrame++) {
    if(mrg->gps[iFrame] == 0) continue;
    if(FdFrMrgrGetNParts(mrg, iFrame) == mrg->nSources) {
      CfgMsgAddInfo("FdFrMrgr: frame %.1f will have all sources. "
	 	      "Output frame %.1f",
		      mrg->gps[iFrame], mrg->gps[iOldest]);
      return(FdFrMrgrOut(mrg, iOldest));}}
 
  /*-----------------------if there are still empty frame slots do nothing---*/
   for(iFrame=0;  iFrame<mrg->nFrames; iFrame++) {
    if(mrg->gps[iFrame] == 0) return(NULL);}

  /*---------------there are no more empty slots, returns the oldest frame---*/
  /*-----------------find the first received frame (for debugging purpose)---*/
  iFirst = 0;
  minLatency = 1e20;
  for(iSource=0; iSource < mrg->nSources; iSource++) {
    latency = mrg->latencies[iOldest + mrg->nFrames*iSource];
    if(latency == FDFRMRGR_BAD_LATENCY) continue;
    if(latency > minLatency) continue;
    minLatency = latency;
    iFirst = iSource;}

  CfgMsgAddWarning("FdFrMrgr: Could not wait longer for frame parts,"
                   " output %.1f, nSources=%d first:%s",
		   mrg->gps[iOldest], FdFrMrgrGetNParts(mrg, iOldest),
                   mrg->sNames[iFirst]);

  /*---------------------------------find if a source start to be missing---*/
  for(iSource=0; iSource < mrg->nSources; iSource++) {
    latency = mrg->latencies[iOldest + mrg->nFrames*iSource];
    if(latency == FDFRMRGR_BAD_LATENCY) {
      if(mrg->missingSt[iSource] == FR_NO)
        CfgMsgAddWarning("FdFrMrgr:%.1f frames from %s start to be missing"
		" (isReady)",  mrg->gps[iOldest], mrg->sNames[iSource]);
      mrg->missingSt[iSource] = FR_YES;}}

  return(FdFrMrgrOut(mrg, iOldest));
}
/*---------------------------------------------------------------------------*/
FrameH* FdFrMrgrProc(FdFrMrgr* mrg, FrameH* frame, char* sender)
/*---------------------------------------------------------------------------*/
/* This function merge frames from the cm input                              */
/*---------------------------------------------------------------------------*/
{
  if(frame != NULL) FdFrMrgrFeed(mrg, frame, sender);

  frame = FdFrMrgrIsReady(mrg);

  return(frame);
}
