/*-----------------------------------------------------------------------------*/
/* FdEvents.c to manage events */
/*-----------------------------------------------------------------------------*/

#include "FdIO.h"

/*---------------------------------------------------------private structure---*/
typedef struct FdEvtReject FdEvtReject;


struct FdEvtReject{         /* to remove unwanted FrEvents            */
  FrTag*       tag;         /* tag used to select event               */
  char*        parameter;   /* parameter on which the cut is applied  */
  double       vMin;        /* minimum value                          */
  double       vMax;        /* maximum value                          */
  int          downSampling;/* downSampling factor applied            */
  int          nEvents;     /* events count                           */
  FdAction*    action;      /* FdAction object                         */
};  

void FdEvtRejectProcess(FdEvtReject *r, FrameH* frame);

/*-----------------------------------------------------------------------------*/
int FdEvtRejectNew(void *arg, CfgDataType_t *dataType)
/*-----------------------------------------------------------------------------*/
{
  FdEvtReject *r;
  char *tag;

  r = calloc(sizeof(FdEvtReject), 1);
  if(r == NULL) CfgMsgAddFatal("malloc FdEvtReject failed");

  r->action = FdActionNew((FdAction**) arg,   (void*) r, 
			   FdEvtRejectProcess, "EventsReject");

  FrStrCpy(&tag,            CfgParseGetNextString(dataType));
  FrStrCpy(&(r->parameter), CfgParseGetNextString(dataType));
  r->vMin          =        CfgParseGetNextReal  (dataType);
  r->vMax          =        CfgParseGetNextReal  (dataType);
  r->downSampling  =        CfgParseGetNextDec   (dataType);

  if(r->downSampling < 0)
    CfgMsgAddFatal("Events dowsampling should be >= 0 and not %d",
		   r->downSampling);

  CfgMsgAddInfo("Events %s with %g < %s < %g will be down sampled by %d", 
		tag, r->vMin, r->parameter, r->vMax, r->downSampling);

  r->tag = FrTagNew(tag);
  free(tag);

  r->nEvents = 0;

  return(CFG_OK);
}
/*-----------------------------------------------------------------------------*/
void  FdEvtRejectProcessOne(FdEvtReject *r, FrameH* frame)
/*-----------------------------------------------------------------------------*/
{
  FrEvent *evt, **root, **next;
  double value;
  int id;

  for(root = &(frame->event); *root != NULL; root = next) {
    evt = *root;
    next = &(evt->next);

    if(FrTagMatch(r->tag, evt->name) == FR_NO) continue;

    if(strcmp(r->parameter,"amplitude") == 0) 
      value = evt->amplitude;
    else
      value = FrEventGetParam(evt, r->parameter);
    if(value <  r->vMin) continue;
    if(value >= r->vMax) continue;
    r->nEvents++;

    if(r->nEvents == r->downSampling && r->downSampling > 0) {
      r->nEvents = 0;
      id = FrEventGetParamId(evt, "downSampling");
      if(id < 0) FrEventAddParam(evt, "downSampling", r->downSampling); 
      else       evt->parameters[id] *= r->downSampling;
      continue;}

    *root = evt->next;
    FrEventFree(evt);
    next = root;
  }

  return;
}
/*-----------------------------------------------------------------------------*/
void FdEvtRejectProcess(FdEvtReject *r, FrameH* frame)
/*-----------------------------------------------------------------------------*/
{
  FdAction *next;

  if(frame != NULL) FdEvtRejectProcessOne(r, frame);

  next = r->action->next;
  if(next != NULL) next->action(next->data, frame);

  return;
}

/*-----------------------------------------------------------------------------*/
void FdEvtParserAdd(FdIO *fdIO)
/*-----------------------------------------------------------------------------*/
{
  CfgParseGetFunctionAdd(fdIO->parser, "FDIN_REJECT_EVENTS", 
			 FdEvtRejectNew,  (void *) &(fdIO->actionsIn), 5,
			 CfgString, CfgString, CfgReal, CfgReal, CfgDec);
 
  CfgParseGetFunctionAdd(fdIO->parser, "FDOUT_REJECT_EVENTS", 
			 FdEvtRejectNew,  (void *) &(fdIO->actionsOut), 5,
			 CfgString, CfgString, CfgReal, CfgReal, CfgDec); 

  return;
}
