/*---------------------------------------------------------------------------*/
/*        Copyright (c) 1996 LAL Orsay, UPS-IN2P3-CNRS (France).             */
/*                                                                           */
/* Redistribution and use in source and binary forms, with or without        */
/* modification, are permitted provided that the following conditions        */
/* are met:                                                                  */
/* 1. Redistributions of source code must retain the above copyright         */
/*    notice, this list of conditions and the following disclaimer.          */
/* 2. Redistributions in binary form must reproduce the above copyright      */
/*    notice, this list of conditions and the following disclaimer in the    */
/*    documentation and/or other materials provided with the distribution.   */
/* 3. All advertising materials mentioning features or use of this software  */
/*    must display the following acknowledgement:                            */
/*      This product includes software developed by the Computer Application */
/*      Development Group at LAL Orsay (Laboratoire de l'Accelerateur        */
/*      Linaire - UPS-IN2P3-CNRS).                                           */
/* 4. Neither the name of the Institute nor of the Laboratory may be used    */
/*    to endorse or promote products derived from this software without      */
/*    specific prior written permission.                                     */
/*                                                                           */
/* THIS SOFTWARE IS PROVIDED BY THE LAL AND CONTRIBUTORS ``AS IS'' AND       */
/* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE     */
/* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR        */
/* PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE LAL OR CONTRIBUTORS BE      */
/* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR       */
/* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF      */
/* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS  */
/* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN   */
/* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)   */
/* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF    */
/* THE POSSIBILITY OF SUCH DAMAGE.                                           */
/*---------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <CmTransaction.h>

/*----------------------------------------------------------------*/
typedef struct _CmTransactionRec
{
  int id;
  char* info;
  void* object;
  CmTransactionState state;
} CmTransactionRec;
/*----------------------------------------------------------------*/

/*----------------------------------------------------------------*/
static CmTransactionRec* TransactionArray = NULL;
static int               Transactions = 0;
static int               TerminatedTransactions = 0;
/*----------------------------------------------------------------*/

/*----------------------------------------------------------------*/
static int get_new_id ()
/*----------------------------------------------------------------*/
{
  time_t t;
  int id;
  static int offset = 0;

  offset++;

  id = (int) time (&t) + (getpid () * 100000000) + offset;

    /* printf ("id = %d\n", id); */

  return (id);
}

/*----------------------------------------------------------------*/
static void CmTransactionSetState (CmTransaction transaction, 
                                   CmTransactionState state)
/*----------------------------------------------------------------*/
{
  if (state == transaction->state) return;

  if (transaction->state == CmTransactionTerminated)
    {
      TerminatedTransactions--;
    }
  else if (state == CmTransactionTerminated)
    {
      TerminatedTransactions++;
    }

  transaction->state = state;
}

/*----------------------------------------------------------------*/
static void CmTransactionSetInfo (CmTransaction transaction, 
                                  const char* info)
/*----------------------------------------------------------------*/
{
  if (transaction->info != NULL)
    {
      free (transaction->info);
      transaction->info = NULL;
    }

  if (info == NULL) return;
  if (strlen (info) == 0) return;

  transaction->info = (char*) malloc (strlen (info) + 1);
  strcpy (transaction->info, info);
}

/*----------------------------------------------------------------*/
static void CmTransactionSetObject (CmTransaction transaction, 
                                    void* object)
/*----------------------------------------------------------------*/
{
  transaction->object = object;
}

/*----------------------------------------------------------------*/
static CmTransaction CmFindTransaction (int id)
/*----------------------------------------------------------------*/
{
  int number;

  if (TransactionArray == NULL) return (NULL);

  for (number = 0; number < Transactions; number++)
    {
      CmTransaction transaction = NULL;

      transaction = &TransactionArray[number];
      if (transaction->id == id) return (transaction);
    }

  return (NULL);
}

/*----------------------------------------------------------------*/
int CmOpenTransaction (const char* info, void* object)
/*----------------------------------------------------------------*/
{
  CmTransaction transaction = NULL;
  int number;
  int id;

  id = get_new_id ();

  if (TransactionArray != NULL)
    {
      for (number = 0; number < Transactions; number++)
        {
          transaction = &TransactionArray[number];
          if (transaction->state == CmTransactionClosed)
            {
              transaction->id = id;
              CmTransactionSetInfo (transaction, info);
              CmTransactionSetObject (transaction, object);
              CmTransactionSetState (transaction, CmTransactionPending);

              return (id);
            }
        }
    }

  Transactions++;

  if (TransactionArray == NULL)
    {
      TransactionArray = (CmTransaction) malloc (Transactions * 
                                                 sizeof (CmTransactionRec));
    }
  else
    {
      TransactionArray = (CmTransaction) realloc (
          TransactionArray, Transactions * sizeof (CmTransactionRec));
    }

  transaction = &TransactionArray[Transactions - 1];

  transaction->id     = id;
  transaction->info   = NULL;
  transaction->object = NULL;
  transaction->state  = CmTransactionPending;

  CmTransactionSetInfo (transaction, info);
  CmTransactionSetObject (transaction, object);
              
  return (id);
}

/*----------------------------------------------------------------*/
CmTransactionStatus CmCloseTransaction (int id)
/*----------------------------------------------------------------*/
{
  CmTransaction transaction = NULL;

  transaction = CmFindTransaction (id);

  if (transaction == NULL) return (CmTransactionNotFound);

  CmTransactionSetState (transaction, CmTransactionClosed);

  return (CmTransactionOk);
}

/*----------------------------------------------------------------*/
CmTransactionStatus CmTerminateTransaction (int id)
/*----------------------------------------------------------------*/
{
  CmTransaction transaction = NULL;

  transaction = CmFindTransaction (id);

  if (transaction == NULL) return (CmTransactionNotFound);

    /*
      The transition from Closed to Terminated is not permitted
      */
  if (transaction->state == CmTransactionClosed)
    {
      return (CmTransactionAlreadyClosed);
    }

  CmTransactionSetState (transaction, CmTransactionTerminated);

  return (CmTransactionOk);
}

/*----------------------------------------------------------------*/
CmTransactionStatus CmRestartTransaction (int id)
/*----------------------------------------------------------------*/
{
  CmTransaction transaction = NULL;

  transaction = CmFindTransaction (id);

  if (transaction == NULL) return (CmTransactionNotFound);

    /*
      The transition from Closed to Pending is not permitted
      */
  if (transaction->state == CmTransactionClosed)
    {
      return (CmTransactionAlreadyClosed);
    }

  CmTransactionSetState (transaction, CmTransactionPending);

  return (CmTransactionOk);
}

/*----------------------------------------------------------------*/
CmTransactionState CmGetTransactionState (int id)
/*----------------------------------------------------------------*/
{
  CmTransaction transaction = NULL;

  transaction = CmFindTransaction (id);

  if (transaction == NULL) return (CmTransactionClosed);

  return (transaction->state);
}

/*----------------------------------------------------------------*/
char* CmGetTransactionInfo (int id)
/*----------------------------------------------------------------*/
{
  CmTransaction transaction = NULL;

  transaction = CmFindTransaction (id);

  if (transaction == NULL) return ("");

  if (transaction->info == NULL) return ("");
  return (transaction->info);
}

/*----------------------------------------------------------------*/
void* CmGetTransactionObject (int id)
/*----------------------------------------------------------------*/
{
  CmTransaction transaction = NULL;

  transaction = CmFindTransaction (id);

  if (transaction == NULL) return (NULL);

  return (transaction->object);
}

/*----------------------------------------------------------------*/
int CmIsTransactionTerminated (int id)
/*----------------------------------------------------------------*/
{
  CmTransaction transaction = NULL;

  transaction = CmFindTransaction (id);

  if (transaction == NULL) return (1);

  return (transaction->state != CmTransactionPending);
}

/*----------------------------------------------------------------*/
int CmTerminatedTransactionCount ()
/*----------------------------------------------------------------*/
{
  return (TerminatedTransactions);
}

/*----------------------------------------------------------------*/
void CmDumpTransactions ()
/*----------------------------------------------------------------*/
{
  int number;
  static char* state_name[] = 
  {
    (char*) "Closed",
    (char*) "Pending",
    (char*) "Terminated"
  };

  if (TransactionArray == NULL) return;

  for (number = 0; number < Transactions; number++)
    {
      CmTransaction transaction;
      char* info = (char*) "";

      transaction = &TransactionArray[number];
      if (transaction->info != NULL) info = transaction->info;

      printf ("# Transaction %s id=%d state=%s\n", 
              info, 
              transaction->id, 
              state_name[transaction->state]);
    }
}
