/*---------------------------------------------------------------------------*/
/*        Copyright (c) 1996 LAL Orsay, UPS-IN2P3-CNRS (France).             */
/*                                                                           */
/* Redistribution and use in source and binary forms, with or without        */
/* modification, are permitted provided that the following conditions        */
/* are met:                                                                  */
/* 1. Redistributions of source code must retain the above copyright         */
/*    notice, this list of conditions and the following disclaimer.          */
/* 2. Redistributions in binary form must reproduce the above copyright      */
/*    notice, this list of conditions and the following disclaimer in the    */
/*    documentation and/or other materials provided with the distribution.   */
/* 3. All advertising materials mentioning features or use of this software  */
/*    must display the following acknowledgement:                            */
/*      This product includes software developed by the Computer Application */
/*      Development Group at LAL Orsay (Laboratoire de l'Accelerateur        */
/*      Linaire - UPS-IN2P3-CNRS).                                           */
/* 4. Neither the name of the Institute nor of the Laboratory may be used    */
/*    to endorse or promote products derived from this software without      */
/*    specific prior written permission.                                     */
/*                                                                           */
/* THIS SOFTWARE IS PROVIDED BY THE LAL AND CONTRIBUTORS ``AS IS'' AND       */
/* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE     */
/* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR        */
/* PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE LAL OR CONTRIBUTORS BE      */
/* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR       */
/* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF      */
/* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS  */
/* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN   */
/* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)   */
/* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF    */
/* THE POSSIBILITY OF SUCH DAMAGE.                                           */
/*---------------------------------------------------------------------------*/
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <CmMessage.h>
#include <Cvt.h>
#include <CmHandle.h>

#ifdef _Cvt_
#undef _Cvt_
#endif

#ifdef vms
#define _Cvt_
/*
#include <ucx$rpcxdr.h>
*/
#include <Xdr.h>
#endif

#ifdef _WIN32
#define _Cvt_
#include <xdr.h>
#endif

#ifndef _Cvt_
#define _Cvt_
#include <rpc/types.h>
#include <rpc/xdr.h>
#endif



/*--------------------------------------------------------------*/
typedef struct _CvtRec
{
  CvtSwap swap;
  char shortMove[sizeof(short)];
  char intMove[sizeof(int)];
} CvtRec;
/*--------------------------------------------------------------*/

/*--------------------------------------------------------------*/
static int CvtInitialized = 0;
static int CvtRecNumber   = 0;

static List CvtList;
/*--------------------------------------------------------------*/

/*--------------------------------------------------------------*/
/*    Prototypes locaux.                                        */
/*--------------------------------------------------------------*/

/*--------------------------------------------------------------*/
/*    Initialisation globale.                                   */
/*--------------------------------------------------------------*/
void CvtForceStartup ()
{
  CvtInitialized = 0;
  CvtStartup ();
}

/*--------------------------------------------------------------*/
/*    Initialisation globale.                                   */
/*--------------------------------------------------------------*/
void CvtStartup ()
{
  if (!CvtInitialized)
    {
      CvtInitialized = 1; 
      CvtRecNumber = 0;

      ListInit (&CvtList);
    }
}

/*--------------------------------------------------------------*/
/*    Terminaison globale.                                      */
/*--------------------------------------------------------------*/
void CvtCleanup ()
{
  if (CvtInitialized)
    {
      Entry* entry;
      List* list = &CvtList;

      CvtInitialized = 0;

      entry = list->first;

      while (entry != NULL)
        {
          Cvt cvt;

          cvt = (Cvt) entry->reference;
          if (cvt != NULL)
            {
              CvtDelete (cvt);
              entry->reference = NULL;
            }

          entry = entry->next;
        }

      ListClear (list);
    }
}

/*--------------------------------------------------------------*/
void CvtCheckBalance ()
/*--------------------------------------------------------------*/
{
  if (CvtRecNumber != 0)
    {
      fprintf (stderr, "Cvt>>> Bad balance on Cvt objets : %d\n",
               CvtRecNumber);
    }
}

/*--------------------------------------------------------------*/
static Cvt CvtFind (char* swap)
/*--------------------------------------------------------------*/
{
  Cvt result = NULL;
  Entry* entry;
  List* list = &CvtList;

  entry = list->first;

  while (entry != NULL)
    {
      Cvt cvt;

      cvt = (Cvt) entry->reference;
      if ((cvt != NULL) && (!strcmp ((char*) cvt->swap, swap)))
        {
          result = cvt;
          break;
        }
      entry = entry->next;
    }

  return (result);
}

/*--------------------------------------------------------------*/
/*    Cree un objet Cvt.                                        */
/*--------------------------------------------------------------*/
Cvt CvtNew (char* swap)
{
  static union
    {
      short s;
      unsigned char b[sizeof(short)];
    } shortLoc, shortExt;

  static union
    {
      int i;
      unsigned char b[sizeof(int)];
    } intLoc, intExt;

  Cvt cvt;
  int items;
  int i;
  int j;

  if (!swap) return (NULL);

  cvt = CvtFind (swap);
  if (cvt != NULL) return (cvt);

  sscanf (CvtGetSwap(), "%hx-%x", &shortLoc.s, &intLoc.i);
  items = sscanf (swap, "%hx-%x", &shortExt.s, &intExt.i);
  if (!items) return (NULL);

  cvt = (Cvt) malloc (sizeof (CvtRec));
  CvtRecNumber++;

  strcpy ((char*) cvt->swap, swap);

  for (i = 0; i < sizeof(short); i++)
    {
      cvt->shortMove[i] = 0;
      for (j = 0; j < sizeof(short); j++)
	{
	  if (shortExt.b[i] == shortLoc.b[j])
	    {
	      cvt->shortMove[i] = j;
	      break;
	    }
	}
    }

  for (i = 0; i < sizeof(int); i++)
    {
      cvt->intMove[i] = 0;
      for (j = 0; j < sizeof(int); j++)
	{
	  if (intExt.b[i] == intLoc.b[j])
	    {
	      cvt->intMove[i] = j;
	      break;
	    }
	}
    }

  ListAdd (&CvtList, cvt);

  return (cvt);
}

/*--------------------------------------------------------------*/
/*   Detruit un objet Cvt.                                      */
/*--------------------------------------------------------------*/
void CvtDelete (Cvt cvt)
{
  if (cvt == NULL) return;

  free (cvt);

  CvtRecNumber--;
}

/*--------------------------------------------------------------*/
/*   Lit un short a une adresse.                                */
/*--------------------------------------------------------------*/
short CvtGetShort (Cvt cvt, unsigned char* address)
{
  static union
    {
      short s;
      unsigned char b[sizeof(short)];
    } loc;


  if (address == NULL) return (0);

  if (cvt == NULL)
    {
      memcpy (&loc.s, address, sizeof(short));
    }
  else
    {
      int i;

      loc.s = 0;

      for (i = 0; i < sizeof(short); i++)
	{
	  loc.b[cvt->shortMove[i]] = address[i];
	}
    }

  return (loc.s);
}

/*--------------------------------------------------------------*/
/*    Copie un short a une adresse.                             */
/*--------------------------------------------------------------*/
void CvtPutShort (short value, unsigned char* address)
{
  if (address == NULL) return;
  memcpy (address, &value, sizeof(short));
}

/*--------------------------------------------------------------*/
/*   Lit un entier a une adresse.                               */
/*--------------------------------------------------------------*/
int CvtGetInt (Cvt cvt, unsigned char* address)
{
  static union
    {
      int i;
      unsigned char b[sizeof(int)];
    } loc;


  if (address == NULL) return (0);

  if (cvt == NULL)
    {
      memcpy (&loc.i, address, sizeof(int));
    }
  else
    {
      int i;

      loc.i = 0;

      for (i = 0; i < sizeof(int); i++)
	{
	  loc.b[cvt->intMove[i]] = address[i];
	}
    }

  return (loc.i);
}

/*--------------------------------------------------------------*/
/*    Copie un entier a une adresse.                            */
/*--------------------------------------------------------------*/
void CvtPutInt (int value, unsigned char* address)
{
  if (address == NULL) return;
  memcpy (address, &value, sizeof(int));
}

/*--------------------------------------------------------------*/
/*    Copie un float  a une adresse.                            */
/*--------------------------------------------------------------*/
void CvtPutFloat (float value, unsigned char* address)
{
  static XDR xdrs;
  float f = value;

  if (address == NULL) return;
  xdrmem_create (&xdrs, (char*) address, sizeof(float), XDR_ENCODE);
  xdr_float (&xdrs, &f);
}

/*--------------------------------------------------------------*/
/*    Lit un float dans une adresse.                            */
/*--------------------------------------------------------------*/
float CvtGetFloat (unsigned char* address)
{
  static XDR xdrs;
  float f = (float) 0.0;

  if (address == NULL) return (f);
  xdrmem_create (&xdrs, (char*) address, sizeof(float), XDR_DECODE);
  xdr_float (&xdrs, &f);

  return (f);
}

/*--------------------------------------------------------------*/
/*    Copie un tableau de floats a une adresse.                 */
/*--------------------------------------------------------------*/
void CvtPutFloats (unsigned char* address, int elements)
{
  static XDR xdrs;
  float* ptr = (float*) address;
  float f;
  int i;

  if (address == NULL) return;

  for (i = 0; i < elements; i++)
    {
      xdrmem_create (&xdrs, (char*) ptr, sizeof(float), XDR_ENCODE);
      f = *ptr;
      xdr_float (&xdrs, &f);
      ptr++;
    }
}

/*--------------------------------------------------------------*/
/*    Lit un tableau de floats dans une adresse.                */
/*--------------------------------------------------------------*/
void CvtGetFloats (unsigned char* address, int elements)
{
  static XDR xdrs;
  float* ptr = (float*) address;
  float f;
  int i;

  if (address == NULL) return;

  for (i = 0; i < elements; i++)
    {
      xdrmem_create (&xdrs, (char*) ptr, sizeof(float), XDR_DECODE);
      xdr_float (&xdrs, &f);
      *ptr = f;
      ptr++;
    }
}

/*--------------------------------------------------------------*/
/*    Copie un double  a une adresse.                           */
/*--------------------------------------------------------------*/
void CvtPutDouble (double value, unsigned char* address)
{
  static XDR xdrs;
  double d = value;

  if (address == NULL) return;
  xdrmem_create (&xdrs, (char*) address, sizeof(double), XDR_ENCODE);
  xdr_double (&xdrs, &d);
}

/*--------------------------------------------------------------*/
/*    Lit un double dans une adresse.                           */
/*--------------------------------------------------------------*/
double CvtGetDouble (unsigned char* address)
{
  static XDR xdrs;
  double d = 0;

  if (address == NULL) return (0);
  xdrmem_create (&xdrs,(char*)  address, sizeof(double), XDR_DECODE);
  xdr_double (&xdrs, &d);

  return (d);
}

/*--------------------------------------------------------------*/
/*    Copie un tableau de doubles a une adresse.                */
/*--------------------------------------------------------------*/
void CvtPutDoubles (unsigned char* address, int elements)
{
  static XDR xdrs;
  double* ptr = (double*) address;
  double d;
  int i;

  if (address == NULL) return;

  for (i = 0; i < elements; i++)
    {
      xdrmem_create (&xdrs,(char*)  ptr, sizeof(double), XDR_ENCODE);
      d = *ptr;
      xdr_double (&xdrs, &d);
      ptr++;
    }
}

/*--------------------------------------------------------------*/
/*    Lit un tableau de doubles dans une adresse.               */
/*--------------------------------------------------------------*/
void CvtGetDoubles (unsigned char* address, int elements)
{
  static XDR xdrs;
  double* ptr = (double*) address;
  double d;
  int i;

  if (address == NULL) return;

  for (i = 0; i < elements; i++)
    {
      xdrmem_create (&xdrs, (char*) ptr, sizeof(double), XDR_DECODE);
      xdr_double (&xdrs, &d);
      *ptr = d;
      ptr++;
    }
}

/*--------------------------------------------------------------*/
/*    Regenere le texte d'origine.                              */
/*--------------------------------------------------------------*/
char* CvtToText (Cvt cvt)
{
  if (cvt == NULL) return (0);

  return ((char*) cvt->swap);
}

/*--------------------------------------------------------------*/
/*    Produit une reference de swap pour short.                 */
/*--------------------------------------------------------------*/
char* CvtGetSwap ()
{
  static int done = 0;
  static CvtSwap swap;
  static unsigned char shortByte[sizeof(short)];
  static unsigned char intByte[sizeof(int)];

  int i;
  int j;

  if (!done)
    {
      CvtPutShort (0x0102, shortByte);
      for (i = 0; i < sizeof(short); i++)
	{
	  sprintf (&swap[i*2], "%02.2x", shortByte[i]);
	}
      
      j = 2*sizeof(short);
      swap[j] = '-';
      j++;
	
      CvtPutInt (0x01020304, intByte);
      for (i = 0; i < sizeof(int); i++)
	{
	  sprintf (&swap[j + (i*2)], "%02.2x", intByte[i]);
	}
    }
  return ((char*) swap);
}





