# This file is part of PyCm, a Python interface to the Cm library
#
# Copyright (C) 2020-2021  European Gravitational Observatory
#
# Author list: Franco Carbognani: franco.carbognani@ego-gw.it
#                   Rhys Poulton: poulton@ego-gw.it
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

import pytest
import PyCm as cm
import multiprocessing as mp
import time




q = mp.Queue(100)

_CM_TYPE = {  # CmMessageGetItemType -> appropriate getter and converter
    cm.CmMessageItemTail: (None, None),
    cm.CmMessageItemChar: (cm.CmMessageGetChar, bytes),
    cm.CmMessageItemShort: (cm.CmMessageGetShort, int),
    cm.CmMessageItemInt: (cm.CmMessageGetInt, int),
    cm.CmMessageItemLong: (cm.CmMessageGetLong, int),
    cm.CmMessageItemFloat: (cm.CmMessageGetFloat, float),
    cm.CmMessageItemDouble: (cm.CmMessageGetDouble, float),
    cm.CmMessageItemText: (cm.CmMessageGetText, bytes),
    cm.CmMessageItemBytes: (cm.CmMessageGetBytes, NotImplemented),
    cm.CmMessageItemArray: (cm.CmMessageGetArray, NotImplemented),
}


def parse_cm_message(message):
    """Extract arguments from cm-message and return as list of Python objects"""
    items = []
    while True:
        getter, converter = _CM_TYPE[cm.CmMessageGetItemType(message)]
        if getter is None:
            break
        item = getter(message)  # weird error with "cm send ... -i 1.23"
        item = converter(item)
        items.append(item)
    return items


class test(object):
    __test__ = False

    @cm.CmMessageHandler
    def handle_test(message, sender, _):
        #Handler to parse the cm message and put it in the queue
        message = parse_cm_message(message)
        q.put(message)
        return cm.CmMessageOk

    def reciever(self,cm_name):
        #Thread to recieve the cmtest
        ok = cm.CmMessageOpenServer(cm_name)
        if not ok:
            raise SystemExit("Could not start with cm name %s" %cm_name)
        cm.CmMessageInstallHandler(self.handle_test, b'test') 
        while True:
            cm.CmMessageWaitWithTimeout(0.01)

def pytest_configure():
    return {"recieve_cm_name":None,
            "recieve_process": None,
            "send_cm_name": None     
    }

def test_install_handler():
    cm.CmMessageInstallHandler(test.handle_test, b'cm_type')

def test_start_recieve_server_process():
    pytest.recieve_cm_name = b"test_cm_receiver"
    pytest.send_cm_name = b"test_cm_sender"

    pytest.recieve_process = mp.Process(target=test().reciever,args=(pytest.recieve_cm_name,))
    pytest.recieve_process.daemon = True
    pytest.recieve_process.start()
    
    #wait 1 second so the recive server can start    
    time.sleep(1)

def test_start_send_server():
    ok = cm.CmMessageOpenServer(pytest.send_cm_name)
    if not ok:
        raise SystemExit("Could not start with cm name %s" %send_cm_name)

def test_send_recieve_char():

    #Check the recieve process is still alive
    assert pytest.recieve_process.is_alive()==True, "The recieve process died, cannot continue with tests"

    #Try sending an char
    data = b"t"
    message = cm.CmMessageNew()
    assert message, 'could not create CmMessage'
    cm.CmMessageSetType(message, b"test")
    cm.CmMessagePutText(message, data)
    res = cm.CmMessageSend(message, pytest.recieve_cm_name)
    if not res:
        raise SystemExit("Could not send char 't'")
    recieved_message = q.get(timeout=5)
    assert data==recieved_message[0]
    cm.CmMessageDelete(message)

def test_send_recieve_short():

    #Check the recieve process is still alive
    assert pytest.recieve_process.is_alive()==True, "The recieve process died, cannot continue with tests"

    #Try sending an short
    data = 32767
    message = cm.CmMessageNew()
    assert message, 'could not create CmMessage'
    cm.CmMessageSetType(message, b"test")
    cm.CmMessagePutShort(message, data)
    res = cm.CmMessageSend(message, pytest.recieve_cm_name)
    if not res:
        raise SystemExit("Could not send short '32767'")
    recieved_message = q.get(timeout=5)
    assert data==recieved_message[0]
    cm.CmMessageDelete(message)

def test_send_recieve_int():

    #Check the recieve process is still alive
    assert pytest.recieve_process.is_alive()==True, "The recieve process died, cannot continue with tests"
        
    #Try sending an int
    data = 2147483647
    message = cm.CmMessageNew()
    assert message, 'could not create CmMessage'
    cm.CmMessageSetType(message, b"test")
    cm.CmMessagePutInt(message, data)
    res = cm.CmMessageSend(message, pytest.recieve_cm_name)
    if not res:
        raise SystemExit("Could not send int '2147483647'")
    recieved_message = q.get(timeout=5)
    assert data==recieved_message[0]
    cm.CmMessageDelete(message)

def test_send_recieve_long():

    #Check the recieve process is still alive
    assert pytest.recieve_process.is_alive()==True, "The recieve process died, cannot continue with tests"

    #Try sending an long
    data = 9223372036854775807
    message = cm.CmMessageNew()
    assert message, 'could not create CmMessage'
    cm.CmMessageSetType(message, b"test")
    cm.CmMessagePutLong(message, data)
    res = cm.CmMessageSend(message, pytest.recieve_cm_name)
    if not res:
        raise SystemExit("Could not send long '9223372036854775807'")
    recieved_message = q.get(timeout=5)
    assert data==recieved_message[0]
    cm.CmMessageDelete(message)

def test_send_recieve_float():

    #Check the recieve process is still alive
    assert pytest.recieve_process.is_alive()==True, "The recieve process died, cannot continue with tests"

    #Try sending a float
    data = 1.5
    message = cm.CmMessageNew()
    assert message, 'could not create CmMessage'
    cm.CmMessageSetType(message, b"test")
    cm.CmMessagePutFloat(message, data)
    res = cm.CmMessageSend(message, pytest.recieve_cm_name)
    if not res:
        raise SystemExit("Could not send float '1.5'")
    recieved_message = q.get(timeout=5)
    assert data==recieved_message[0]
    cm.CmMessageDelete(message)

def test_send_recieve_double():

    #Check the recieve process is still alive
    assert pytest.recieve_process.is_alive()==True, "The recieve process died, cannot continue with tests"

    #Try sending a double
    data = 1e40
    message = cm.CmMessageNew()
    assert message, 'could not create CmMessage'
    cm.CmMessageSetType(message, b"test")
    cm.CmMessagePutDouble(message, data)
    res = cm.CmMessageSend(message, pytest.recieve_cm_name)
    if not res:
        raise SystemExit("Could not send double '1e40'")
    recieved_message = q.get(timeout=5)
    assert data==recieved_message[0]
    cm.CmMessageDelete(message)
 
def test_send_recieve_string():

    #Check the recieve process is still alive
    assert pytest.recieve_process.is_alive()==True, "The recieve process died, cannot continue with tests"
 
    #Try sending an string
    data = b"test"
    message = cm.CmMessageNew()
    assert message, 'could not create CmMessage'
    cm.CmMessageSetType(message, b"test")
    cm.CmMessagePutText(message, data)
    res = cm.CmMessageSend(message, pytest.recieve_cm_name)
    if not res:
        raise SystemExit("Could not send string 'test'")
    recieved_message = q.get(timeout=5)
    assert data==recieved_message[0]
    cm.CmMessageDelete(message)
    

def test_send_recieve_sms():

    #Check the recieve process is still alive
    assert pytest.recieve_process.is_alive()==True, "The recieve process died, cannot continue with tests"

    #Try sending an string
    data = b"test"
    message = cm.CmMessageNew()
    assert message, 'could not create CmMessage'
    cm.CmMessagePutText(message, data)
    cm.CmMessagePutText(message, data)
    cm.CmMessagePutInt(message,1)
    cm.CmMessageSetType(message, b'FbGetAllSmsData')
    res = cm.CmMessageSend(message, pytest.recieve_cm_name)
    if not res:
        raise SystemExit("Failed sending message to get all sms data")

    #Wait for the recieve of the message
    time.sleep(5)
    assert pytest.recieve_process.is_alive()==True, "The receive of cm message type FbGetAllSmsData failed"
    cm.CmMessageDelete(message)
