/*---------------------------------------------------------------------------*/
/*        Copyright (c) 1996 LAL Orsay, UPS-IN2P3-CNRS (France).             */
/*                                                                           */
/* Redistribution and use in source and binary forms, with or without        */
/* modification, are permitted provided that the following conditions        */
/* are met:                                                                  */
/* 1. Redistributions of source code must retain the above copyright         */
/*    notice, this list of conditions and the following disclaimer.          */
/* 2. Redistributions in binary form must reproduce the above copyright      */
/*    notice, this list of conditions and the following disclaimer in the    */
/*    documentation and/or other materials provided with the distribution.   */
/* 3. All advertising materials mentioning features or use of this software  */
/*    must display the following acknowledgement:                            */
/*      This product includes software developed by the Computer Application */
/*      Development Group at LAL Orsay (Laboratoire de l'Accelerateur        */
/*      Linaire - UPS-IN2P3-CNRS).                                           */
/* 4. Neither the name of the Institute nor of the Laboratory may be used    */
/*    to endorse or promote products derived from this software without      */
/*    specific prior written permission.                                     */
/*                                                                           */
/* THIS SOFTWARE IS PROVIDED BY THE LAL AND CONTRIBUTORS ``AS IS'' AND       */
/* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE     */
/* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR        */
/* PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE LAL OR CONTRIBUTORS BE      */
/* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR       */
/* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF      */
/* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS  */
/* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN   */
/* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)   */
/* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF    */
/* THE POSSIBILITY OF SUCH DAMAGE.                                           */
/*---------------------------------------------------------------------------*/
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <CmMessage.h>
#include <CmVersion.h>
#include <Reg.h>
#include <Hash.h>
/*#include <time.h>*/
/*#include <sys/types.h>*/
#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/time.h>
#include <time.h> 

#define tostring(x) #x

#ifndef CLOCKS_PER_SEC
#define CLOCKS_PER_SEC 1000000
#endif

#include <signal.h>

#ifdef unix
#include <fcntl.h>
#endif

#define CFG_TIME_FMT        "%Y-%m-%d-%Hh%Mm%S-UTC"

/*- Local type definitions. ------------------------------------*/
typedef enum
{
  PortAvailable,
  PortInUse,
  PortBusy
} PortState;

typedef struct _HostRec* Host;
typedef struct _HostRec
{
  char* name;
  int first_port;
  int port_number;
  PortState* ports;
} HostRec;

typedef struct _ExtTextRec* ExtText;
typedef struct _ExtTextRec
{
  char* text;
  int allocated;
  int used;
} ExtTextRec;
/*--------------------------------------------------------------*/

/*- Global variable declarations. ------------------------------*/
static CmNameServerInfos NameServerInfos = NULL;

static int       CleanupPeriod     = 300;  /* in seconds */
/* static char      WakeupText[]      = "wakeup"; */
static int       ShouldRestart     = 0;
static int       Finished          = 0;
static HTable    Hosts             = NULL;
/* static char*     Domain            = NULL; */
static char      LogName[256];
static int       NSDebug           = 0;
static int       NeedTrace         = 0;
static int       UseTimer          = 0;

/*
  The thresholds for the log file.
*/
static int       LineCount         = 0;
static int       MaxLines          = 5000;
static int       MaxBytes          = 100000;

static FILE*     Err = NULL;

/*--------------------------------------------------------------*/

/*- Local function declarations. -------------------------------*/
static void DoCleanup ();

static int DoPrint (char* format, ...);
static int Printer (char* format, va_list args);

static CmMessageStatus NSGetAddressHandler (CmMessage message, char* sender);
static CmMessageStatus NSGetPortHandler (CmMessage message, char* sender);
static CmMessageStatus NSGetNewPortHandler (CmMessage message, char* sender);
static CmMessageStatus NSGetNamesHandler (CmMessage message, char* sender);
static CmMessageStatus NSGetConnectsHandler (CmMessage message, char* sender);
static CmMessageStatus NSGetCommentsHandler (CmMessage message, char* sender);
static CmMessageStatus NSGetStatisticsHandler (CmMessage message, 
                                               char* sender);
static CmMessageStatus NSReconnectHandler (CmMessage message, char* sender);
static CmMessageStatus NSGetPortsHandler (CmMessage message, char* sender);
static CmMessageStatus NSGetPeriodHandler (CmMessage message, char* sender);
static CmMessageStatus NSSetPeriodHandler (CmMessage message, char* sender);
static CmMessageStatus NSDoCleanupHandler (CmMessage message, char* sender);
static CmMessageStatus NSStopHandler (CmMessage message, char* sender);
static CmMessageStatus NSRestartHandler (CmMessage message, char* sender);
static CmMessageStatus NSForceFaultHandler (CmMessage message, char* sender);
static CmMessageStatus NSSetBlindHandler (CmMessage message, char* sender);
static CmMessageStatus NSGetVersionHandler (CmMessage message, char* sender);

static int SendVersion (char* dest);
static Host HostFind (char* name);
static Host HostNew (char* name);
static char* HostNamer (Host This);

static void PortInitialize ();
static int PortAllocate (char* host_name);
static int PortReallocate (char* host_name, int oldPort);
static void PortMark (char* host_name, int port);
static void PortRelease (char* host_name, int port);


static void ConnectionHandler (CmConnect connect);
static void DisconnectionHandler (CmConnect connect);

static char* ConnectBuildFileName (CmConnect connect);
static char* ConnectBuildBackupFileName (CmConnect connect);
static void ConnectSaveInfos (CmConnect connect);
static void ConnectKillInfos (CmConnect connect);
static void ConnectReadInfos (char* file_name);

static int SetupInfos ();
static void LogClose ();
static void LogReset ();
static void LogCheck ();

static void catcher (int signal_number);
static void push_timer (int seconds);
/* static int pipe_handler (); */
/* static void pipe_setup (); */
/* static int pipe_send (const char* buffer); */
/*--------------------------------------------------------------*/

#include "CmHistory.c"

/*--------------------------------------------------------------*/
/*    The NameNerver.                                           */
/*--------------------------------------------------------------*/

/*--------------------------------------------------------------*/
int main (int argc, char* argv[])
/*--------------------------------------------------------------*/
/* Arguments :                                                  */
/*                                                              */
/*   domain-name   : default is given by $CMDOMAIN              */
/*                                                              */
/*--------------------------------------------------------------*/
{
  int error = 0;

  NeedTrace = 1;
  memset( LogName, 0, sizeof LogName );
  LogReset ();
  LogCheck ();

  if (getenv ("CMNAMESERVERDEBUG") != NULL)
    {
      NSDebug = 1;
    }

  if (!NSDebug)
    {
      CmMessageInstallPrinter (Printer);
    }

  if (argc >= 2)
    {
      CmConnectSetDomain (argv[1]);
    }

  PortInitialize ();

  NameServerInfos = CmNameServerGetInfos ();
  if (NameServerInfos == NULL)
    {
      CmConnectExit (CmErrorBadEnvironment);
    }

  if (!CmMessageOpenServer (NameServerInfos->name))
    {
      CmConnectExit (CmErrorBadSocket);
    }
  DoPrint ("%s is running on %s\n",
           NameServerInfos->name,  NameServerInfos->host);

  CmConnectInstallConnectionHandler (ConnectionHandler);
  CmConnectInstallDisconnectionHandler (DisconnectionHandler);

  CmConnectInstallConnectionHandler (ConnectionHandler);

  CmMessageInstallHandler ((CmMessageHandler) NSGetAddressHandler,
                           "NSGetAddress");
  CmMessageInstallHandler ((CmMessageHandler) NSGetPortHandler,
                           "NSGetPort");
  CmMessageInstallHandler ((CmMessageHandler) NSGetNewPortHandler,
                           "NSGetNewPort");
  CmMessageInstallHandler ((CmMessageHandler) NSGetNamesHandler,
                           "NSGetNames");
  CmMessageInstallHandler ((CmMessageHandler) NSGetConnectsHandler,
                           "NSGetConnects");
  CmMessageInstallHandler ((CmMessageHandler) NSGetCommentsHandler,
                           "NSGetComments");
  CmMessageInstallHandler ((CmMessageHandler) NSGetStatisticsHandler,
                           "NSGetStatistics");
  CmMessageInstallHandler ((CmMessageHandler) NSReconnectHandler,
                           "NSReconnect");
  CmMessageInstallHandler ((CmMessageHandler) NSGetPortsHandler,
                           "NSGetPorts");
  CmMessageInstallHandler ((CmMessageHandler) NSGetPeriodHandler,
                           "NSGetPeriod");
  CmMessageInstallHandler ((CmMessageHandler) NSSetPeriodHandler,
                           "NSSetPeriod");
  CmMessageInstallHandler ((CmMessageHandler) NSDoCleanupHandler,
                           "NSDoCleanup");
  CmMessageInstallHandler ((CmMessageHandler) NSStopHandler,
                           "NSStop");
  CmMessageInstallHandler ((CmMessageHandler) NSRestartHandler,
                           "NSRestart");
  CmMessageInstallHandler ((CmMessageHandler) NSForceFaultHandler,
                           "NSForceFault");
  CmMessageInstallHandler ((CmMessageHandler) NSSetBlindHandler,
                           "NSSetBlind");
  CmMessageInstallHandler ((CmMessageHandler) NSGetVersionHandler,
                           "NSGetVersion");

  CmServerSetComment ("ServerAlive");

  SetupInfos ();

  if (getenv ("NAMESERVER_USE_TIMER") != NULL) UseTimer = 1;
  DoPrint("Main> UseTimer %d - period %d\n", UseTimer, CleanupPeriod );
	      
  if (UseTimer != 0)
    {
/*       pipe_setup (); */
      push_timer (CleanupPeriod);
    }

  NeedTrace = 0;

  while (!Finished)
    {
      CmConnectCondition condition;
      
      if (UseTimer == 0)
        {
          condition = CmMessageWaitWithTimeout (CleanupPeriod);
        }
      else
        {
          condition = CmMessageWait ();
	  if( condition == CmConnectErrorCondition && errno == EINTR ) {
	    condition = CmConnectTimeoutDetection;
	    push_timer (CleanupPeriod);
	  }
        }

/*       DoPrint("Loop> CmMessageWait rtn %d - UseTimer %d - period %d\n", */
/* 	      condition, UseTimer, CleanupPeriod ); */
      switch (condition)
	{
	case CmConnectNormalCompletion :
	  break;
	case CmConnectNoHandler :
          DoPrint ("No handler\n");
	  break;
	case CmConnectTimeoutDetection :
	  {
            DoCleanup ();
	  }
	  break;
	case CmConnectBreakDetection :
	  if (ShouldRestart)
            {
              error = CmErrorRecovered;
            }
	  break;
	case CmConnectErrorCondition :
	  error = CmErrorBadSocket;
	  Finished = 1;
	  break;
	}
    }

  CmConnectInstallDisconnectionHandler (NULL);
  CmMessageCleanup ();

  LogClose ();

  return (error);
}

/*----------------------------------------------------------------*/
static void DoCleanup ()
/*----------------------------------------------------------------*/
{
  static CmConnect me = NULL;
  static CmMessage cleanup = NULL;

  CmConnect c;
  CmConnectIterator it;

  history_push ("DoCleanup>");

  if (me == NULL)
    {
      me = CmConnectWhoAmI ();
    }
  
  if (cleanup == NULL)
    {
      cleanup = CmMessageNew ();
      CmMessageReset (cleanup);
      CmMessageSetType (cleanup, "NSWakeup");
    }

  DoPrint ("DoCleanup> period %d\n", CleanupPeriod);
  
  fflush (stdout);
  fflush (stderr);
  
  LogCheck ();
  
  it = CmConnectGetIterator ();
  while ((c = CmConnectGetNextAlive (it)))
    {
      if (!CmConnectIsLocal (c) && !CmConnectIsNameServer (c))
        {
          char* name = CmConnectGetName (c);

          if (name != NULL && strcmp (name, "Pipe_Master_Producer") )
	    {
	      if (!CmMessageSend (cleanup, name))
		{
		  DoPrint ("DoCleanup> killing connect %s\n", name);
                  
		  CmConnectKill (c);
		}
            }
        }
    }
  CmConnectIteratorDelete (it);
}

/*----------------------------------------------------------------*/
static int DoPrint (char* format, ...)
/*----------------------------------------------------------------*/
{
  int result;
  va_list args;
  char* history_line;

  while ((history_line = history_pop ()) != NULL)
    {
      DoPrint ("%s\n", history_line);
    }

  va_start (args, format);
  
  result = Printer (format, args);
      
  va_end (args);

  return (result);
}

/*--------------------------------------------------------------*/
static int Printer (char* format, va_list args)
/*--------------------------------------------------------------*/
{
  int result;
  /* char* time; */

  static int new_line = 1;

  if (new_line)
    {
      char strTime[50];
      time_t current;
  
      new_line = 0;
      /*--------------------------------------------------------*/
      time( &current );
      strftime( strTime, sizeof strTime, CFG_TIME_FMT,
		gmtime(&current));
      fprintf (stderr, "%s> ", strTime);
    }

  result = vfprintf (stderr, format, args);

  fflush (stderr);

  if (strchr (format, '\n') != NULL) 
    {
      new_line = 1;
      LineCount++;
      if (LineCount > MaxLines)
        {
          LogCheck ();
        }
    }

  return (result);
}

/*--------------------------------------------------------------*/
/*    Allocate a port number to a new connection.               */
/*--------------------------------------------------------------*/
static void ConnectionHandler (CmConnect connect)
{
  int port;
  char* name, txt[128];
  CmMessage answer;

  /*CmConnectInstallTimeOut (connect, 5, 0);*/
  name = CmConnectGetName (connect);
  if ( !name || !name[0] ) {
    CmConnectSnprint( connect, txt, sizeof txt );
    DoPrint("Cm>ConnectionHandler> NULL name - %s\n", txt );
    return;
  }

  history_push ("ConnectionHandler> connect %s", name);

  answer = CmMessageNew ();
  
  CmMessageSetType (answer, "NSPort");
  
  CmMessagePutText (answer, name);
  
  if (!strcmp (name, "<illegal>"))
    {
      CmMessagePutInt  (answer, 0);
      CmMessagePutText (answer, name);
      if (!CmMessageSend (answer, name))
        {
          DoPrint ("ConnectionHandler> Error sending answer to %s\n",
                   name);
        }

      CmMessageDelete (answer);

      return;
    }

  if (CmConnectIsMultiple (connect))
    {
      CmConnect c;
      CmConnectIterator it;
      char* new_name;
      int last_number;
      int number;
      char base_name[256];
      char format[256];

      /*
	 On cherche toutes les connxions dont le nom est :

	           <name>_%d

	 On en deduit le numero max et le min et on change le nom de
	 la connexion nouvelle en

                   <name>_<max+1>

      */

      {
        char* underscore;

        strcpy (base_name, name);
        underscore = strrchr (base_name, '_');
        if (underscore != 0)
          {
            int n;
            if (sscanf (underscore+1, "%d", &n) == 1)
              {
                DoPrint ("ConnectionHandler> Changing the number of a "
                         "multiple connect %s\n", name);
                *underscore = 0;
              }
          }
      }
        
      last_number = 0;

      sprintf (format, "%s_", base_name);
      strcat (format, "%d");

      it = CmConnectGetIterator ();
      while ((c = CmConnectGetNextAlive (it)))
        {
          char* connect_name;
          int connect_number;

          if (!CmConnectIsLocal (c))
            {
              connect_name = CmConnectGetName (c);
              if (connect_name != NULL)
                {
                  if (sscanf (connect_name, format, &connect_number) == 1)
                    {
                      if (connect_number > last_number)
                        {
                          last_number = connect_number;
                        }
                    }
                }
	    }
        }
      CmConnectIteratorDelete (it);

      number = last_number + 1;

      ConnectKillInfos (connect);

      new_name = (char*) malloc (strlen (base_name) + 40);
      sprintf (new_name, "%s_%d", base_name, number);
      CmConnectSetName (connect, new_name);
      free (new_name);
    }

  name = CmConnectGetName (connect);
  if ( !name || !name[0] ) {
    CmConnectSnprint( connect, txt, sizeof txt );
    name = "???";
    DoPrint("Cm>ConnectionHandler> NULL name - %s - set to %s\n", txt, name );
  }

  if (!(port = CmConnectGetPort (connect)))
    {
      char* host = CmConnectGetHost (connect);

      if (host != NULL)
        {
          port = PortAllocate (host);
          CmConnectSetPort (connect, port);
        }
      else
        {
          DoPrint ("ConnectionHandler> NULL host name for connect %s\n", 
                   name);
        }

      CmMessagePutInt  (answer, port);
      CmMessagePutText (answer, name);

      if (!CmMessageSend (answer, name))
        {
          DoPrint ("ConnectionHandler> Error sending answer to %s\n",
                   name);
        }
    }
  else
    {

      PortMark (CmConnectGetHost (connect), port);
    }
      
  CmMessageDelete (answer);

  if (NeedTrace)
    {
      DoPrint ("Installing %s %d %s %s (name,port,host,owner) - socket_id %d\n", 
               name, port,
               CmConnectGetHost (connect), 
               CmConnectGetOwner (connect),
	       CmConnectGetSocketId (connect) );
    }

  ConnectSaveInfos (connect);
}

/*--------------------------------------------------------------*/
/*    Release a connection (and its port).                      */
/*--------------------------------------------------------------*/
static void DisconnectionHandler (CmConnect connect)
{
  int port;
  char* name;
  char txt[128];
  static int cmnserv_shutdown = 0;

  name =  CmConnectGetName (connect);
  if ( !name || !name[0] ) {
    CmConnectSnprint( connect, txt, sizeof txt );
    DoPrint("Cm>DisconnectionHandler> NULL name - %s\n", txt );
    return;
  }

  
  history_push ("DisconnectionHandler> connect %s", name);

  port = CmConnectGetPort (connect);
  PortRelease (CmConnectGetHost (connect), port);
  if (strcmp (name, "<illegal>") != 0)
  {
      /* fcarbogn 07/04/03, changed != to == (SPR#4) */
      /* fcarbogn, jwd modifications related to SPR#21 */
      if(!cmnserv_shutdown)
      {  
      	if (strcmp (name, "CmNameServer") != 0) /* if name do not match "CmNameServer" */
      	{
	      ConnectKillInfos (connect);
      	} else { 				/* if name do match "CmNameServer" */
	  cmnserv_shutdown = 1;
	}
      }
	      
 
      /*
      CmConnect server = CmConnectWhoAmI ();
      CmConnect new_connect;

      host = CmConnectGetHost (connect);
      owner = CmConnectGetOwner (connect);

      new_connect = CmConnectNewWithAddress (server, name, port, host);
      if (new_connect != NULL)
        {
          CmConnectSetOwner (new_connect, owner);
          
          DoPrint ("Re-installing %s host %s port %d user %s\n", 
                   name, host, port, owner);

          ConnectSaveInfos (new_connect);
        }
      else
        {
          if (NeedTrace)
            {
              DoPrint ("Uninstalling %s host %s port %d user %s\n", 
                       name, host, port, owner);
            }

        }
      */
    }
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSGetAddressHandler (CmMessage message, char* sender)
/*--------------------------------------------------------------*/
/*                                                              */
/*  Expected received syntax :                                  */
/*                                                              */
/*    char* name              Connection name                   */
/*   [int   transaction       Transaction id for the sender]    */
/*                                                              */
/*  Answer syntax : type = NSAddress                            */
/*                                                              */
/*    char* name              Connection name                   */
/*                                                              */
/*    char* host              host name of the Connect          */
/*    int port                port number of the Connect        */
/*    bool is_multiple                                          */
/*    char* owner             owner of the Connect              */
/*                                                              */
/*   [int   transaction       Transaction id for the sender]    */
/*                                                              */
/*--------------------------------------------------------------*/
{
  char* name = NULL;
  int number = 0;
  CmConnect connect;
  int port = 0;
  CmMessage answer;
  int transaction = 0;

  name   = CmMessageGetText (message);

  history_push ("NSGetAddressHandler> name %s", name);

    /*
      Ceci est une reminiscence (obsolete et inutilisee) de v7r4
      */
  CmMessageGetInt (message);

  if (CmMessageGetItemType (message) == CmMessageItemInt)
    {
      transaction = CmMessageGetInt (message);
    }

  answer = CmMessageNew ();
  CmMessageSetType (answer, "NSAddress");
  
  connect = CmConnectGetNthReference (name, number);
  CmMessagePutText (answer, name);
  if (connect)  {
   port = CmConnectGetPort (connect);
  }
 

  if (port)
    {
      CmMessagePutText (answer, CmConnectGetHost (connect));
      CmMessagePutInt  (answer, CmConnectGetPort (connect));
      if (CmConnectIsMultiple (connect)) CmMessagePutInt (answer, 1);
      else CmMessagePutInt (answer, 0);
      CmMessagePutText (answer, CmConnectGetOwner (connect));
    }

  if (transaction != 0)
    {
      CmMessagePutInt (answer, transaction);
    }

  if (!CmMessageSend (answer, sender))
    {
      DoPrint ("NSGetAddressHandler> Error sending answer to %s\n",
               sender);
    }

  CmMessageDelete (answer);

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSGetPortHandler (CmMessage message, char* sender)
/*--------------------------------------------------------------*/
{
  char* name;
  CmConnect connect;
  CmMessage answer;

  name   = CmMessageGetText (message);

  history_push ("NSGetPortHandler> name %s", name);

  answer = CmMessageNew ();
  CmMessageSetType (answer, "NSPort");
  
  connect = CmMessageGetConnect (message);
  if (connect != NULL)
    {
      CmMessagePutText (answer, name);
      CmMessagePutInt (answer, CmConnectGetPort (connect));
      name = CmConnectGetName (connect);
      if (name == NULL) name = "???";
      CmMessagePutText (answer, name);
    }
  else
    {
      CmMessagePutText (answer, name);
      CmMessagePutInt (answer, 0);
      CmMessagePutText (answer, "");
    }

  if (!CmMessageSend (answer, sender))
    {
      DoPrint ("NSGetPortHandler> Error sending answer to %s\n",
               sender);
    }

  CmMessageDelete (answer);

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSGetNewPortHandler (CmMessage message, char* sender)
/*--------------------------------------------------------------*/
/*                                                              */
/*  Expected received syntax :                                  */
/*                                                              */
/*    char* name              Connection name                   */
/*   [int   transaction       Transaction id for the sender]    */
/*                                                              */
/*  Answer syntax : type = NSNewPort                            */
/*                                                              */
/*    char* name              Connection name                   */
/*                                                              */
/*    int new_port            new port number                   */
/*   [int   transaction       Transaction id for the sender]    */
/*                                                              */
/*--------------------------------------------------------------*/
{
  char* name;
  CmConnect connect = NULL;
  int old_port = 0;
  int new_port = 0;
  char* host = NULL;
  CmMessage answer;
  int transaction = 0;

  name   = CmMessageGetText (message);
  if (name == NULL) name = "???";

  history_push ("NSGetNewPortHandler> name %s", name);

  if (CmMessageGetItemType (message) == CmMessageItemInt)
    {
      transaction = CmMessageGetInt (message);
    }

  answer = CmMessageNew ();
  CmMessageSetType (answer, "NSNewPort");
  
  connect = CmConnectGetReference (name);
  CmMessagePutText (answer, name);

  if (connect != NULL)
    {
      old_port = CmConnectGetPort (connect);
      host = CmConnectGetHost (connect);
    }

  if (host == NULL)
    {
      DoPrint ("NSGetNewPortHandler> NULL host name for connect %s\n", 
               name);
    }
  else
    {
      new_port = PortReallocate (host, old_port);
      CmConnectSetPort (connect, new_port);
    }
      
  CmMessagePutInt (answer, new_port);

  if (transaction != 0)
    {
      CmMessagePutInt (answer, transaction);
    }

  if (!CmMessageSend (answer, sender))
    {
      DoPrint ("NSGetNewPortHandler> Error sending answer to %s\n",
               sender);
    }

  CmMessageDelete (answer);

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSGetNamesHandler (CmMessage message, char* sender)
/*--------------------------------------------------------------*/
{
  char* pattern = NULL;
  CmConnect connect;
  RegExp reg = NULL;
  CmConnectIterator it;
  CmMessage answer;

  history_push ("NSGetNamesHandler>");

  if (CmMessageGetItemType (message) == CmMessageItemText)
    pattern = CmMessageGetText (message);
  
  answer = CmMessageNew ();
  CmMessageSetType (answer, "NSNames");
  
  if (pattern) reg = RegExpNew (pattern);

  it = CmConnectGetIterator ();
  while ((connect = CmConnectGetNextAlive (it)))
    {
      if (!CmConnectIsLocal (connect))
        {
          char* name = CmConnectGetName (connect);
          if (name != NULL)
            {
              if (strcmp (name, "Pipe_Master_Producer"))
                {
                  if (!reg || RegExpMatch (reg, CmConnectGetName (connect)))
                    CmMessagePutText (answer, CmConnectGetName (connect));
                }
            }
        }
    }
  CmConnectIteratorDelete (it);
  RegExpDelete (reg);
  
  if (!CmMessageSend (answer, sender))
    {
      DoPrint ("NSGetNamesHandler> Error sending answer to %s\n",
               sender);
    }

  CmMessageDelete (answer);

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSGetConnectsHandler (CmMessage message, char* sender)
/*--------------------------------------------------------------*/
{
  CmConnect connect;
  CmConnectIterator it;
  CmMessage answer;

  history_push ("NSGetConnectsHandler>");

  answer = CmMessageNew ();
  CmMessageSetType (answer, "NSConnects");
  
  it = CmConnectGetIterator ();
  while ((connect = CmConnectGetNextAlive (it)))
    {
      if (!CmConnectIsLocal (connect))
        {
          char* name = CmConnectGetName (connect);
          if (name != NULL)
            {
              if (strcmp (name, "Pipe_Master_Producer"))
                {
                  CmMessagePutText (answer, CmConnectGetName (connect));
                  CmMessagePutText (answer, CmConnectGetHost (connect));
                  CmMessagePutInt (answer, CmConnectGetPort (connect));
                  CmMessagePutText (answer, CmConnectGetOwner (connect));
                }
            }
        }
    }
  CmConnectIteratorDelete (it);
  
  if (!CmMessageSend (answer, sender))
    {
      DoPrint ("NSGetConnectsHandler> Error sending answer to %s\n",
               sender);
    }

  CmMessageDelete (answer);

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSGetCommentsHandler (CmMessage message, char* sender)
/*--------------------------------------------------------------*/
{
  CmConnect connect;
  CmConnectIterator it;
  CmMessage answer;

  history_push ("NSGetCommentsHandler>");

  answer = CmMessageNew ();
  CmMessageSetType (answer, "NSComments");
  
  it = CmConnectGetIterator ();
  while ((connect = CmConnectGetNextAlive (it)))
    {
      if (!CmConnectIsLocal (connect))
        {
          char* name = CmConnectGetName (connect);
          if (name != NULL)
            {
              CmMessagePutText (answer, name);
              CmMessagePutText (answer, CmConnectGetComment (connect));
            }
        }
    }
  CmConnectIteratorDelete (it);
  
  if (!CmMessageSend (answer, sender))
    {
      DoPrint ("NSGetCommentsHandler> Error sending answer to %s\n",
               sender);
    }

  CmMessageDelete (answer);

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSGetStatisticsHandler (CmMessage message, 
                                               char* sender)
/*--------------------------------------------------------------*/
{
  CmConnect connect;
  CmConnectIterator it;
  CmMessage answer;
  CmConnectStatisticsRec stat;

  history_push ("NSGetStatisticsHandler>");

  answer = CmMessageNew ();
  CmMessageSetType (answer, "NSComments");
  
  it = CmConnectGetIterator ();
  while ((connect = CmConnectGetNextAlive (it)))
    {
      if (!CmConnectIsLocal (connect))
        {
          char* name = CmConnectGetName (connect);
          if (name != NULL)
            {
              CmMessagePutText (answer, name);
              CmConnectGetStatistics (connect, &stat);
              CmMessagePutInt (answer, stat.sent);
              CmMessagePutInt (answer, stat.bytes_sent);
              CmMessagePutInt (answer, stat.received);
              CmMessagePutInt (answer, stat.bytes_received);
            }
        }
    }
  CmConnectIteratorDelete (it);
  
  if (!CmMessageSend (answer, sender))
    {
      DoPrint ("NSGetStatisticsHandler> Error sending answer to %s\n",
               sender);
    }

  CmMessageDelete (answer);

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSReconnectHandler (CmMessage message, 
                                           char* sender)
/*--------------------------------------------------------------*/
{
  char* name = NULL;
  int port = 0;
  char* host = NULL;
  char* owner = NULL;
  char txt[128]; 

  name = CmMessageGetText (message);
  if ( !name || !name[0]) {
    CmConnectSnprint( CmMessageGetConnect(message), txt, sizeof txt );
    DoPrint("Cm>NSReconnectionHandler> NULL name - %s\n", txt );
    return(CmMessageOk);
  }


  history_push ("NSReconnectHandler> name %s", name);

  if (CmConnectGetReference (name) == NULL)
    {
      port = CmMessageGetInt (message);
      host = CmMessageGetText (message);
      owner = CmMessageGetText (message);

      if (port != 0)
        {
          CmConnect server;
          CmConnect connect;
          
          server = CmConnectWhoAmI ();
          
          connect = CmConnectNewWithAddress (server, name, port, host);
          if (connect != NULL)
            {
              CmConnectSetOwner (connect, owner);
              
              ConnectSaveInfos (connect);
              DoPrint ("NSReconnectHandler> Application %s re-installed\n", 
                       name);
            }
          else
            {
              DoPrint ("NSReconnectHandler> Application %s not found\n",
                       name);
            }
        }
    }
  else
    {
      DoPrint ("NSReconnectHandler> Application %s already there\n",
               name);
    }

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSGetPortsHandler (CmMessage message, char* sender)
/*--------------------------------------------------------------*/
{
  HIterator it;
  Host host;
  int port;
  CmMessage answer;
  
  history_push ("NSGetPortsHandler>");

  answer = CmMessageNew ();
  CmMessageSetType (answer, "NSPorts");
  
  it = HIteratorNew (Hosts);
  while ((host = (Host) HIteratorNext(it)) != NULL)
    {
      CmMessagePutText (answer, host->name);
      for (port = 0; port < host->port_number; port++)
        {
          if (host->ports[port] != PortAvailable)
            {
              CmMessagePutInt (answer, port + host->first_port);
              CmMessagePutInt (answer, host->ports[port]);
            }
        }
    }
  HIteratorDelete (it);
  
  if (!CmMessageSend (answer, sender))
    {
      DoPrint ("NSGetPortsHandler> Error sending answer to %s\n",
               sender);
    }

  CmMessageDelete (answer);

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSGetPeriodHandler (CmMessage message, char* sender)
/*--------------------------------------------------------------*/
{
  CmMessage answer;
  
  history_push ("NSGetPeriodHandler>");

  answer = CmMessageNew ();
  CmMessageSetType (answer, "NSPeriod");
  
  CmMessagePutInt (answer, CleanupPeriod);
  
  if (!CmMessageSend (answer, sender))
    {
      DoPrint ("NSGetPeriodHandler> Error sending answer to %s\n",
               sender);
    }

  CmMessageDelete (answer);

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSSetPeriodHandler (CmMessage message, char* sender)
/*--------------------------------------------------------------*/
{
  int period = 0;
  CmMessage answer;

  history_push ("NSSetPeriodHandler>");

  period = CmMessageGetInt (message);
  
  if (period > 0) { 
    CleanupPeriod = period;
    DoPrint("NSSetPeriodHandler> CleanupPeriod set to %d\n",  CleanupPeriod );
  }

  answer = CmMessageNew ();
  CmMessageSetType (answer, "NSPeriod");
  
  CmMessagePutInt (answer, CleanupPeriod);
  
  if (!CmMessageSend (answer, sender))
    {
      DoPrint ("NSSetPeriodHandler> Error sending answer to %s\n",
               sender);
    }

  CmMessageDelete (answer);
  if( UseTimer )  {
    push_timer (CleanupPeriod);
  }
  return (CmMessageBreak);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSDoCleanupHandler (CmMessage message, char* sender)
/*--------------------------------------------------------------*/
{
  char* name;
  
  CmConnect connect;
  RegExp reg;
  int killed;
  CmMessage answer;

  history_push ("NSDoCleanupHandler>");

  name = CmMessageGetText (message);
  
  CmConnectWhoAmI ();
  
  answer = CmMessageNew ();
  
  killed = 0;
  reg = RegExpNew (name);
  if (reg)
    {
      CmConnectIterator it;
      
      CmMessageSetType (answer, "NSWakeup");
  
      it = CmConnectGetIterator ();
      while ((connect = CmConnectGetNextAlive (it)))
        {
          if (!CmConnectIsLocal (connect) && !CmConnectIsNameServer (connect) )
            {
              name = CmConnectGetName (connect);
              if (name != NULL && strcmp (name, "Pipe_Master_Producer") &&
		  RegExpMatch (reg, name) ) 
		{
		  if (!CmMessageSend (answer, name))
		    {
		      DoPrint ("DoCleanupHandler> The connection %s "
			       "is dead\n",
			       name);
		      CmConnectKill (connect);
		      killed = 1;
		    }
		}
            }
        }
      CmConnectIteratorDelete (it);
      RegExpDelete (reg);
    }
  
  CmMessageReset (answer);
  CmMessageSetType (answer, "NSCleanup");
  
  if (killed) CmMessagePutText (answer, " -> killed.");
  else CmMessagePutText (answer, " -> not found or still alive.");
  
  if (!CmMessageSend (answer, sender))
    {
      DoPrint ("NSDoCleanupHandler> Error sending answer to %s\n",
               sender);
    }

  CmMessageDelete (answer);

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSStopHandler (CmMessage message, char* sender)
/*--------------------------------------------------------------*/
{
  CmMessage answer;
  
  history_push ("NSStopHandler>");

  Finished = 1;
  
  answer = CmMessageNew ();
  CmMessageSetType (answer, "NSStopped");
  
  if (!CmMessageSend (answer, sender))
    {
      DoPrint ("NSStopHandler> Error sending answer to %s\n",
               sender);
    }

  CmMessageDelete (answer);
  
  return (CmMessageBreak);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSRestartHandler (CmMessage message, char* sender)
/*--------------------------------------------------------------*/
{
  CmMessage answer;
  
  history_push ("NSRestartHandler>");

  Finished = 1;
  ShouldRestart = 1;
  
  answer = CmMessageNew ();
  CmMessageSetType (answer, "NSStopped");
  
  if (!CmMessageSend (answer, sender))
    {
      DoPrint ("NSRestartHandler> Error sending answer to %s\n",
               sender);
    }

  CmMessageDelete (answer);
  
  return (CmMessageBreak);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSForceFaultHandler (CmMessage message, char* sender)
/*--------------------------------------------------------------*/
{
  CmMessage answer;
  
  history_push ("NSForceFaultHandler>");

  answer = CmMessageNew ();
  CmMessageSetType (answer, "NSFault");
  
  if (!CmMessageSend (answer, sender))
    {
      DoPrint ("NSForceFaultHandler> Error sending answer to %s\n",
               sender);
    }

  CmMessageDelete (answer);
  

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSSetBlindHandler (CmMessage message, char* sender)
/*--------------------------------------------------------------*/
{
  int seconds = CmMessageGetInt (message);

  CmMessage answer;
  
  history_push ("NSSetBlindHandler>");

  answer = CmMessageNew ();
  CmMessageSetType (answer, "NSBlind");
  
  if (!CmMessageSend (answer, sender))
    {
      DoPrint ("NSSetBlindHandler> Error sending answer to %s\n",
               sender);
    }

  CmMessageDelete (answer);
  
  DoPrint ("NSSetBlindHandler> starting blind... (%d seconds)\n", seconds);

  sleep (seconds);

  DoPrint ("NSSetBlindHandler> finished blind...\n");

  return (CmMessageOk);
}

/*--------------------------------------------------------------*/
static CmMessageStatus NSGetVersionHandler (CmMessage message, char* sender)
/*--------------------------------------------------------------*/
{
  int status ;
  history_push ("NSGetVersionHandler>");

  status = SendVersion (sender);

  if (!status)
    {
      DoPrint ("NSGetVersionHandler> Error sending answer to %s\n",
               sender);
    }

  return (CmMessageOk);
}

/*--------------------------------------------------------*/
static int SendVersion (char* dest)
/*--------------------------------------------------------*/
{
  CmMessage message = NULL;
  int status;

  message = CmMessageNew ();

  CmMessagePutText (message, CMVERSION);
  CmMessageSetType (message, "NSVersion");

  status = CmMessageSend (message, dest);

  CmMessageDelete (message);

  return (status);
}


/*--------------------------------------------------------------*/
char* full_host_name (char* name)
/*--------------------------------------------------------------*/
{
  static char full_name[256];

  struct hostent* hostentPtr;
  unsigned int a0, a1, a2, a3;
  int items;

  strcpy (full_name, name);

  items = sscanf (name, "%u.%u.%u.%u", &a0, &a1, &a2, &a3);
  if (items == 4)
    {
      return (full_name);
    }

  if ((hostentPtr = gethostbyname (full_name)) != NULL)
    {
      strcpy (full_name, (char*) hostentPtr->h_name);
    }

  return (full_name);
}

/*--------------------------------------------------------------*/
static Host HostFind (char* name)
/*--------------------------------------------------------------*/
{
  Host This;

  history_push ("HostFind> name %s", name);

  This = (Host) HTableFindReference (Hosts, name);

  return (This);
}

/*--------------------------------------------------------------*/
static Host HostNew (char* name)
/*--------------------------------------------------------------*/
{
  Host This;
  int port;
  char* full_name;

  history_push ("HostNew> name %s", name);

  if (name == NULL) return (NULL);

  full_name = full_host_name (name);

  This = HostFind (full_name);
  if (This) return (This);

  This = (Host) malloc (sizeof(HostRec));

  This->name = (char*) malloc (strlen (full_name) + 1);
  strcpy (This->name, full_name);

  This->first_port  = NameServerInfos->firstPort;
  This->port_number = NameServerInfos->ports;
  
  This->ports = (PortState*) malloc (This->port_number * sizeof (PortState));
  for (port = 0; port < This->port_number; port++)
    {
      This->ports[port] = PortAvailable;
    }

  HTableAddReference (Hosts, (HReference) This);
  
  return (This);
}

/*--------------------------------------------------------------*/
static char* HostNamer (Host This)
/*--------------------------------------------------------------*/
{
  static char null_string[1] = {0};

  history_push ("HostNamer> name %s", This->name);

  if (This == NULL) return (null_string);
  if (This->name == NULL) return (null_string);

  return (This->name);
}

/*--------------------------------------------------------------*/
/*   Initialize the port manager.                               */
/*--------------------------------------------------------------*/
static void PortInitialize ()
{

  history_push ("PortInitialize>");

  Hosts = HTableNew (50, (HNamer) HostNamer);
}

/*--------------------------------------------------------------*/
/*   Find a free port number.                                   */
/*--------------------------------------------------------------*/
static int PortAllocate (char* host_name)
{
  Host host;
  int port;
  static Host my_host = NULL;
  int has_busy = 0;

  history_push ("PortAllocate> host_name %s", host_name);

  if (my_host == NULL)
    {
      CmConnect me = CmConnectWhoAmI ();
      char* my_host_name = CmConnectGetHost (me);
      my_host = HostNew (my_host_name);
    }

  host = HostNew (host_name);
  if (host == NULL)
    {
      DoPrint ("Impossible to create a new Host.\n");
      return (0);
    }
  
  for (port = 0; port < host->port_number; port++)
    {
      if (host->ports[port] == PortAvailable)
	{
          if (host == my_host)
            {
              if (CmConnectTestPort (host->first_port + port))
                {
                  host->ports[port] = PortInUse;
                  return (host->first_port + port);
                }
              else
                {
                  host->ports[port] = PortBusy;
                  if (!has_busy)
                    {
                      has_busy = 1;

                      DoPrint ("Allocate> The port %d is unavailable "
                               "in host %s.\n",
                               host->first_port + port, host_name);
                    }
                }
            }
          else
            {
              host->ports[port] = PortInUse;
              return (host->first_port + port);
            }
	}
    }

  DoPrint ("No more free port number in host %s.\n", host_name);

/*
  Prepare for next attempt to get a free port number.

  All busy ports will be retried !
  */

  for (port = 0; port < host->port_number; port++)
    {
      if (host->ports[port] == PortBusy)
        {
          host->ports[port] = PortAvailable;
        }
    }

  return (0);
}

/*--------------------------------------------------------------*/
/*   Find another free port number (the old one is busy)        */
/*--------------------------------------------------------------*/
static int PortReallocate (char* host_name, int old_port)
{
  Host host;

  history_push ("PortReallocate> host_name %s port %d", host_name, old_port);

  host = HostNew (host_name);
  if (host == NULL)
    {
      DoPrint ("Impossible to create a new Host.\n");
      return (0);
    }
  
  old_port -= host->first_port;
  if ((old_port >= 0) && (old_port < host->port_number))
    {
      host->ports[old_port] = PortBusy;
      DoPrint ("Reallocate> The port %d is unavailable in host %s.\n",
               host->first_port + old_port, host_name);
    }

  return (PortAllocate (host_name));
}

/*--------------------------------------------------------------*/
/*   Mark a port number as used.                                */
/*--------------------------------------------------------------*/
static void PortMark (char* host_name, int port)
{
  Host host;

  history_push ("PortMark> host_name %s port %d", host_name, port);

  host = HostNew (host_name);
  if (host == NULL)
    {
      DoPrint ("Impossible to create a new Host.\n");
      return;
    }
  
  port -= host->first_port;
  if ((port < 0) || (port >= host->port_number)) return;

  host->ports[port] = PortInUse;
}

/*--------------------------------------------------------------*/
/*   Release a port number.                                     */
/*--------------------------------------------------------------*/
static void PortRelease (char* host_name, int port)
{
  Host host;

  history_push ("PortRelease> host_name %s port %d", host_name, port);

  host = HostNew (host_name);
  if (host == NULL)
    {
      DoPrint ("Impossible to create a new Host.\n");
      return;
    }
  
  port -= host->first_port;
  if ((port < 0) || (port >= host->port_number)) return;

  host->ports[port] = PortAvailable;
}




/*--------------------------------------------------------------*/
/*   Build the file name for a Connect.                         */
/*--------------------------------------------------------------*/
static char* ConnectBuildFileName (CmConnect connect)
{
  static char name[256];
  char* connect_name;

  name[0] = 0;

  connect_name = CmConnectGetName (connect);
  if (connect_name != NULL)
    {
      sprintf (name, "%s.dat", CmConnectGetName (connect));
    }

  return (name);
}

/*--------------------------------------------------------------*/
/*   Build the file name for a Connect.                         */
/*--------------------------------------------------------------*/
static char* ConnectBuildBackupFileName (CmConnect connect)
{
  static char name[256];
  char* connect_name;

  name[0] = 0;

  connect_name = CmConnectGetName (connect);
  if (connect_name != NULL)
    {
      sprintf (name, "%s.bck", CmConnectGetName (connect));
    }
  return (name);
}

/*---------------------------------------------------------------*/
/*   Save the infos for a connection.                            */
/*---------------------------------------------------------------*/
static void ConnectSaveInfos (CmConnect connect)
{
  FILE* file;
  char* file_name;

  if (!connect) return;
  if (connect == CmConnectWhoAmI()) return;

  file_name = ConnectBuildFileName (connect);
  if (file_name)
    {
      file = fopen (file_name, "w");
      if (file)
	{
	  fprintf (file, "port %d\n", CmConnectGetPort (connect));
	  fprintf (file, "host %s\n", CmConnectGetHost (connect));
	  fprintf (file, "owner %s\n", CmConnectGetOwner (connect));
	  fclose (file);
	}
      else
        {
          char* name = CmConnectGetName (connect);
          if (name == NULL) name = "???";

          DoPrint ("Impossible to save infos for %s\n", name);
        }
    }
}

/*---------------------------------------------------------------*/
/*   Kill the infos for a connection.                            */
/*---------------------------------------------------------------*/
static void ConnectKillInfos (CmConnect connect)
{
  char* name;

  if (!connect) return;
  if (connect == CmConnectWhoAmI()) return;

  name = ConnectBuildFileName (connect);
  if (name)
    {
      ConnectBuildBackupFileName (connect);
      unlink (name);
      /*rename (name, bck_name);*/

    }
}

/*---------------------------------------------------------------*/
/*   Read the infos from a file and create the correponding      */
/*  connection.                                                  */
/*---------------------------------------------------------------*/
static void ConnectReadInfos (char* file_name)
{
  FILE* file;
  static char name[256];
  static char host[256];
  static char owner[256];
  int port;
  CmConnect server;
  CmConnect connect;
  char* suffix;

  server = CmConnectWhoAmI ();
  
  file = fopen (file_name, "r");
  if (file)
    {
      port = 0;
      host[0] = 0;
      owner[0] = 0;

      fscanf (file, "port %d\n", &port);
      if (port != 0)
	{
	  fscanf (file, "host %s\n", host);
	  fscanf (file, "owner %s\n", owner);
	}
      fclose (file);

      if ((port != 0) &&
          (strlen (host) > 0))
	{
	  strcpy (name, file_name);
	  suffix = strstr (name, ".dat");
	  if (suffix) *suffix = 0;
	  
	  connect = CmConnectNewWithAddress (server, name, port, host);
          if (connect != NULL)
            {
              CmConnectSetOwner (connect, owner);

              ConnectSaveInfos (connect);
            }
          else
            {
              unlink (file_name);
            }
        }
      else
        {
          DoPrint ("Bad format for info file %s\n", file_name);
        }
    }
  else
    {
      DoPrint ("Impossible to read infos from %s\n", file_name);
    }

}

/*---------------------------------------------------------------*/
static int SetupInfos ()
/*---------------------------------------------------------------*/
{
  mode_t mask;
  char*  root;
  char*  domain;
  int    new_infos;
  char*  file_name;
  int    status;
  static char* format = "%s/%s";

  domain = CmConnectGetDomain ();
  if (domain == NULL)
    {
      DoPrint ("Domain is not defined\n");
      exit (CmErrorBadEnvironment);
    }
  
  root = NameServerInfos->repository;
  if (root == NULL)
    {
      DoPrint ("Domain repository is not defined\n");
      exit (CmErrorBadEnvironment);
    }
      
  file_name = (char*) malloc (strlen (format) +
                              strlen (root) +
                              strlen (domain) + 1);
  
  mask = S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH;

  strcpy (file_name, root);

  status = mkdir (file_name, mask);
  if (status < 0)
    {
      if (errno != EEXIST)
	{
	  DoPrint ("Impossible to create the %s directory.\n",
		   file_name);
	  free (file_name);
          exit (CmErrorBadEnvironment);
	}
    }

  strcat (file_name, "/");
  strcat (file_name, domain);

  status = mkdir (file_name, mask);
  if (status < 0)
    {
      if (errno == EEXIST)
	{
	  new_infos = 0;
	}
      else
	{
	  DoPrint ("Impossible to create the %s directory.\n",
		   file_name);
	  free (file_name);
          exit (CmErrorBadEnvironment);
	}
    }
  else new_infos = 1;

  status = chdir (file_name);
  if (status < 0)
    {
      DoPrint ("Impossible to move to %s.\n",
	       file_name);
      free (file_name);
      return (0);
    }

  free (file_name);

  /*
     Then look for files *.dat into this directory/
     */

  if (new_infos == 0)
    {
      FILE* f;
      char* nl;
      static char line[256];

      f = popen ("ls *.dat", "r");
      if (f != NULL)
	{
	  while (!feof (f))
	    {
	      if (fgets (line, sizeof(line), f))
		{
		  nl = strchr (line, '\n');
		  if (nl) *nl = 0;

		  ConnectReadInfos (line);
		}
	    }
	  pclose (f);
	}
    }

  return (1);
}

/*---------------------------------------------------------------*/
static void LogClose ()
/*---------------------------------------------------------------*/
{
  if (NSDebug) return;

   if (Err != NULL) 
    {
      fclose (Err);
      Err = NULL;
    }
}

/*---------------------------------------------------------------*/
static void LogReset ()
/*---------------------------------------------------------------*/
{
  /* char temp[80]; */
  char strTime[50];
  time_t current;
  

  LineCount = 0;
  
  if (NSDebug) return;

  LogClose ();

  /* strcpy (temp, CmConnectTime ()); */
  
  /* for (;;) */
  /*   { */
  /*     char* colon; */

  /*     colon = strchr (temp, ':'); */
  /*     if (colon) *colon = '_'; */
  /*     else break; */
  /*   } */

  /* for (;;) */
  /*   { */
  /*     char* space; */

  /*     space = strchr (temp, ' '); */
  /*     if (space) *space = '_'; */
  /*     else break; */
  /*   } */

  /* temp[19] = 0; */
  /*--------------------------------------------------------*/
  time( &current );
  strftime( strTime, sizeof strTime, CFG_TIME_FMT,
	    gmtime(&current));
  

  sprintf (LogName, "NS_%s.log", strTime);

  if ((Err = freopen (LogName, "w", stderr)) == NULL)
    {
      DoPrint ("Impossible to redirect stderr to %s\n",
               LogName);
    }
}

/*---------------------------------------------------------------*/
static void LogCheck ()
/*---------------------------------------------------------------*/
{
  FILE* file;
  long bytes;
  
  if (NSDebug) 
    {
      LineCount = 0;
      return;
    }
  
  file = fopen (LogName, "r");
  if (file)
    {
      fseek (file, 0L, 2);
      bytes = ftell (file);
      fclose (file);

      if ((bytes > MaxBytes) ||
          (LineCount > MaxLines)) 
        {
          LogReset ();
        }
    }
}

/*---------------------------------------------------------------*/
static void catcher (int signal_number)
/*---------------------------------------------------------------*/
{
/*   pipe_send ("cleanup"); */
  push_timer( CleanupPeriod );
}

/*---------------------------------------------------------------*/
static void push_timer (int seconds)
/*---------------------------------------------------------------*/
{
  static struct itimerval newt;
  static struct itimerval oldt;

  newt.it_interval.tv_sec  = 0;
  newt.it_interval.tv_usec = 0;
  
  newt.it_value.tv_sec     = seconds;
  newt.it_value.tv_usec    = 0;
  
  signal (SIGALRM, catcher);

  setitimer (ITIMER_REAL, &newt, &oldt);
}

/* static int PipeFd[2]; */

/*----------------------------------------------------------*/
/* static int pipe_handler () */
/*----------------------------------------------------------*/
/* Handler for the message sent through the pipe            */
/*----------------------------------------------------------*/
/* { */
/*   char buffer[256]; */
/*   int bytes; */

/*   /\* Empty the pipe *\/ */

/* #ifdef unix */
/*   bytes = read (PipeFd[0], buffer, sizeof (buffer)); */
/* #else */
/*   bytes = -1; */
/* #endif */

/* /\*   DoPrint("Pipe_handler> byte %d, errno %d\n", bytes, errno ); *\/ */
/*   if (bytes == -1) return (CmConnectStatusNoData); */

/* /\*   DoCleanup (); *\/ */
/* /\*   push_timer (CleanupPeriod); *\/ */

/*   return (CmConnectStatusNoData); */
/* } */

/*----------------------------------------------------------*/
/* static void pipe_setup () */
/*----------------------------------------------------------*/
/* Setup for the pipe                                       */
/* this function MUST be called only once                   */
/* and after the Cm init                                    */
/*----------------------------------------------------------*/
/* { */
/*   CmConnect connect; */
/*   static int done = 0; */

/*   if (!done) */
/*     { */
/*       done = 1; */

/* #ifdef unix */
/*       pipe (PipeFd); */
/*       fcntl (PipeFd[0], F_SETFL, O_NONBLOCK); */
/* #endif */

/*       connect = CmConnectNewWithSocket ("Pipe_Master_Producer", PipeFd[0]); */

/*       CmConnectInitHandlers ( */
/*           connect,  */
/*           (CmConnectReceiveHandler) pipe_handler, */
/*           (CmConnectRearmHandler) 0); */
/*     } */
/* } */

/*----------------------------------------------------------*/
/* static int pipe_send (const char* buffer) */
/*----------------------------------------------------------*/
/* { */
/*   int bytes = 0; */

/* #ifdef unix */
/*   bytes = write (PipeFd[1], buffer, strlen (buffer)); */
/* #endif */

/*   return (bytes); */
/* } */


